"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _notebooks2 = require("../../adaptors/notebooks");
var _saved_objects_paragraphs_router = require("../../adaptors/notebooks/saved_objects_paragraphs_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerParaRoute(router) {
  /* --> Updates the input content in a paragraph
   * --> Runs the paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/update/run/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await _notebooks2.BACKEND.updateRunFetchParagraph(context.observability_plugin.observabilityClient, request, _notebooks.wreckOptions);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Updates the input content in a paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const saveResponse = await _notebooks2.BACKEND.updateFetchParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Adds a new paragraph
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addResponse = await _notebooks2.BACKEND.addFetchNewParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Update paragraphs in backend with paragraphs passed in
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/set_paragraphs/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await _notebooks2.BACKEND.updateNote(opensearchNotebooksClient, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Deletes a paragraph
   * --> Fetches the all other Paragraphs as a list
   * --> Deletes all paragraphs if paragraphId is not provided
   */
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await _notebooks2.BACKEND.deleteFetchParagraphs(opensearchNotebooksClient, params, _notebooks.wreckOptions);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/clearall/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const clearParaResponse = await _notebooks2.BACKEND.clearAllFetchParagraphs(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.createParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/clearall`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const clearParaResponse = await (0, _saved_objects_paragraphs_router.clearParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await (0, _saved_objects_paragraphs_router.deleteParagraphs)(params, context.core.savedObjects.client);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/update/run`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await (0, _saved_objects_paragraphs_router.updateRunFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.updateFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/set_paragraphs`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          })),
          dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          }))
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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