"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultBackend = void 0;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
var _sample_notebooks = require("../../common/helpers/notebooks/sample_notebooks");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class DefaultBackend {
  constructor() {
    _defineProperty(this, "backend", 'Default Backend');
    // Creates a new notebooks with sample markdown text
    _defineProperty(this, "createNewNotebook", newNotebookName => {
      const noteObject = {
        dateCreated: new Date().toISOString(),
        name: newNotebookName,
        dateModified: new Date().toISOString(),
        backend: 'Default',
        paragraphs: []
      };
      return {
        object: noteObject
      };
    });
    // indexes a notebook with body provided
    _defineProperty(this, "indexNote", async function (client, body) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            notebook: body
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Doc Error:' + error);
      }
    });
    // updates a notebook with updateBody provided as parameter
    _defineProperty(this, "updateNote", async function (client, noteId, updateBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: noteId,
          body: {
            notebook: updateBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Doc Error:' + error);
      }
    });
    // fetched a notebook by Id
    _defineProperty(this, "getNote", async function (client, noteId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: noteId
        });
        if (response.observabilityObjectList.length === 0) {
          throw new Error('notebook id not found');
        }
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Doc Error:' + error);
      }
    });
    // gets first `FETCH_SIZE` notebooks available
    _defineProperty(this, "viewNotes", async function (client, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'notebook'
        });
        return response.observabilityObjectList.map(notebook => ({
          path: notebook.notebook.name,
          id: notebook.objectId,
          dateCreated: notebook.notebook.dateCreated,
          dateModified: notebook.notebook.dateModified
        }));
      } catch (error) {
        if (error.body.error.type === 'index_not_found_exception') {
          return [];
        } else throw new Error('View Notebooks Error:' + error);
      }
    });
    /* Fetches a notebook by id
     * Param: noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "fetchNote", async function (client, noteId, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, noteId);
        return {
          path: noteObject.notebook.name,
          dateCreated: noteObject.notebook.dateCreated,
          dateModified: noteObject.notebook.dateModified,
          paragraphs: noteObject.notebook.paragraphs
        };
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });
    /* Adds a notebook to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addNote", async function (client, params, _wreckOptions) {
      try {
        const newNotebook = this.createNewNotebook(params.name);
        const opensearchClientResponse = await this.indexNote(client, newNotebook.object);
        return {
          status: 'OK',
          message: opensearchClientResponse,
          body: opensearchClientResponse.objectId
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Adds sample notebooks to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addSampleNotes", async function (client, visIds, _wreckOptions) {
      try {
        const notebooks = (0, _sample_notebooks.getSampleNotebooks)(visIds);
        const newNotebooks = [];
        for (let i = 0; i < notebooks.length; i++) {
          const notebook = notebooks[i];
          await this.indexNote(client, notebook.notebook).then(response => {
            newNotebooks.push({
              id: response.objectId,
              name: notebook.notebook.name,
              dateModified: notebook.dateModified,
              dateCreated: notebook.dateCreated
            });
          });
        }
        return {
          status: 'OK',
          message: '',
          body: newNotebooks
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Renames a notebook
     * Params: name -> new name for the notebook to be renamed
     *         noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "renameNote", async function (client, params, _wreckOptions) {
      try {
        const updateNotebook = {
          name: params.name,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          status: 'OK',
          message: opensearchClientResponse
        };
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });
    /* Clone a notebook
     * Params: name -> new name for the cloned notebook
     *         noteId -> Id for the notebook to be cloned
     */
    _defineProperty(this, "cloneNote", async function (client, params, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, params.noteId);
        const newNotebook = this.createNewNotebook(params.name);
        const cloneNotebook = {
          ...newNotebook.object
        };
        cloneNotebook.paragraphs = noteObject.notebook.paragraphs;
        const opensearchClientIndexResponse = await this.indexNote(client, cloneNotebook);
        return {
          status: 'OK',
          body: {
            ...cloneNotebook,
            id: opensearchClientIndexResponse.objectId
          }
        };
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });
    /* Delete a notebook
     * Param: noteId -> Id for the notebook to be deleted
     */
    _defineProperty(this, "deleteNote", async function (client, noteList, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: noteList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });
    /* Export a notebook
     * Param: noteId -> Id for the notebook to be exported
     */
    _defineProperty(this, "exportNote", async function (client, noteId, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, noteId);
        return {
          status: 'OK',
          body: opensearchClientGetResponse
        };
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });
    /* Import a notebook
     * Params: noteObj -> note Object to be imported
     */
    _defineProperty(this, "importNote", async function (client, noteObj, _wreckOptions) {
      try {
        const newNoteObject = {
          ...noteObj
        };
        newNoteObject.id = 'note_' + (0, _uuid.v4)();
        newNoteObject.dateCreated = new Date().toISOString();
        newNoteObject.dateModified = new Date().toISOString();
        const opensearchClientIndexResponse = await this.indexNote(client, newNoteObject);
        return {
          status: 'OK',
          message: opensearchClientIndexResponse,
          body: opensearchClientIndexResponse.objectId
        };
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });
    /* Updates input for required paragraphs
     * Params: paragraphs -> list of paragraphs
     *         paragraphId -> Id of paragraph to be updated
     *         paragraphInput -> Input to be added
     */
    _defineProperty(this, "updateParagraph", function (paragraphs, paragraphId, paragraphInput, paragraphType) {
      try {
        const updatedParagraphs = [];
        paragraphs.map(paragraph => {
          const updatedParagraph = {
            ...paragraph
          };
          if (paragraph.id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            updatedParagraph.input.inputText = paragraphInput;
            if (paragraphType.length > 0) {
              updatedParagraph.input.inputType = paragraphType;
            }
          }
          updatedParagraphs.push(updatedParagraph);
        });
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Update Paragraph Error:' + error);
      }
    });
    // Creates new paragraph with the given input and input type
    _defineProperty(this, "createParagraph", function (paragraphInput, inputType) {
      try {
        let paragraphType = 'MARKDOWN';
        if (inputType === 'VISUALIZATION') {
          paragraphType = 'VISUALIZATION';
        }
        if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          paragraphType = 'OBSERVABILITY_VISUALIZATION';
        }
        if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
          paragraphType = 'QUERY';
        }
        const inputObject = {
          inputType: paragraphType,
          inputText: paragraphInput
        };
        const outputObjects = [{
          outputType: paragraphType,
          result: '',
          execution_time: '0s'
        }];
        const newParagraph = {
          id: 'paragraph_' + (0, _uuid.v4)(),
          dateCreated: new Date().toISOString(),
          dateModified: new Date().toISOString(),
          input: inputObject,
          output: outputObjects
        };
        return newParagraph;
      } catch (error) {
        throw new Error('Create Paragraph Error:' + error);
      }
    });
    /* Runs a paragraph
     * Currently only runs markdown by copying input.inputText to result
     * UI renders Markdown
     */
    _defineProperty(this, "runParagraph", async function (paragraphs, paragraphId, _client) {
      try {
        const updatedParagraphs = [];
        let index = 0;
        for (index = 0; index < paragraphs.length; ++index) {
          const startTime = (0, _performanceNow.default)();
          const updatedParagraph = {
            ...paragraphs[index]
          };
          if (paragraphs[index].id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'QUERY',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'OBSERVABILITY_VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: 'Please select an input type (%sql, %ppl, or %md)',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            }
          }
          updatedParagraphs.push(updatedParagraph);
        }
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Runs it
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content and output result)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, _wreckOptions) {
      try {
        const scopedClient = client.asScoped(request);
        const params = request.body;
        const opensearchClientGetResponse = await this.getNote(scopedClient, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType);
        const updatedOutputParagraphs = await this.runParagraph(updatedInputParagraphs, params.paragraphId, client);
        const updateNotebook = {
          paragraphs: updatedOutputParagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(scopedClient, params.noteId, updateNotebook);
        let resultParagraph = {};
        let index = 0;
        for (index = 0; index < updatedOutputParagraphs.length; ++index) {
          if (params.paragraphId === updatedOutputParagraphs[index].id) {
            resultParagraph = updatedOutputParagraphs[index];
          }
        }
        return resultParagraph;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateFetchParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput);
        const updateNotebook = {
          paragraphs: updatedInputParagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        let resultParagraph = {};
        updatedInputParagraphs.map(paragraph => {
          if (params.paragraphId === paragraph.id) {
            resultParagraph = paragraph;
          }
        });
        return resultParagraph;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });
    /* --> Fetches the Paragraph
     * --> Adds a Paragraph with input content
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be fetched
     */
    _defineProperty(this, "addFetchNewParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const paragraphs = opensearchClientGetResponse.notebook.paragraphs;
        const newParagraph = this.createParagraph(params.paragraphInput, params.inputType);
        paragraphs.splice(params.paragraphIndex, 0, newParagraph);
        const updateNotebook = {
          paragraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return newParagraph;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });
    /* --> Deletes a Paragraph with id
     * --> Fetches the all other Paragraphs as a list
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be deleted
     */
    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        if (params.paragraphId !== undefined) {
          opensearchClientGetResponse.notebook.paragraphs.map(paragraph => {
            if (paragraph.id !== params.paragraphId) {
              updatedparagraphs.push(paragraph);
            }
          });
        }
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        console.log('error', error);
        throw new Error('Delete Paragraph Error:' + error);
      }
    });
    /* --> Clears output for all the paragraphs
     * --> Fetches the all Paragraphs as a list (with cleared outputs)
     * --> Updates the notebook in index
     * Param: noteId -> Id of notebook to be cleared
     */
    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        opensearchClientGetResponse.notebook.paragraphs.map(paragraph => {
          const updatedParagraph = {
            ...paragraph
          };
          updatedParagraph.output = [];
          updatedparagraphs.push(updatedParagraph);
        });
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  }
}
exports.DefaultBackend = DefaultBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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