# coding=utf-8
# --------------------------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for license information.
# --------------------------------------------------------------------------------------------

from knack.help_files import helps


helps['aks'] = """
type: group
short-summary: Manage Azure Kubernetes Services.
"""

helps["aks check-acr"] = """
type: command
short-summary: Validate an ACR is accessible from an AKS cluster.
parameters:
  - name: --acr
    short-summary: The FQDN of the ACR.
  - name: --node-name
    short-summary: The name of a specific node to perform acr pull test checks. If not specified, it will be checked on a random node.
examples:
  - name: Validate the ACR is accessible from the AKS cluster.
    text: az aks check-acr --name MyManagedCluster --resource-group MyResourceGroup --acr myacr.azurecr.io
    crafted: true
"""

helps['aks browse'] = """
type: command
short-summary: Show the dashboard for a Kubernetes cluster in a web browser.
parameters:
  - name: --disable-browser
    type: bool
    short-summary: Don't launch a web browser after establishing port-forwarding.
    long-summary: Add this argument when launching a web browser manually, or for automated testing.
  - name: --listen-port
    short-summary: The listening port for the dashboard.
    long-summary: Add this argument when the default listening port is used by another process or unavailable.
  - name: --listen-address
    short-summary: The listening address for the dashboard.
    long-summary: Add this argument to listen on a specific IP address.
examples:
  - name: Show the dashboard for a Kubernetes cluster in a web browser. (autogenerated)
    text: az aks browse --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks create'] = """
type: command
short-summary: Create a new managed Kubernetes cluster.
parameters:
  - name: --generate-ssh-keys
    type: string
    short-summary: Generate SSH public and private key files if missing. The keys will be stored in the ~/.ssh directory.
  - name: --service-principal
    type: string
    short-summary: Service principal used for authentication to Azure APIs.
  - name: --skip-subnet-role-assignment
    type: bool
    short-summary: Skip role assignment for subnet (advanced networking).
    long-summary: If specified, please make sure your service principal has the access to your subnet.
  - name: --client-secret
    type: string
    short-summary: Secret associated with the service principal. This argument is required if `--service-principal` is specified.
  - name: --node-vm-size -s
    type: string
    short-summary: Size of Virtual Machines to create as Kubernetes nodes.
  - name: --dns-name-prefix -p
    type: string
    short-summary: Prefix for hostnames that are created. If not specified, generate a hostname using the managed cluster and resource group names.
  - name: --node-count -c
    type: int
    short-summary: Number of nodes in the Kubernetes node pool. After creating a cluster, you can change the size of its node pool with `az aks scale`.
  - name: --zones -z
    type: string array
    short-summary: Availability zones where agent nodes will be placed. Also, to install agent nodes to more than one zones you need to pass zone numbers (1,2 or 3) separated by blanks.  For example -  To have all 3 zones, you are expected to enter `--zones 1 2 3`
  - name: --node-osdisk-size
    type: int
    short-summary: Size in GiB of the OS disk for each node in the node pool. Minimum 30 GiB.
  - name: --node-osdisk-type
    type: string
    short-summary: "OS disk type to be used for machines in a given agent pool: Ephemeral or Managed. Defaults to 'Ephemeral' when possible in conjunction with VM size and OS disk size. May not be changed for this pool after creation."
  - name: --kubernetes-version -k
    type: string
    short-summary: Version of Kubernetes to use for creating the cluster, such as "1.16.9".
    populator-commands:
      - "`az aks get-versions`"
  - name: --os-sku
    type: string
    short-summary: The OS SKU of the agent node pool. Ubuntu or CBLMariner.
  - name: --ssh-key-value
    type: string
    short-summary: Public key path or key contents to install on node VMs for SSH access. For example, 'ssh-rsa AAAAB...snip...UcyupgH azureuser@linuxvm'.
  - name: --admin-username -u
    type: string
    short-summary: User account to create on node VMs for SSH access.
  - name: --windows-admin-username
    type: string
    short-summary: User account to create on windows node VMs.
    long-summary: |-
      Rules for windows-admin-username:
          - restriction: Cannot end in "."
          - Disallowed values: "administrator", "admin", "user", "user1", "test", "user2", "test1", "user3", "admin1", "1", "123", "a", "actuser", "adm", "admin2", "aspnet", "backup", "console", "david", "guest", "john", "owner", "root", "server", "sql", "support", "support_388945a0", "sys", "test2", "test3", "user4", "user5".
          - Minimum-length: 1 character
          - Max-length: 20 characters
      Reference: https://learn.microsoft.com/dotnet/api/microsoft.azure.management.compute.models.virtualmachinescalesetosprofile.adminusername?view=azure-dotnet
  - name: --windows-admin-password
    type: string
    short-summary: User account password to use on windows node VMs.
    long-summary: |-
      Rules for windows-admin-password:
          - Minimum-length: 14 characters
          - Max-length: 123 characters
          - Complexity requirements: 3 out of 4 conditions below need to be fulfilled
            * Has lower characters
            * Has upper characters
            * Has a digit
            * Has a special character (Regex match [\\W_])
          - Disallowed values:  "abc@123", "P@$$w0rd", "P@ssw0rd", "P@ssword123", "Pa$$word", "pass@word1", "Password!", "Password1", "Password22", "iloveyou!"
      Reference: https://learn.microsoft.com/dotnet/api/microsoft.azure.management.compute.models.virtualmachinescalesetosprofile.adminpassword?view=azure-dotnet
  - name: --enable-ahub
    type: bool
    short-summary: Enable Azure Hybrid User Benefits (AHUB) for Windows VMs.
  - name: --enable-aad
    type: bool
    short-summary: Enable managed AAD feature for cluster.
  - name: --aad-admin-group-object-ids
    type: string
    short-summary: Comma-separated list of aad group object IDs that will be set as cluster admin.
  - name: --aad-tenant-id
    type: string
    short-summary: The ID of an Azure Active Directory tenant.
  - name: --dns-service-ip
    type: string
    short-summary: An IP address assigned to the Kubernetes DNS service.
    long-summary: This address must be within the Kubernetes service address range specified by "--service-cidr". For example, 10.0.0.10.
  - name: --docker-bridge-address
    type: string
    short-summary: A specific IP address and netmask for the Docker bridge, using standard CIDR notation.
    long-summary: This address must not be in any Subnet IP ranges, or the Kubernetes service address range. For example, 172.17.0.1/16.
  - name: --load-balancer-sku
    type: string
    short-summary: Azure Load Balancer SKU selection for your cluster. basic or standard. Defaults to 'standard'
    long-summary: Select between Basic or Standard Azure Load Balancer SKU for your AKS cluster.
  - name: --load-balancer-managed-outbound-ip-count
    type: int
    short-summary: Load balancer managed outbound IP count.
    long-summary: Desired number of managed outbound IPs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only.
  - name: --load-balancer-managed-outbound-ipv6-count
    type: int
    short-summary: Load balancer managed outbound IPv6 IP count.
    long-summary: Desired number of managed outbound IPv6 IPs for load balancer outbound connection. Valid for dual-stack (--ip-families IPv4,IPv6) only.
  - name: --load-balancer-outbound-ips
    type: string
    short-summary: Load balancer outbound IP resource IDs.
    long-summary: Comma-separated public IP resource IDs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only.
  - name: --load-balancer-outbound-ip-prefixes
    type: string
    short-summary: Load balancer outbound IP prefix resource IDs.
    long-summary: Comma-separated public IP prefix resource IDs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only.
  - name: --load-balancer-outbound-ports
    type: int
    short-summary: Load balancer outbound allocated ports.
    long-summary: Desired static number of outbound ports per VM in the load balancer backend pool. By default, set to 0 which uses the default allocation based on the number of VMs.
  - name: --load-balancer-idle-timeout
    type: int
    short-summary: Load balancer idle timeout in minutes.
    long-summary: Desired idle timeout for load balancer outbound flows, default is 30 minutes. Please specify a value in the range of [4, 100].
  - name: --load-balancer-backend-pool-type
    type: string
    short-summary: Load balancer backend pool type.
    long-summary: Define the LoadBalancer backend pool type of managed inbound backend pool. The nodeIP means the VMs will be attached to the LoadBalancer by adding its private IP address to the backend pool. The nodeIPConfiguration means the VMs will be attached to the LoadBalancer by referencing the backend pool ID in the VM's NIC.
  - name: --nat-gateway-managed-outbound-ip-count
    type: int
    short-summary: NAT gateway managed outbound IP count.
    long-summary: Desired number of managed outbound IPs for NAT gateway outbound connection. Please specify a value in the range of [1, 16]. Valid for Standard SKU load balancer cluster with managedNATGateway outbound type only.
  - name: --nat-gateway-idle-timeout
    type: int
    short-summary: NAT gateway idle timeout in minutes.
    long-summary: Desired idle timeout for NAT gateway outbound flows, default is 4 minutes. Please specify a value in the range of [4, 120]. Valid for Standard SKU load balancer cluster with managedNATGateway outbound type only.
  - name: --outbound-type
    type: string
    short-summary: How outbound traffic will be configured for a cluster.
    long-summary: Select between loadBalancer, userDefinedRouting, managedNATGateway, userAssignedNATGateway and none. If not set, defaults to type loadBalancer. Requires --vnet-subnet-id to be provided with a preconfigured route table and --load-balancer-sku to be Standard.
  - name: --auto-upgrade-channel
    type: string
    short-summary: Specify the upgrade channel for autoupgrade.
  - name: --node-os-upgrade-channel
    type: string
    short-summary: Manner in which the OS on your nodes is updated. It could be NodeImage, None, SecurityPatch or Unmanaged.
  - name: --enable-cluster-autoscaler
    type: bool
    short-summary: Enable cluster autoscaler, default value is false.
    long-summary: If specified, please make sure the kubernetes version is larger than 1.10.6.
  - name: --min-count
    type: int
    short-summary: Minimum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [1, 1000].
  - name: --max-count
    type: int
    short-summary: Maximum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [1, 1000].
  - name: --vm-set-type
    type: string
    short-summary: Agent pool vm set type. VirtualMachineScaleSets or AvailabilitySet. Defaults to 'VirtualMachineScaleSets'
  - name: --enable-addons -a
    type: string
    short-summary: Enable the Kubernetes addons in a comma-separated list.
    long-summary: |-
        These addons are available:
            - http_application_routing : configure ingress with automatic public DNS name creation.
            - monitoring               : turn on Log Analytics monitoring. Uses the Log Analytics Default Workspace if it exists, else creates one.
                                         Specify "--workspace-resource-id" to use an existing workspace.
                                         Specify "--enable-msi-auth-for-monitoring" to use Managed Identity Auth.
                                         Specify "--enable-syslog" to enable syslog data collection from nodes. Note MSI must be enabled
                                         Specify "--data-collection-settings" to configure data collection settings
                                         Specify "--ampls-resource-id" for private link. Note MSI must be enabled.
                                         Specify "--enable-high-log-scale-mode" to enable high log scale mode for container logs. Note MSI must be enabled.
                                         If monitoring addon is enabled --no-wait argument will have no effect
            - azure-policy             : enable Azure policy. The Azure Policy add-on for AKS enables at-scale enforcements and safeguards on your clusters in a centralized, consistent manner.
                                         Learn more at aka.ms/aks/policy.
            - virtual-node             : enable AKS Virtual Node.
                                         Requires --aci-subnet-name to provide the name of an existing subnet for the Virtual Node to use.
                                         aci-subnet-name must be in the same vnet which is specified by --vnet-subnet-id (required as well).
            - confcom                  : enable confcom addon, this will enable SGX device plugin by default.
            - open-service-mesh        : enable Open Service Mesh addon.
            - azure-keyvault-secrets-provider : enable Azure Keyvault Secrets Provider addon.
  - name: --disable-rbac
    type: bool
    short-summary: Disable Kubernetes Role-Based Access Control.
  - name: --max-pods -m
    type: int
    short-summary: The maximum number of pods deployable to a node.
    long-summary: If not specified, defaults based on network-plugin. 30 for "azure", 110 for "kubenet", or 250 for "none".
  - name: --network-plugin
    type: string
    short-summary: The Kubernetes network plugin to use.
    long-summary: Specify "azure" for routable pod IPs from VNET, "kubenet" for non-routable pod IPs with an overlay network, or "none" for no networking configured. Defaults to "kubenet".
  - name: --network-plugin-mode
    type: string
    short-summary: The network plugin mode to use.
    long-summary: |
        Used to control the mode the network plugin should operate in. For example, "overlay" used with
        --network-plugin=azure will use an overlay network (non-VNET IPs) for pods in the cluster.
  - name: --network-policy
    type: string
    short-summary: Network Policy Engine to use.
    long-summary: |
        Azure provides three Network Policy Engines for enforcing network policies that can be used together with "azure" network plugin. The following values can be specified:
          - "azure" for Azure Network Policy Manager,
          - "cilium" for Azure CNI Powered by Cilium,
          - "calico" for open-source network and network security solution founded by Tigera,
          - "none" when no Network Policy Engine is installed (default value).
        Defaults to "none" (network policy disabled).
  - name: --network-dataplane
    type: string
    short-summary: The network dataplane to use.
    long-summary: |
        Network dataplane used in the Kubernetes cluster.
        Specify "azure" to use the Azure dataplane (default) or "cilium" to enable Cilium dataplane.
  - name: --no-ssh-key -x
    type: string
    short-summary: Do not use or create a local SSH key.
    long-summary: To access nodes after creating a cluster with this option, use the Azure Portal.
  - name: --pod-cidr
    type: string
    short-summary: A CIDR notation IP range from which to assign pod IPs when kubenet is used.
    long-summary: This range must not overlap with any Subnet IP ranges. For example, 172.244.0.0/16.
  - name: --message-of-the-day
    type: string
    short-summary: Path to a file containing the desired message of the day. Only valid for linux nodes. Will be written to /etc/motd.
  - name: --service-cidr
    type: string
    short-summary: A CIDR notation IP range from which to assign service cluster IPs.
    long-summary: This range must not overlap with any Subnet IP ranges. For example, 10.0.0.0/16.
  - name: --service-cidrs
    type: string
    short-summary: A comma-separated list of CIDR notation IP ranges from which to assign service cluster IPs.
    long-summary: Each range must not overlap with any Subnet IP ranges. For example, "10.0.0.0/16,2001:abcd::/108".
  - name: --pod-cidrs
    type: string
    short-summary: A comma-separated list of CIDR notation IP ranges from which to assign pod IPs when kubenet is used.
    long-summary: Each range must not overlap with any Subnet IP ranges. For example, "172.244.0.0/16,fd0:abcd::/64".
  - name: --ip-families
    type: string
    short-summary: A comma-separated list of IP versions to use for cluster networking.
    long-summary: Each IP version should be in the format IPvN. For example, IPv4.
  - name: --vnet-subnet-id
    type: string
    short-summary: The ID of a subnet in an existing VNet into which to deploy the cluster.
  - name: --pod-subnet-id
    type: string
    short-summary: The ID of a subnet in an existing VNet into which to assign pods in the cluster (requires azure network-plugin).
  - name: --ppg
    type: string
    short-summary: The ID of a PPG.
  - name: --enable-node-public-ip
    type: bool
    short-summary: Enable VMSS node public IP.
  - name: --node-public-ip-prefix-id
    type: string
    short-summary: Public IP prefix ID used to assign public IPs to VMSS nodes.
  - name: --workspace-resource-id
    type: string
    short-summary: The resource ID of an existing Log Analytics Workspace to use for storing monitoring data. If not specified, uses the default Log Analytics Workspace if it exists, otherwise creates one.
  - name: --enable-msi-auth-for-monitoring
    type: bool
    short-summary: Enable Managed Identity Auth for Monitoring addon.
  - name: --enable-syslog
    type: bool
    short-summary: Enable syslog data collection for Monitoring addon
  - name: --data-collection-settings
    type: string
    short-summary: Path to JSON file containing data collection settings for Monitoring addon.
  - name: --ampls-resource-id
    type: string
    short-summary: Resource ID of Azure Monitor Private Link scope for Monitoring Addon.
  - name: --enable-high-log-scale-mode
    type: bool
    short-summary: Enable High Log Scale Mode for Container Logs.
  - name: --tier
    type: string
    short-summary: Specify SKU tier for managed clusters. '--tier standard' enables a standard managed cluster service with a financially backed SLA. '--tier free' does not have a financially backed SLA.
  - name: --attach-acr
    type: string
    short-summary: Grant the 'acrpull' role assignment to the ACR specified by name or resource ID.
  - name: --enable-private-cluster
    type: string
    short-summary: Enable private cluster.
  - name: --private-dns-zone
    type: string
    short-summary: Private dns zone mode for private cluster.
    long-summary: Allowed values are "system", "none" or custom private dns zone resource id. If not set, defaults to type system. Requires --enable-private-cluster to be used.
  - name: --fqdn-subdomain
    type: string
    short-summary: Prefix for FQDN that is created for private cluster with custom private dns zone scenario.
  - name: --disable-public-fqdn
    type: bool
    short-summary: Disable public fqdn feature for private cluster.
  - name: --api-server-authorized-ip-ranges
    type: string
    short-summary: Comma-separated list of authorized apiserver IP ranges. Set to 0.0.0.0/32 to restrict apiserver traffic to node pools.
  - name: --enable-managed-identity
    type: bool
    short-summary: Using a system assigned managed identity to manage cluster resource group. You can explicitly specify "--service-principal" and "--client-secret" to disable managed identity, otherwise it will be enabled.
  - name: --assign-identity
    type: string
    short-summary: Specify an existing user assigned identity for control plane's usage in order to manage cluster resource group.
  - name: --assign-kubelet-identity
    type: string
    short-summary: Specify an existing user assigned identity for kubelet's usage, which is typically used to pull image from ACR.
  - name: --node-osdisk-diskencryptionset-id -d
    type: string
    short-summary: ResourceId of the disk encryption set to use for enabling encryption at rest on agent node os disk.
  - name: --enable-workload-identity
    type: bool
    short-summary: Enable workload identity addon.
  - name: --aci-subnet-name
    type: string
    short-summary: The name of a subnet in an existing VNet into which to deploy the virtual nodes.
  - name: --appgw-name
    type: string
    short-summary: Name of the application gateway to create/use in the node resource group. Use with ingress-azure addon.
  - name: --appgw-subnet-cidr
    type: string
    short-summary: Subnet CIDR to use for a new subnet created to deploy the Application Gateway. Use with ingress-azure addon.
  - name: --appgw-id
    type: string
    short-summary: Resource Id of an existing Application Gateway to use with AGIC. Use with ingress-azure addon.
  - name: --appgw-subnet-id
    type: string
    short-summary: Resource Id of an existing Subnet used to deploy the Application Gateway. Use with ingress-azure addon.
  - name: --appgw-watch-namespace
    type: string
    short-summary: Specify the namespace, which AGIC should watch. This could be a single string value, or a comma-separated list of namespaces.
  - name: --enable-sgxquotehelper
    type: bool
    short-summary: Enable SGX quote helper for confcom addon.
  - name: --enable-encryption-at-host
    type: bool
    short-summary: Enable EncryptionAtHost, default value is false.
  - name: --enable-ultra-ssd
    type: bool
    short-summary: Enable UltraSSD, default value is false.
  - name: --enable-azure-rbac
    type: bool
    short-summary: Enable Azure RBAC to control authorization checks on cluster.
  - name: --edge-zone
    type: string
    short-summary: The name of the Edge Zone.
  - name: --disable-local-accounts
    type: bool
    short-summary: If set to true, getting static credential will be disabled for this cluster.
  - name: --tags
    type: string
    short-summary: The tags of the managed cluster. The managed cluster instance and all resources managed by the cloud provider will be tagged.
  - name: --aks-custom-headers
    type: string
    short-summary: Comma-separated key-value pairs to specify custom headers.
  - name: --enable-secret-rotation
    type: bool
    short-summary: Enable secret rotation. Use with azure-keyvault-secrets-provider addon.
  - name: --rotation-poll-interval
    type: string
    short-summary: Set interval of rotation poll. Use with azure-keyvault-secrets-provider addon.
  - name: --enable-fips-image
    type: bool
    short-summary: Use FIPS-enabled OS on agent nodes.
  - name: --enable-windows-gmsa
    type: bool
    short-summary: Enable Windows gmsa.
  - name: --gmsa-dns-server
    type: string
    short-summary: Specify DNS server for Windows gmsa for this cluster.
    long-summary: |-
        You do not need to set this if you have set DNS server in the VNET used by the cluster.
        You must set or not set --gmsa-dns-server and --gmsa-root-domain-name at the same time when setting --enable-windows-gmsa.
  - name: --gmsa-root-domain-name
    type: string
    short-summary: Specify root domain name for Windows gmsa for this cluster.
    long-summary: |-
        You do not need to set this if you have set DNS server in the VNET used by the cluster.
        You must set or not set --gmsa-dns-server and --gmsa-root-domain-name at the same time when setting --enable-windows-gmsa.
  - name: --snapshot-id
    type: string
    short-summary: The source snapshot id used to create this cluster.
  - name: --kubelet-config
    type: string
    short-summary: Path to JSON file containing Kubelet configurations for agent nodes. https://aka.ms/aks/custom-node-config
  - name: --linux-os-config
    type: string
    short-summary: Path to JSON file containing OS configurations for Linux agent nodes. https://aka.ms/aks/custom-node-config
  - name: --node-resource-group
    type: string
    short-summary: The node resource group is the resource group where all customer's resources will be created in, such as virtual machines.
  - name: --k8s-support-plan
    type: string
    short-summary: Choose from "KubernetesOfficial" or "AKSLongTermSupport", with "AKSLongTermSupport" you get 1 extra year of CVE patchs.
  - name: --ca-certs --custom-ca-trust-certificates
    type: string
    short-summary: Path to a file containing up to 10 blank line separated certificates. Only valid for Linux nodes.
    long-summary: These certificates are used by Custom CA Trust feature and will be added to trust stores of nodes.
  - name: --enable-defender
    type: bool
    short-summary: Enable Microsoft Defender security profile.
  - name: --defender-config
    type: string
    short-summary: Path to JSON file containing Microsoft Defender profile configurations.
  - name: --host-group-id
    type: string
    short-summary: The fully qualified dedicated host group id used to provision agent node pool.
  - name: --enable-azure-keyvault-kms
    type: bool
    short-summary: Enable Azure KeyVault Key Management Service.
  - name: --azure-keyvault-kms-key-id
    type: string
    short-summary: Identifier of Azure Key Vault key.
  - name: --azure-keyvault-kms-key-vault-network-access
    type: string
    short-summary: Network Access of Azure Key Vault.
    long-summary: Allowed values are "Public", "Private". If not set, defaults to type "Public". Requires --azure-keyvault-kms-key-id to be used.
  - name: --azure-keyvault-kms-key-vault-resource-id
    type: string
    short-summary: Resource ID of Azure Key Vault.
  - name: --enable-image-cleaner
    type: bool
    short-summary: Enable ImageCleaner Service.
  - name: --image-cleaner-interval-hours
    type: int
    short-summary: ImageCleaner scanning interval.
  - name: --disable-disk-driver
    type: bool
    short-summary: Disable AzureDisk CSI Driver.
  - name: --disable-file-driver
    type: bool
    short-summary: Disable AzureFile CSI Driver.
  - name: --disable-snapshot-controller
    type: bool
    short-summary: Disable CSI Snapshot Controller.
  - name: --enable-blob-driver
    type: bool
    short-summary: Enable AzureBlob CSI Driver.
  - name: --http-proxy-config
    type: string
    short-summary: HTTP Proxy configuration for this cluster.
  - name: --gpu-instance-profile
    type: string
    short-summary: GPU instance profile to partition multi-gpu Nvidia GPUs.
  - name: --enable-oidc-issuer
    type: bool
    short-summary: Enable OIDC issuer.
  - name: --enable-keda
    type: bool
    short-summary: Enable KEDA workload auto-scaler.
  - name: --enable-azure-monitor-metrics
    type: bool
    short-summary: Enable a kubernetes cluster with the Azure Monitor managed service for Prometheus integration.
  - name: --azure-monitor-workspace-resource-id
    type: string
    short-summary: Resource ID of the Azure Monitor Workspace
  - name: --ksm-metric-labels-allow-list
    type: string
    short-summary: Comma-separated list of additional Kubernetes label keys that will be used in the resource' labels metric. By default the metric contains only name and namespace labels. To include additional labels provide a list of resource names in their plural form and Kubernetes label keys you would like to allow for them (e.g. '=namespaces=[k8s-label-1,k8s-label-n,...],pods=[app],...)'. A single '*' can be provided per resource instead to allow any labels, but that has severe performance implications (e.g. '=pods=[*]').
  - name: --ksm-metric-annotations-allow-list
    type: string
    short-summary: Comma-separated list of additional Kubernetes label keys that will be used in the resource' labels metric. By default the metric contains only name and namespace labels. To include additional labels provide a list of resource names in their plural form and Kubernetes label keys you would like to allow for them (e.g.'=namespaces=[k8s-label-1,k8s-label-n,...],pods=[app],...)'. A single '*' can be provided per resource instead to allow any labels, but that has severe performance implications (e.g. '=pods=[*]').
  - name: --grafana-resource-id
    type: string
    short-summary: Resource ID of the Azure Managed Grafana Workspace
  - name: --enable-windows-recording-rules
    type: bool
    short-summary: Enable Windows Recording Rules when enabling the Azure Monitor Metrics addon
  - name: --nodepool-taints
    type: string
    short-summary: The node taints for all node pool.
  - name: --nodepool-labels
    type: string
    short-summary: The node labels for all node pool. See https://aka.ms/node-labels for syntax of labels.
  - name: --enable-vpa
    type: bool
    short-summary: Enable vertical pod autoscaler for cluster.
  - name: --nodepool-allowed-host-ports
    type: string
    short-summary: Expose host ports on the node pool. When specified, format should be a space-separated list of ranges with protocol, eg. `80/TCP 443/TCP 4000-5000/TCP`.
  - name: --nodepool-asg-ids
    type: string
    short-summary: The IDs of the application security groups to which the node pool's network interface should belong. When specified, format should be a space-separated list of IDs.
  - name: --node-public-ip-tags
    type: string
    short-summary: The ipTags of the node public IPs.
  - name: --crg-id
    type: string
    short-summary: The crg id used to associate the new cluster with the existed Capacity Reservation Group resource.
  - name: --enable-asm --enable-azure-service-mesh
    type: bool
    short-summary: Enable Azure Service Mesh addon.
  - name: --enable-app-routing
    type: bool
    short-summary: Enable Application Routing addon.
  - name: --revision
    type: string
    short-summary: Azure Service Mesh revision to install.
  - name: --enable-cost-analysis
    type: bool
    short-summary: Enable exporting Kubernetes Namespace and Deployment details to the Cost Analysis views in the Azure portal. For more information see aka.ms/aks/docs/cost-analysis.
  - name: --enable-secure-boot
    type: bool
    short-summary: Enable Secure Boot on all node pools in the cluster. Must use VMSS agent pool type.
  - name: --enable-vtpm
    type: bool
    short-summary: Enable vTPM on all node pools in the cluster. Must use VMSS agent pool type.
  - name: --enable-acns
    type: bool
    short-summary: Enable advanced network functionalities on a cluster. Enabling this will incur additional costs. For non-cilium clusters, acns security will be disabled by default until further notice.
  - name: --disable-acns-observability
    type: bool
    short-summary: Used to disable advanced networking observability features on a clusters when enabling advanced networking features with "--enable-acns".
  - name: --disable-acns-security
    type: bool
    short-summary: Used to disable advanced networking security features on a clusters when enabling advanced networking features with "--enable-acns".
  - name: --nrg-lockdown-restriction-level
    type: string
    short-summary: Restriction level on the managed node resource group.
    long-summary: The restriction level of permissions allowed on the cluster's managed node resource group, supported values are Unrestricted, and ReadOnly (recommended ReadOnly).
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the managed cluster, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new cluster.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new cluster to be created, but to prevent updating an existing cluster. Other values will be ignored.
  - name: --bootstrap-artifact-source
    type: string
    short-summary: Configure artifact source when bootstraping the cluster.
    long-summary: |
        The artifacts include the addon image. Use "Direct" to download artifacts from MCR, "Cache" to downalod artifacts from Azure Container Registry.
  - name: --bootstrap-container-registry-resource-id
    type: string
    short-summary: Configure container registry resource ID. Must use "Cache" as bootstrap artifact source.

examples:
  - name: Create a Kubernetes cluster with an existing SSH public key.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --ssh-key-value /path/to/publickey
  - name: Create a Kubernetes cluster with a specific version.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --kubernetes-version 1.16.9
  - name: Create a Kubernetes cluster with a larger node pool.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --node-count 7
  - name: Create a kubernetes cluster with default kubernetes version, default SKU load balancer (Standard) and default vm set type (VirtualMachineScaleSets).
    text: az aks create -g MyResourceGroup -n MyManagedCluster
  - name: Create a kubernetes cluster with standard SKU load balancer and two AKS created IPs for the load balancer outbound connection usage.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-managed-outbound-ip-count 2
  - name: Create a kubernetes cluster with a standard SKU load balancer, with two outbound AKS managed IPs an idle flow timeout of 5 minutes and 8000 allocated ports per machine
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-managed-outbound-ip-count 2 --load-balancer-idle-timeout 5 --load-balancer-outbound-ports 8000
  - name: Create a kubernetes cluster with standard SKU load balancer and use the provided public IPs for the load balancer outbound connection usage.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-outbound-ips <ip-resource-id-1,ip-resource-id-2>
  - name: Create a kubernetes cluster with standard SKU load balancer and use the provided public IP prefixes for the load balancer outbound connection usage.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-outbound-ip-prefixes <ip-prefix-resource-id-1,ip-prefix-resource-id-2>
  - name: Create a kubernetes cluster with a AKS managed NAT gateway, with two outbound AKS managed IPs an idle flow timeout of 4 minutes
    text: az aks create -g MyResourceGroup -n MyManagedCluster --nat-gateway-managed-outbound-ip-count 2 --nat-gateway-idle-timeout 4 --outbound-type managedNATGateway --generate-ssh-keys
  - name: Create a kubernetes cluster with basic SKU load balancer and AvailabilitySet vm set type.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-sku basic --vm-set-type AvailabilitySet
  - name: Create a kubernetes cluster with authorized apiserver IP ranges.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --api-server-authorized-ip-ranges 193.168.1.0/24,194.168.1.0/24,195.168.1.0
  - name: Create a kubernetes cluster which enables managed identity.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-managed-identity
  - name: Create a kubernetes cluster with userDefinedRouting, standard load balancer SKU and a custom subnet preconfigured with a route table
    text: az aks create -g MyResourceGroup -n MyManagedCluster --outbound-type userDefinedRouting --load-balancer-sku standard --vnet-subnet-id customUserSubnetVnetID
  - name: Create a kubernetes cluster with supporting Windows agent pools.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-sku Standard --network-plugin azure --windows-admin-username azure --windows-admin-password 'replacePassword1234$'
  - name: Create a kubernetes cluster with supporting Windows agent pools with AHUB enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-sku Standard --network-plugin azure --windows-admin-username azure --windows-admin-password 'replacePassword1234$' --enable-ahub
  - name: Create a kubernetes cluster with managed AAD enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-aad --aad-admin-group-object-ids <id-1,id-2> --aad-tenant-id <id>
  - name: Create a kubernetes cluster with server side encryption using your owned key.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --node-osdisk-diskencryptionset-id <disk-encryption-set-resource-id>
  - name: Create a kubernetes cluster with ephemeral OS enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --node-osdisk-type Ephemeral --node-osdisk-size 48
  - name: Create a kubernetes cluster with EncryptionAtHost enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-encryption-at-host
  - name: Create a kubernetes cluster with UltraSSD enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-ultra-ssd
  - name: Create a kubernetes cluster with Azure RBAC enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-aad --enable-azure-rbac
  - name: Create a kubernetes cluster with custom control plane identity and kubelet identity.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --assign-identity <control-plane-identity-resource-id> --assign-kubelet-identity <kubelet-identity-resource-id>
  - name: Create a kubernetes cluster in the Edge Zone.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --location <location> --kubernetes-version 1.20.7 --edge-zone <edge-zone-name>
  - name: Create a kubernetes cluster with a specific OS SKU
    text: az aks create -g MyResourceGroup -n MyManagedCluster --os-sku Ubuntu
  - name: Create a kubernetes cluster with custom tags
    text: az aks create -g MyResourceGroup -n MyManagedCluster --tags "foo=bar" "baz=qux"
  - name: Create a kubernetes cluster with custom headers
    text: az aks create -g MyResourceGroup -n MyManagedCluster --aks-custom-headers WindowsContainerRuntime=containerd
  - name: Create a kubernetes cluster with FIPS-enabled OS
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-fips-image
  - name: Create a kubernetes cluster with enabling Windows gmsa and with setting DNS server in the vnet used by the cluster.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-sku Standard --network-plugin azure --windows-admin-username azure --windows-admin-password 'replacePassword1234$' --enable-windows-gmsa
  - name: Create a kubernetes cluster with enabling Windows gmsa but without setting DNS server in the vnet used by the cluster.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --load-balancer-sku Standard --network-plugin azure --windows-admin-username azure --windows-admin-password 'replacePassword1234$' --enable-windows-gmsa --gmsa-dns-server "10.240.0.4" --gmsa-root-domain-name "contoso.com"
  - name: create a kubernetes cluster with a snapshot id.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --kubernetes-version 1.20.9 --snapshot-id "/subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/snapshots/mysnapshot1"
  - name: create a kubernetes cluster with support of hostgroup id.
    text: az aks create -g MyResourceGroup -n MyMC --kubernetes-version 1.20.13 --location westus2 --host-group-id /subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/hostGroups/myHostGroup --node-vm-size VMSize --enable-managed-identity --assign-identity <user_assigned_identity_resource_id>
  - name: Create a kubernetes cluster with no CNI installed.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --network-plugin none
  - name: Create a kubernetes cluster with KEDA workload autoscaler enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-keda
  - name: Create a kubernetes cluster with the Azure Monitor managed service for Prometheus integration enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-azure-monitor-metrics
  - name: Create a kubernetes cluster with vertical pod autoscaler enaled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-vpa
  - name: create a kubernetes cluster with a Capacity Reservation Group(CRG) ID.
    text: az aks create -g MyResourceGroup -n MyMC --kubernetes-version 1.20.9 --node-vm-size VMSize --assign-identity "subscriptions/SubID/resourceGroups/RGName/providers/Microsoft.ManagedIdentity/userAssignedIdentities/myID" --enable-managed-identity --crg-id "subscriptions/SubID/resourceGroups/RGName/providers/Microsoft.ContainerService/CapacityReservationGroups/MyCRGID"
  - name: Create a kubernetes cluster with Azure Service Mesh enabled.
    text: az aks create -g MyResourceGroup -n MyManagedCluster --enable-azure-service-mesh
"""

helps['aks update'] = """
type: command
short-summary: Update a managed Kubernetes cluster. When called with no optional arguments this attempts to move the cluster to its goal state without changing the current cluster configuration. This can be used to move out of a non succeeded state.
parameters:
  - name: --enable-cluster-autoscaler -e
    type: bool
    short-summary: Enable cluster autoscaler.
  - name: --disable-cluster-autoscaler -d
    type: bool
    short-summary: Disable cluster autoscaler.
  - name: --update-cluster-autoscaler -u
    type: bool
    short-summary: Update min-count or max-count for cluster autoscaler.
  - name: --min-count
    type: int
    short-summary: Minimum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [1, 1000]
  - name: --max-count
    type: int
    short-summary: Maximum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [1, 1000]
  - name: --tier
    type: string
    short-summary: Specify SKU tier for managed clusters. '--tier standard' enables a standard managed cluster service with a financially backed SLA. '--tier free' changes a standard managed cluster to a free one.
  - name: --ip-families
    type: string
    short-summary: A comma-separated list of IP versions to use for cluster networking.
    long-summary: Each IP version should be in the format IPvN. For example, IPv4.
  - name: --network-plugin
    type: string
    short-summary: The Kubernetes network plugin to use.
    long-summary: Specify "azure" along with --network-plugin-mode=overlay to update a cluster to use Azure CNI Overlay. For more information see https://aka.ms/aks/azure-cni-overlay
  - name: --network-plugin-mode
    type: string
    short-summary: Update the mode of a network plugin to migrate to a different pod networking setup.
  - name: --network-policy
    type: string
    short-summary: Update Network Policy Engine.
    long-summary: |
        Azure provides three Network Policy Engines for enforcing network policies. The following values can be specified:
          - "azure" for Azure Network Policy Manager,
          - "cilium" for Azure CNI Powered by Cilium,
          - "calico" for open-source network and network security solution founded by Tigera,
          - "none" to uninstall Network Policy Engine (Azure Network Policy Manager or Calico).
        Defaults to "none" (network policy disabled).
  - name: --pod-cidr
    type: string
    short-summary: Update the pod CIDR for a cluster. Used when updating a cluster from Azure CNI to Azure CNI Overlay.
  - name: --network-dataplane
    type: string
    short-summary: The network dataplane to use.
    long-summary: |
        Network dataplane used in the Kubernetes cluster.
        Specify "azure" to use the Azure dataplane (default) or "cilium" to enable Cilium dataplane.
  - name: --load-balancer-managed-outbound-ip-count
    type: int
    short-summary: Load balancer managed outbound IP count.
    long-summary: Desired number of managed outbound IPs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only. If the new value is greater than the original value, new additional outbound IPs will be created. If the value is less than the original value, existing outbound IPs will be deleted and outbound connections may fail due to configuration update.
  - name: --load-balancer-managed-outbound-ipv6-count
    type: int
    short-summary: Load balancer managed outbound IPv6 IP count.
    long-summary: Desired number of managed outbound IPv6 IPs for load balancer outbound connection. Valid for dual-stack (--ip-families IPv4,IPv6) only.
  - name: --load-balancer-outbound-ips
    type: string
    short-summary: Load balancer outbound IP resource IDs.
    long-summary: Comma-separated public IP resource IDs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only.
  - name: --load-balancer-outbound-ip-prefixes
    type: string
    short-summary: Load balancer outbound IP prefix resource IDs.
    long-summary: Comma-separated public IP prefix resource IDs for load balancer outbound connection. Valid for Standard SKU load balancer cluster only.
  - name: --load-balancer-outbound-ports
    type: int
    short-summary: Load balancer outbound allocated ports.
    long-summary: Desired static number of outbound ports per VM in the load balancer backend pool. By default, set to 0 which uses the default allocation based on the number of VMs.
  - name: --load-balancer-idle-timeout
    type: int
    short-summary: Load balancer idle timeout in minutes.
    long-summary: Desired idle timeout for load balancer outbound flows, default is 30 minutes. Please specify a value in the range of [4, 100].
  - name: --load-balancer-backend-pool-type
    type: string
    short-summary: Load balancer backend pool type.
    long-summary: Define the LoadBalancer backend pool type of managed inbound backend pool. The nodeIP means the VMs will be attached to the LoadBalancer by adding its private IP address to the backend pool. The nodeIPConfiguration means the VMs will be attached to the LoadBalancer by referencing the backend pool ID in the VM's NIC.
  - name: --nat-gateway-managed-outbound-ip-count
    type: int
    short-summary: NAT gateway managed outbound IP count.
    long-summary: Desired number of managed outbound IPs for NAT gateway outbound connection. Please specify a value in the range of [1, 16]. Valid for Standard SKU load balancer cluster with managedNATGateway outbound type only.
  - name: --nat-gateway-idle-timeout
    type: int
    short-summary: NAT gateway idle timeout in minutes.
    long-summary: Desired idle timeout for NAT gateway outbound flows, default is 4 minutes. Please specify a value in the range of [4, 120]. Valid for Standard SKU load balancer cluster with managedNATGateway outbound type only.
  - name: --outbound-type
    type: string
    short-summary: How outbound traffic will be configured for a cluster.
    long-summary: This option will change the way how the outbound connections are managed in the AKS cluster. Available options are loadbalancer, managedNATGateway, userAssignedNATGateway, userDefinedRouting and none. For custom vnet, loadbalancer, userAssignedNATGateway and userDefinedRouting are supported. For aks managed vnet, loadbalancer, managedNATGateway and userDefinedRouting are supported.
  - name: --auto-upgrade-channel
    type: string
    short-summary: Specify the upgrade channel for autoupgrade.
  - name: --node-os-upgrade-channel
    type: string
    short-summary: Manner in which the OS on your nodes is updated. It could be NodeImage, None, SecurityPatch or Unmanaged.
  - name: --attach-acr
    type: string
    short-summary: Grant the 'acrpull' role assignment to the ACR specified by name or resource ID.
  - name: --detach-acr
    type: string
    short-summary: Disable the 'acrpull' role assignment to the ACR specified by name or resource ID.
  - name: --api-server-authorized-ip-ranges
    type: string
    short-summary: Comma-separated list of authorized apiserver IP ranges. Set to "" to allow all traffic on a previously restricted cluster. Set to 0.0.0.0/32 to restrict apiserver traffic to node pools.
  - name: --enable-aad
    type: bool
    short-summary: Enable managed AAD feature for cluster.
  - name: --aad-admin-group-object-ids
    type: string
    short-summary: Comma-separated list of aad group object IDs that will be set as cluster admin.
  - name: --aad-tenant-id
    type: string
    short-summary: The ID of an Azure Active Directory tenant.
  - name: --enable-ahub
    type: bool
    short-summary: Enable Azure Hybrid User Benefits (AHUB) feature for cluster.
  - name: --disable-ahub
    type: bool
    short-summary: Disable Azure Hybrid User Benefits (AHUB) feature for cluster.
  - name: --windows-admin-password
    type: string
    short-summary: User account password to use on windows node VMs.
    long-summary: |-
      Rules for windows-admin-password:
          - Minimum-length: 14 characters
          - Max-length: 123 characters
          - Complexity requirements: 3 out of 4 conditions below need to be fulfilled
            * Has lower characters
            * Has upper characters
            * Has a digit
            * Has a special character (Regex match [\\W_])
          - Disallowed values:  "abc@123", "P@$$w0rd", "P@ssw0rd", "P@ssword123", "Pa$$word", "pass@word1", "Password!", "Password1", "Password22", "iloveyou!"
      Reference: https://learn.microsoft.com/dotnet/api/microsoft.azure.management.compute.models.virtualmachinescalesetosprofile.adminpassword?view=azure-dotnet
  - name: --enable-managed-identity
    type: bool
    short-summary: Update current cluster to use managed identity to manage cluster resource group.
  - name: --assign-identity
    type: string
    short-summary: Specify an existing user assigned identity to manage cluster resource group.
  - name: --assign-kubelet-identity
    type: string
    short-summary: Update cluster's kubelet identity to an existing user assigned identity. Please note this operation will recreate all agent nodes in the cluster.
  - name: --enable-azure-rbac
    type: bool
    short-summary: Enable Azure RBAC to control authorization checks on cluster.
  - name: --disable-azure-rbac
    type: bool
    short-summary: Disable Azure RBAC to control authorization checks on cluster.
  - name: --enable-public-fqdn
    type: bool
    short-summary: Enable public fqdn feature for private cluster.
  - name: --disable-public-fqdn
    type: bool
    short-summary: Disable public fqdn feature for private cluster.
  - name: --private-dns-zone
    type: string
    short-summary: The private dns zone mode for private cluster.
    long-summary: |-
        Only allow changing private dns zone from byo/system mode to none for private cluster. Others are denied.
  - name: --disable-local-accounts
    type: bool
    short-summary: If set to true, getting static credential will be disabled for this cluster.
  - name: --enable-local-accounts
    type: bool
    short-summary: If set to true, will enable getting static credential for this cluster.
  - name: --tags
    type: string
    short-summary: The tags of the managed cluster. The managed cluster instance and all resources managed by the cloud provider will be tagged.
  - name: --aks-custom-headers
    type: string
    short-summary: Comma-separated key-value pairs to specify custom headers.
  - name: --enable-workload-identity
    type: bool
    short-summary: Enable workload identity addon.
  - name: --disable-workload-identity
    type: bool
    short-summary: Disable workload identity addon.
  - name: --enable-secret-rotation
    type: bool
    short-summary: Enable secret rotation. Use with azure-keyvault-secrets-provider addon.
  - name: --disable-secret-rotation
    type: bool
    short-summary: Disable secret rotation. Use with azure-keyvault-secrets-provider addon.
  - name: --rotation-poll-interval
    type: string
    short-summary: Set interval of rotation poll. Use with azure-keyvault-secrets-provider addon.
  - name: --k8s-support-plan
    type: string
    short-summary: Choose from "KubernetesOfficial" or "AKSLongTermSupport", with "AKSLongTermSupport" you get 1 extra year of CVE patchs.
  - name: --enable-windows-gmsa
    type: bool
    short-summary: Enable Windows gmsa on cluster.
  - name: --gmsa-dns-server
    type: string
    short-summary: Specify DNS server for Windows gmsa on cluster.
    long-summary: |-
        You do not need to set this if you have set DNS server in the VNET used by the cluster.
        You must set or not set --gmsa-dns-server and --gmsa-root-domain-name at the same time when setting --enable-windows-gmsa.
  - name: --gmsa-root-domain-name
    type: string
    short-summary: Specify root domain name for Windows gmsa on cluster.
    long-summary: |-
        You do not need to set this if you have set DNS server in the VNET used by the cluster.
        You must set or not set --gmsa-dns-server and --gmsa-root-domain-name at the same time when setting --enable-windows-gmsa.
  - name: --disable-windows-gmsa
    type: bool
    short-summary: Disable Windows gmsa on cluster.
  - name: --enable-defender
    type: bool
    short-summary: Enable Microsoft Defender security profile.
  - name: --disable-defender
    type: bool
    short-summary: Disable defender profile.
  - name: --ca-certs --custom-ca-trust-certificates
    type: string
    short-summary: Path to a file containing up to 10 blank line separated certificates. Only valid for Linux nodes.
    long-summary: These certificates are used by Custom CA Trust feature and will be added to trust stores of nodes.
  - name: --defender-config
    type: string
    short-summary: Path to JSON file containing Microsoft Defender profile configurations.
  - name: --enable-azure-keyvault-kms
    type: bool
    short-summary: Enable Azure KeyVault Key Management Service.
  - name: --disable-azure-keyvault-kms
    type: bool
    short-summary: Disable Azure KeyVault Key Management Service.
  - name: --azure-keyvault-kms-key-id
    type: string
    short-summary: Identifier of Azure Key Vault key.
  - name: --azure-keyvault-kms-key-vault-network-access
    type: string
    short-summary: Network Access of Azure Key Vault.
    long-summary: Allowed values are "Public", "Private". If not set, defaults to type "Public". Requires --azure-keyvault-kms-key-id to be used.
  - name: --azure-keyvault-kms-key-vault-resource-id
    type: string
    short-summary: Resource ID of Azure Key Vault.
  - name: --enable-image-cleaner
    type: bool
    short-summary: Enable ImageCleaner Service.
  - name: --disable-image-cleaner
    type: bool
    short-summary: Disable ImageCleaner Service.
  - name: --image-cleaner-interval-hours
    type: int
    short-summary: ImageCleaner scanning interval.
  - name: --enable-disk-driver
    type: bool
    short-summary: Enable AzureDisk CSI Driver.
  - name: --disable-disk-driver
    type: bool
    short-summary: Disable AzureDisk CSI Driver.
  - name: --enable-file-driver
    type: bool
    short-summary: Enable AzureFile CSI Driver.
  - name: --disable-file-driver
    type: bool
    short-summary: Disable AzureFile CSI Driver.
  - name: --enable-snapshot-controller
    type: bool
    short-summary: Enable Snapshot Controller.
  - name: --disable-snapshot-controller
    type: bool
    short-summary: Disable CSI Snapshot Controller.
  - name: --enable-blob-driver
    type: bool
    short-summary: Enable AzureBlob CSI Driver.
  - name: --disable-blob-driver
    type: bool
    short-summary: Disable AzureBlob CSI Driver.
  - name: --http-proxy-config
    type: string
    short-summary: HTTP Proxy configuration for this cluster.
  - name: --enable-oidc-issuer
    type: bool
    short-summary: Enable OIDC issuer.
  - name: --enable-keda
    type: bool
    short-summary: Enable KEDA workload auto-scaler.
  - name: --disable-keda
    type: bool
    short-summary: Disable KEDA workload auto-scaler.
  - name: --enable-azure-monitor-metrics
    type: bool
    short-summary: Enable a kubernetes cluster with the Azure Monitor managed service for Prometheus integration.
  - name: --azure-monitor-workspace-resource-id
    type: string
    short-summary: Resource ID of the Azure Monitor Workspace
  - name: --ksm-metric-labels-allow-list
    type: string
    short-summary: Comma-separated list of additional Kubernetes label keys that will be used in the resource' labels metric. By default the metric contains only name and namespace labels. To include additional labels provide a list of resource names in their plural form and Kubernetes label keys you would like to allow for them (e.g. '=namespaces=[k8s-label-1,k8s-label-n,...],pods=[app],...)'. A single '*' can be provided per resource instead to allow any labels, but that has severe performance implications (e.g. '=pods=[*]').
  - name: --ksm-metric-annotations-allow-list
    type: string
    short-summary: Comma-separated list of additional Kubernetes label keys that will be used in the resource' labels metric. By default the metric contains only name and namespace labels. To include additional labels provide a list of resource names in their plural form and Kubernetes label keys you would like to allow for them (e.g.'=namespaces=[k8s-label-1,k8s-label-n,...],pods=[app],...)'. A single '*' can be provided per resource instead to allow any labels, but that has severe performance implications (e.g. '=pods=[*]').
  - name: --grafana-resource-id
    type: string
    short-summary: Resource ID of the Azure Managed Grafana Workspace
  - name: --enable-windows-recording-rules
    type: bool
    short-summary: Enable Windows Recording Rules when enabling the Azure Monitor Metrics addon
  - name: --disable-azure-monitor-metrics
    type: bool
    short-summary: Disable Azure Monitor Metrics Profile. This will delete all DCRA's associated with the cluster, any linked DCRs with the data stream = prometheus-stream and the recording rule groups created by the addon for this AKS cluster.
  - name: --nodepool-taints
    type: string
    short-summary: The node taints for all node pool.
  - name: --nodepool-labels
    type: string
    short-summary: The node labels for all node pool. See https://aka.ms/node-labels for syntax of labels.
  - name: --enable-vpa
    type: bool
    short-summary: Enable vertical pod autoscaler for cluster.
  - name: --disable-vpa
    type: bool
    short-summary: Disable vertical pod autoscaler for cluster.
  - name: --enable-force-upgrade
    type: bool
    short-summary: Enable forceUpgrade cluster upgrade settings override.
  - name: --disable-force-upgrade
    type: bool
    short-summary: Disable forceUpgrade cluster upgrade settings override.
  - name: --upgrade-override-until
    type: string
    short-summary: Until when the cluster upgradeSettings overrides are effective. It needs to be in a valid date-time format that's within the next 30 days. For example, 2023-04-01T13:00:00Z. Note that if --force-upgrade is set to true and --upgrade-override-until is not set, by default it will be set to 3 days from now.
  - name: --enable-cost-analysis
    type: bool
    short-summary: Enable exporting Kubernetes Namespace and Deployment details to the Cost Analysis views in the Azure portal. For more information see aka.ms/aks/docs/cost-analysis.
  - name: --disable-cost-analysis
    type: bool
    short-summary: Disable exporting Kubernetes Namespace and Deployment details to the Cost Analysis views in the Azure portal.
  - name: --enable-acns
    type: bool
    short-summary: Enable advanced network functionalities on a cluster. Enabling this will incur additional costs. For non-cilium clusters, acns security will be disabled by default until further notice.
  - name: --disable-acns
    type: bool
    short-summary: Disable all advanced networking functionalities on a cluster.
  - name: --disable-acns-observability
    type: bool
    short-summary: Used to disable advanced networking observability features on a clusters when enabling advanced networking features with "--enable-acns".
  - name: --disable-acns-security
    type: bool
    short-summary: Used to disable advanced networking security features on a clusters when enabling advanced networking features with "--enable-acns".
  - name: --nrg-lockdown-restriction-level
    type: string
    short-summary: Restriction level on the managed node resource group.
    long-summary: The restriction level of permissions allowed on the cluster's managed node resource group, supported values are Unrestricted, and ReadOnly (recommended ReadOnly).
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the managed cluster, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new cluster.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new cluster to be created, but to prevent updating an existing cluster. Other values will be ignored.
  - name: --bootstrap-artifact-source
    type: string
    short-summary: Configure artifact source when bootstraping the cluster.
    long-summary: |
        The artifacts include the addon image. Use "Direct" to download artifacts from MCR, "Cache" to downalod artifacts from Azure Container Registry.
  - name: --bootstrap-container-registry-resource-id
    type: string
    short-summary: Configure container registry resource ID. Must use "Cache" as bootstrap artifact source.
examples:
  - name: Reconcile the cluster back to its current state.
    text: az aks update -g MyResourceGroup -n MyManagedCluster
  - name: Update a kubernetes cluster with standard SKU load balancer to use two AKS created IPs for the load balancer outbound connection usage.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --load-balancer-managed-outbound-ip-count 2
  - name: Update a kubernetes cluster with standard SKU load balancer to use the provided public IPs for the load balancer outbound connection usage.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --load-balancer-outbound-ips <ip-resource-id-1,ip-resource-id-2>
  - name: Update a kubernetes cluster with a standard SKU load balancer, with two outbound AKS managed IPs an idle flow timeout of 5 minutes and 8000 allocated ports per machine
    text: az aks update -g MyResourceGroup -n MyManagedCluster --load-balancer-managed-outbound-ip-count 2 --load-balancer-idle-timeout 5 --load-balancer-outbound-ports 8000
  - name: Update a kubernetes cluster with standard SKU load balancer to use the provided public IP prefixes for the load balancer outbound connection usage.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --load-balancer-outbound-ip-prefixes <ip-prefix-resource-id-1,ip-prefix-resource-id-2>
  - name: Update a kubernetes cluster of managedNATGateway outbound type with two outbound AKS managed IPs an idle flow timeout of 4 minutes
    text: az aks update -g MyResourceGroup -n MyManagedCluster --nat-gateway-managed-outbound-ip-count 2 --nat-gateway-idle-timeout 4
  - name: Attach AKS cluster to ACR by name "acrName"
    text: az aks update -g MyResourceGroup -n MyManagedCluster --attach-acr acrName
  - name: Update a kubernetes cluster with authorized apiserver ip ranges.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --api-server-authorized-ip-ranges 193.168.1.0/24,194.168.1.0/24
  - name: Disable authorized apiserver ip ranges feature for a kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --api-server-authorized-ip-ranges ""
  - name: Restrict apiserver traffic in a kubernetes cluster to agentpool nodes.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --api-server-authorized-ip-ranges 0.0.0.0/32
  - name: Update a AKS-managed AAD cluster with tenant ID or admin group object IDs.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --aad-admin-group-object-ids <id-1,id-2> --aad-tenant-id <id>
  - name: Migrate a AKS AAD-Integrated cluster or a non-AAD cluster to a AKS-managed AAD cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-aad --aad-admin-group-object-ids <id-1,id-2> --aad-tenant-id <id>
  - name: Enable Azure Hybrid User Benefits featture for a kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-ahub
  - name: Disable Azure Hybrid User Benefits featture for a kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --disable-ahub
  - name: Update Windows password of a kubernetes cluster
    text: az aks update -g MyResourceGroup -n MyManagedCLuster --windows-admin-password "Repl@cePassw0rd12345678"
  - name: Update the cluster to use system assigned managed identity in control plane.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-managed-identity
  - name: Update the cluster to use user assigned managed identity in control plane.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-managed-identity --assign-identity <user_assigned_identity_resource_id>
  - name: Update a non managed AAD AKS cluster to use Azure RBAC
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-aad --enable-azure-rbac
  - name: Update a managed AAD AKS cluster to use Azure RBAC
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-azure-rbac
  - name: Disable Azure RBAC in a managed AAD AKS cluster
    text: az aks update -g MyResourceGroup -n MyManagedCluster --disable-azure-rbac
  - name: Update the tags of a kubernetes cluster
    text: az aks update -g MyResourceGroup -n MyManagedCLuster --tags "foo=bar" "baz=qux"
  - name: Update a kubernetes cluster with custom headers
    text: az aks update -g MyResourceGroup -n MyManagedCluster --aks-custom-headers WindowsContainerRuntime=containerd,AKSHTTPCustomFeatures=Microsoft.ContainerService/CustomNodeConfigPreview
  - name: Enable Windows gmsa for a kubernetes cluster with setting DNS server in the vnet used by the cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-windows-gmsa
  - name: Enable Windows gmsa for a kubernetes cluster without setting DNS server in the vnet used by the cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-windows-gmsa --gmsa-dns-server "10.240.0.4" --gmsa-root-domain-name "contoso.com"
  - name: Disable Windows gmsa for a kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --disable-windows-gmsa
  - name: Enable KEDA workload autoscaler for an existing kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --enable-keda
  - name: Disable KEDA workload autoscaler for an existing kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCluster --disable-keda
  - name: Enable VPA(Vertical Pod Autoscaler) for an existing kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCLuster --enable-vpa
  - name: Disable VPA(Vertical Pod Autoscaler) for an existing kubernetes cluster.
    text: az aks update -g MyResourceGroup -n MyManagedCLuster --disable-vpa
"""

helps['aks delete'] = """
type: command
short-summary: Delete a managed Kubernetes cluster.
examples:
  - name: Delete a managed Kubernetes cluster. (autogenerated)
    text: az aks delete --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks disable-addons'] = """
type: command
short-summary: Disable Kubernetes addons.
parameters:
  - name: --addons -a
    type: string
    short-summary: Disable the Kubernetes addons in a comma-separated list.
examples:
  - name: Disable Kubernetes addons. (autogenerated)
    text: az aks disable-addons --addons virtual-node --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks enable-addons'] = """
type: command
short-summary: Enable Kubernetes addons.
long-summary: |-
    These addons are available:
    - http_application_routing : configure ingress with automatic public DNS name creation.
    - monitoring               : turn on Log Analytics monitoring. Requires "--workspace-resource-id".
                                 Requires "--enable-msi-auth-for-monitoring" for managed identity auth.
                                 Requires "--enable-syslog" to enable syslog data collection from nodes. Note MSI must be enabled.
                                 Requires "--ampls-resource-id" for private link. Note MSI must be enabled.
                                 Requires "--enable-high-log-scale-mode" to enable high log scale mode for container logs. Note MSI must be enabled.
                                 If monitoring addon is enabled --no-wait argument will have no effect
    - virtual-node             : enable AKS Virtual Node. Requires --subnet-name to provide the name of an existing subnet for the Virtual Node to use.
    - azure-policy             : enable Azure policy. The Azure Policy add-on for AKS enables at-scale enforcements and safeguards on your clusters in a centralized, consistent manner.
                                 Learn more at aka.ms/aks/policy.
    - ingress-appgw            : enable Application Gateway Ingress Controller addon.
    - open-service-mesh        : enable Open Service Mesh addon.
    - azure-keyvault-secrets-provider : enable Azure Keyvault Secrets Provider addon.
parameters:
  - name: --addons -a
    type: string
    short-summary: Enable the Kubernetes addons in a comma-separated list.
  - name: --workspace-resource-id
    type: string
    short-summary: The resource ID of an existing Log Analytics Workspace to use for storing monitoring data.
  - name: --enable-msi-auth-for-monitoring
    type: bool
    short-summary: Enable Managed Identity Auth for Monitoring addon.
  - name: --enable-syslog
    type: bool
    short-summary: Enable syslog data collection for Monitoring addon
  - name: --data-collection-settings
    type: string
    short-summary: Path to JSON file containing data collection settings for Monitoring addon.
  - name: --ampls-resource-id
    type: string
    short-summary: Resource ID of Azure Monitor Private Link scope for Monitoring Addon.
  - name: --enable-high-log-scale-mode
    type: bool
    short-summary: Enable High Log Scale Mode for Container Logs.
  - name: --appgw-name
    type: string
    short-summary: Name of the application gateway to create/use in the node resource group. Use with ingress-azure addon.
  - name: --appgw-subnet-cidr
    type: string
    short-summary: Subnet CIDR to use for a new subnet created to deploy the Application Gateway. Use with ingress-azure addon.
  - name: --appgw-id
    type: string
    short-summary: Resource Id of an existing Application Gateway to use with AGIC. Use with ingress-azure addon.
  - name: --appgw-subnet-id
    type: string
    short-summary: Resource Id of an existing Subnet used to deploy the Application Gateway. Use with ingress-azure addon.
  - name: --appgw-watch-namespace
    type: string
    short-summary: Specify the namespace, which AGIC should watch. This could be a single string value, or a comma-separated list of namespaces.
  - name: --enable-sgxquotehelper
    type: bool
    short-summary: Enable SGX quote helper for confcom addon.
  - name: --enable-secret-rotation
    type: bool
    short-summary: Enable secret rotation. Use with azure-keyvault-secrets-provider addon.
  - name: --rotation-poll-interval
    type: string
    short-summary: Set interval of rotation poll. Use with azure-keyvault-secrets-provider addon.
examples:
  - name: Enable Kubernetes addons. (autogenerated)
    text: az aks enable-addons --addons virtual-node --name MyManagedCluster --resource-group MyResourceGroup --subnet MySubnetName
    crafted: true
  - name: Enable ingress-appgw addon with subnet prefix.
    text: az aks enable-addons --name MyManagedCluster --resource-group MyResourceGroup --addons ingress-appgw --appgw-subnet-cidr 10.225.0.0/16 --appgw-name gateway
    crafted: true
  - name: Enable open-service-mesh addon.
    text: az aks enable-addons --name MyManagedCluster --resource-group MyResourceGroup --addons open-service-mesh
    crafted: true
"""

helps['aks get-credentials'] = """
type: command
short-summary: Get access credentials for a managed Kubernetes cluster.
long-summary: By default, the credentials are merged into the .kube/config file so kubectl can use them.  See -f parameter for details.
parameters:
  - name: --admin -a
    type: bool
    short-summary: "Get cluster administrator credentials.  Default: cluster user credentials."
    long-summary: "On clusters with Azure Active Directory integration, this bypasses normal Azure AD authentication and can be used if you're permanently blocked by not having access to a valid Azure AD group with access to your cluster. Requires 'Azure Kubernetes Service Cluster Admin' role."
  - name: --file -f
    type: string
    short-summary: Kubernetes configuration file to update. Use "-" to print YAML to stdout instead.
  - name: --overwrite-existing
    type: bool
    short-summary: Overwrite any existing cluster entry with the same name.
  - name: --output -o
    type: string
    long-summary: Credentials are always in YAML format, so this argument is effectively ignored.
  - name: --public-fqdn
    type: bool
    short-summary: Get private cluster credential with server address to be public fqdn.
  - name: --format
    type: string
    short-summary: Specify the format of the returned credential. Available values are ["exec", "azure"].
                  Only take effect when requesting clusterUser credential of AAD clusters.
examples:
  - name: Get access credentials for a managed Kubernetes cluster. (autogenerated)
    text: az aks get-credentials --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks get-upgrades'] = """
type: command
short-summary: Get the upgrade versions available for a managed Kubernetes cluster.
examples:
  - name: Get the upgrade versions available for a managed Kubernetes cluster
    text: az aks get-upgrades --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks get-versions'] = """
type: command
short-summary: Get the versions available for creating a managed Kubernetes cluster.
examples:
  - name: Get the versions available for creating a managed Kubernetes cluster
    text: az aks get-versions --location westus2
    crafted: true
"""

helps['aks install-cli'] = """
type: command
short-summary: Download and install kubectl, the Kubernetes command-line tool. Download and install kubelogin, a client-go credential (exec) plugin implementing azure authentication.
"""

helps['aks list'] = """
type: command
short-summary: List managed Kubernetes clusters.
"""

helps['aks maintenanceconfiguration'] = """
    type: group
    short-summary: Commands to manage maintenance configurations in managed Kubernetes cluster.
"""

helps['aks maintenanceconfiguration show'] = """
    type: command
    short-summary: show the details of a maintenance configuration in managed Kubernetes cluster.
"""

helps['aks maintenanceconfiguration delete'] = """
    type: command
    short-summary: Delete a maintenance configuration in managed Kubernetes cluster.
"""

helps['aks maintenanceconfiguration list'] = """
    type: command
    short-summary: List maintenance configurations in managed Kubernetes cluster.
"""

helps['aks maintenanceconfiguration add'] = """
    type: command
    short-summary: Add a maintenance configuration in managed Kubernetes cluster.
    parameters:
        - name: --weekday
          type: string
          short-summary: A day in week on which maintenance is allowed. E.g. Monday. Applicable to default maintenance configuration only.
        - name: --start-hour
          type: string
          short-summary: The start time of 1 hour window which maintenance is allowd. E.g. 1 means it's allowd between 1:00 am and 2:00 am. Applicable to default maintenance configuration only.
        - name: --schedule-type
          type: string
          short-summary: Choose either 'Daily', 'Weekly', 'AbsoluteMonthly' or 'RelativeMonthly' for your maintenance schedule. Only applicable to 'aksManagedAutoUpgradeSchedule' and 'aksManagedNodeOSUpgradeSchedule' maintenance configuration.
        - name: --start-date
          type: string
          short-summary: The date the maintenance configuration activates. If not specified, the maintenance window will be active right away."
        - name: --start-time
          type: string
          short-summary: The start time of the maintenance window. Accepted values are from '00:00' to '23:59'. '--utc-offset' applies to this field. For example, '02:00' with '--utc-offset +02:00' means UTC time '00:00'.
        - name: --duration
          type: int
          short-summary: The length of maintenance window range from 4 to 24 hours.
        - name: --utc-offset
          type: string
          short-summary: The UTC offset in format +/-HH:mm. For example, '+05:30' for IST and '-07:00' for PST. If not specified, the default is '+00:00'.
        - name: --interval-days
          type: int
          short-summary: The number of days between each set of occurrences for daily schedule type.
        - name: --interval-weeks
          type: int
          short-summary: The number of weeks between each set of occurrences. Applicable to weekly schedule types only.
        - name: --interval-months
          type: int
          short-summary: The number of months between each set of occurrences. Applicable to absolute and relative monthly schedule types.
        - name: --day-of-week
          type: string
          short-summary: Specify on which day of the week the maintenance occurs. E.g. "Monday". Applicable to weekly and relative monthly schedule types.
        - name: --day-of-month
          type: int
          short-summary: Specify on which day of the month the maintenance occurs. E.g. 1 indicates the 1st of the month. Applicable to absolute monthly schedule type only.
        - name: --week-index
          type: string
          short-summary: Specify on which instance of the allowed days specified in '--day-of-week' the maintenance occurs. Applicable to relative monthly schedule type only.
        - name: --config-file
          type: string
          short-summary: The maintenance configuration json file.
    examples:
        - name: Add default maintenance configuration with --weekday and --start-hour.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n default --weekday Monday  --start-hour 1
              The maintenance is allowed on Monday 1:00am to 2:00am
        - name: Add default maintenance configuration with --weekday. The maintenance is allowd on any time of that day.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n default --weekday Monday
              The maintenance is allowed on Monday.
        - name: Add default maintenance configuration with maintenance configuration json file
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n default --config-file ./test.json
                The content of json file looks below. It means the maintenance is allowed on UTC time Tuesday 1:00am - 3:00 am and Wednesday 1:00am - 2:00am, 6:00am-7:00am
                No maintenance is allowed from 2020-11-26T03:00:00Z to 2020-11-30T12:00:00Z and from 2020-12-26T03:00:00Z to 2020-12-26T12:00:00Z even if they are allowed in the above weekly setting
                {
                      "timeInWeek": [
                        {
                          "day": "Tuesday",
                          "hourSlots": [
                            1,
                            2
                          ]
                        },
                        {
                          "day": "Wednesday",
                          "hourSlots": [
                            1,
                            6
                          ]
                        }
                      ],
                      "notAllowedTime": [
                        {
                          "start": "2021-11-26T03:00:00Z",
                          "end": "2021-11-30T12:00:00Z"
                        },
                        {
                          "start": "2021-12-26T03:00:00Z",
                          "end": "2021-12-26T12:00:00Z"
                        }
                      ]
              }
        - name: Add aksManagedNodeOSUpgradeSchedule maintenance configuration with daily schedule.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n aksManagedNodeOSUpgradeSchedule --schedule-type Daily --interval-days 2 --duration 12 --utc-offset=-08:00 --start-date 2023-01-16 --start-time 00:00
              The maintenance is allowed from 00:00 to 12:00 (adjusted with UTC offset: -08:00) every two days, and this configuration will be effective from 2023-01-16.
        - name: Add aksManagedNodeOSUpgradeSchedule maintenance configuration with weekly schedule.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n aksManagedNodeOSUpgradeSchedule --schedule-type Weekly --day-of-week Friday --interval-weeks 3 --duration 8 --utc-offset +05:30 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on Friday from 09:30 to 17:30 (adjusted with UTC offset: +05:30) every three weeks, and this configuration will be effective from 2023-01-16.
        - name: Add aksManagedAutoUpgradeSchedule maintenance configuration with absolute monthly schedule.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --schedule-type AbsoluteMonthly --day-of-month 15 --interval-months 1 --duration 6 --utc-offset +05:30 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on the 15th of the month from 09:30 to 15:30 (adjusted with UTC offset: +05:30) every month, and this configuration will be effective from 2023-01-16.
        - name: Add aksManagedAutoUpgradeSchedule maintenance configuration with relative monthly schedule.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --schedule-type RelativeMonthly --day-of-week Tuesday --week-index Last --interval-months 3 --duration 6 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on the last Tuesday from 09:30 to 15:30 in default UTC time every 3 months, and this configuration will be effective from 2023-01-16.
        - name: Add aksManagedAutoUpgradeSchedule maintenance configuration with json file.
          text: |
            az aks maintenanceconfiguration add -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --config-file ./test.json
                The content of json file looks below. It means the maintenance is allowed on the 1st of the month from 09:00 to 13:00 (adjusted with UTC offset: -08:00) every 3 months, and this configuration will be effective from 2023-01-16.
                No maintenance is allowed from 2022-12-23 to 2023-01-05 and from 2023-11-23 to 2023-11-26 even if they are allowed in the above monthly setting
                {
                    "maintenanceWindow": {
                        "schedule": {
                            "absoluteMonthly": {
                                "intervalMonths": 3,
                                "dayOfMonth": 1
                            }
                        },
                        "durationHours": 4,
                        "utcOffset": "-08:00",
                        "startTime": "09:00",
                        "notAllowedDates": [
                            {
                                "start": "2022-12-23",
                                "end": "2023-01-05"
                            },
                            {
                                "start": "2023-11-23",
                                "end": "2023-11-26"
                            }
                        ]
                    }
                }
"""

helps['aks maintenanceconfiguration update'] = """
    type: command
    short-summary: Update a maintenance configuration of a managed Kubernetes cluster.
    parameters:
        - name: --weekday
          type: string
          short-summary: A day in week on which maintenance is allowed. E.g. Monday. Applicable to default maintenance configuration only.
        - name: --start-hour
          type: string
          short-summary: The start time of 1 hour window which maintenance is allowd. E.g. 1 means it's allowd between 1:00 am and 2:00 am. Applicable to default maintenance configuration only.
        - name: --schedule-type
          type: string
          short-summary: Choose either 'Daily', 'Weekly', 'AbsoluteMonthly' or 'RelativeMonthly' for your maintenance schedule. Only applicable to 'aksManagedAutoUpgradeSchedule' and 'aksManagedNodeOSUpgradeSchedule' maintenance configuration.
        - name: --start-date
          type: string
          short-summary: The date the maintenance configuration activates. If not specified, the maintenance window will be active right away."
        - name: --start-time
          type: string
          short-summary: The start time of the maintenance window. Accepted values are from '00:00' to '23:59'. '--utc-offset' applies to this field. For example, '02:00' with '--utc-offset +02:00' means UTC time '00:00'.
        - name: --duration
          type: int
          short-summary: The length of maintenance window range from 4 to 24 hours.
        - name: --utc-offset
          type: string
          short-summary: The UTC offset in format +/-HH:mm. For example, '+05:30' for IST and '-07:00' for PST. If not specified, the default is '+00:00'.
        - name: --interval-days
          type: int
          short-summary: The number of days between each set of occurrences for daily schedule type.
        - name: --interval-weeks
          type: int
          short-summary: The number of weeks between each set of occurrences. Applicable to weekly schedule types only.
        - name: --interval-months
          type: int
          short-summary: The number of months between each set of occurrences. Applicable to absolute and relative monthly schedule types.
        - name: --day-of-week
          type: string
          short-summary: Specify on which day of the week the maintenance occurs. E.g. "Monday". Applicable to weekly and relative monthly schedule types.
        - name: --day-of-month
          type: int
          short-summary: Specify on which day of the month the maintenance occurs. E.g. 1 indicates the 1st of the month. Applicable to absolute monthly schedule type only.
        - name: --week-index
          type: string
          short-summary: Specify on which instance of the allowed days specified in '--day-of-week' the maintenance occurs. Applicable to relative monthly schedule type only.
        - name: --config-file
          type: string
          short-summary: The maintenance configuration json file.
    examples:
        - name: Update default maintenance configuration with --weekday and --start-hour.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n default --weekday Monday  --start-hour 1
              The maintenance is allowed on Monday 1:00am to 2:00am
        - name: Update default maintenance configuration with --weekday.The maintenance is allowd on any time of that day.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n default --weekday Monday
              The maintenance is allowed on Monday.
        - name: Update default maintenance configuration with maintenance configuration json file
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n default --config-file ./test.json
                The content of json file looks below. It means the maintenance is allowed on UTC time Tuesday 1:00am - 3:00 am and Wednesday 1:00am - 2:00am, 6:00am-7:00am
                No maintenance is allowed from 2020-11-26T03:00:00Z to 2020-11-30T12:00:00Z and from 2020-12-26T03:00:00Z to 2020-12-26T12:00:00Z even if they are allowed in the above weekly setting
                {
                      "timeInWeek": [
                        {
                          "day": "Tuesday",
                          "hourSlots": [
                            1,
                            2
                          ]
                        },
                        {
                          "day": "Wednesday",
                          "hourSlots": [
                            1,
                            6
                          ]
                        }
                      ],
                      "notAllowedTime": [
                        {
                          "start": "2021-11-26T03:00:00Z",
                          "end": "2021-11-30T12:00:00Z"
                        },
                        {
                          "start": "2021-12-26T03:00:00Z",
                          "end": "2021-12-26T12:00:00Z"
                        }
                      ]
              }
        - name: Update aksManagedNodeOSUpgradeSchedule maintenance configuration with daily schedule.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n aksManagedNodeOSUpgradeSchedule --schedule-type Daily --interval-days 2 --duration 12 --utc-offset=-08:00 --start-date 2023-01-16 --start-time 00:00
              The maintenance is allowed from 00:00 to 12:00 (adjusted with UTC offset: -08:00) every two days. This configuration will be effective from 2023-01-16.
        - name: Update aksManagedNodeOSUpgradeSchedule maintenance configuration with weekly schedule.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n aksManagedNodeOSUpgradeSchedule --schedule-type Weekly --day-of-week Friday --interval-weeks 3 --duration 8 --utc-offset +05:30 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on Friday from 09:30 to 17:30 (adjusted with UTC offset: +05:30) every three weeks. This configuration will be effective from 2023-01-16.
        - name: Update aksManagedAutoUpgradeSchedule maintenance configuration with absolute monthly schedule.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --schedule-type AbsoluteMonthly --day-of-month 15 --interval-months 1 --duration 6 --utc-offset +05:30 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on the 15th of the month from 09:30 to 15:30 (adjusted with UTC offset: +05:30) every month. This configuration will be effective from 2023-01-16.
        - name: Update aksManagedAutoUpgradeSchedule maintenance configuration with relative monthly schedule.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --schedule-type RelativeMonthly --day-of-week Tuesday --week-index Last --interval-months 3 --duration 6 --start-date 2023-01-16 --start-time 09:30
              The maintenance is allowed on the last Tuesday from 09:30 to 15:30 in default UTC time every 3 months. This configuration will be effective from 2023-01-16.
        - name: Update aksManagedAutoUpgradeSchedule maintenance configuration with json file.
          text: |
            az aks maintenanceconfiguration update -g MyResourceGroup --cluster-name test1 -n aksManagedAutoUpgradeSchedule --config-file ./test.json
                The content of json file looks below. It means the maintenance is allowed on the 1st of the month from 09:00 to 13:00 (adjusted with UTC offset: -08:00) every 3 months, and this configuration will be effective from 2023-01-16.
                No maintenance is allowed from 2022-12-23 to 2023-01-05 and from 2023-11-23 to 2023-11-26 even if they are allowed in the above monthly setting
                {
                    "maintenanceWindow": {
                        "schedule": {
                            "absoluteMonthly": {
                                "intervalMonths": 3,
                                "dayOfMonth": 1
                            }
                        },
                        "durationHours": 4,
                        "utcOffset": "-08:00",
                        "startTime": "09:00",
                        "notAllowedDates": [
                            {
                                "start": "2022-12-23",
                                "end": "2023-01-05"
                            },
                            {
                                "start": "2023-11-23",
                                "end": "2023-11-26"
                            }
                        ]
                    }
                }
"""

helps['aks nodepool'] = """
type: group
short-summary: Commands to manage node pools in Kubernetes kubernetes cluster.
"""

helps['aks nodepool add'] = """
type: command
short-summary: Add a node pool to the managed Kubernetes cluster.
parameters:
  - name: --node-vm-size -s
    type: string
    short-summary: Size of Virtual Machines to create as Kubernetes nodes.
  - name: --node-count -c
    type: int
    short-summary: Number of nodes in the Kubernetes agent pool. After creating a cluster, you can change the size of its node pool with `az aks scale`.
  - name: --kubernetes-version -k
    type: string
    short-summary: Version of Kubernetes to use for creating the cluster, such as "1.16.9".
    populator-commands:
      - "`az aks get-versions`"
  - name: --node-osdisk-size
    type: int
    short-summary: Size in GiB of the OS disk for each node in the agent pool. Minimum 30 GiB.
  - name: --node-osdisk-type
    type: string
    short-summary: OS disk type to be used for machines in a given agent pool. Defaults to 'Ephemeral' when possible in conjunction with VM size and OS disk size. May not be changed for this pool after creation. ('Ephemeral' or 'Managed')
  - name: --max-pods -m
    type: int
    short-summary: The maximum number of pods deployable to a node.
    long-summary: If not specified, defaults based on network-plugin. 30 for "azure", 110 for "kubenet", or 250 for "none".
  - name: --zones -z
    type: string array
    short-summary: Availability zones where agent nodes will be placed. Also, to install agent nodes to more than one zone you need to pass zone numbers separated by blanks.  For example -  To have all 3 zones, you are expected to enter `--zones 1 2 3`
  - name: --enable-node-public-ip
    type: bool
    short-summary: Enable VMSS node public IP.
  - name: --node-public-ip-prefix-id
    type: string
    short-summary: Public IP prefix ID used to assign public IPs to VMSS nodes.
  - name: --vnet-subnet-id
    type: string
    short-summary: The Resource Id of a subnet in an existing VNet into which to deploy the cluster.
  - name: --pod-subnet-id
    type: string
    short-summary: The Resource Id of a subnet in an existing VNet into which to assign pods in the cluster (requires azure network-plugin).
  - name: --ppg
    type: string
    short-summary: The ID of a PPG.
  - name: --os-type
    type: string
    short-summary: The OS Type. Linux or Windows.
  - name: --os-sku
    type: string
    short-summary: The OS SKU of the agent node pool. Ubuntu, AzureLinux or Ubuntu2204 for Linux. Windows2019 or Windows2022 for Windows.
  - name: --enable-cluster-autoscaler -e
    type: bool
    short-summary: Enable cluster autoscaler.
  - name: --min-count
    type: int
    short-summary: Minimum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [0, 1000] for user nodepool, and [1,1000] for system nodepool.
  - name: --max-count
    type: int
    short-summary: Maximum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [0, 1000] for user nodepool, and [1,1000] for system nodepool.
  - name: --scale-down-mode
    type: string
    short-summary: "Describe how VMs are added to or removed from nodepools."
  - name: --node-taints
    type: string
    short-summary: The node taints for the node pool.
  - name: --labels
    type: string
    short-summary: The node labels for the node pool. See https://aka.ms/node-labels for syntax of labels.
  - name: --mode
    type: string
    short-summary: The mode for a node pool which defines a node pool's primary function. If set as "System", AKS prefers system pods scheduling to node pools with mode `System`. Learn more at https://aka.ms/aks/nodepool/mode.
  - name: --priority
    type: string
    short-summary: The priority of the node pool.
  - name: --eviction-policy
    type: string
    short-summary: The eviction policy of the Spot node pool. It can only be set when --priority is Spot.
  - name: --spot-max-price
    type: float
    short-summary: It can only be set when --priority is Spot. Specify the maximum price you are willing to pay in US Dollars. Possible values are any decimal value greater than zero or -1 which indicates default price to be up-to on-demand. It can only include up to 5 decimal places.
  - name: --max-surge
    type: string
    short-summary: Extra nodes used to speed upgrade. When specified, it represents the number or percent used, eg. 5 or 33%
  - name: --drain-timeout
    type: int
    short-summary: When nodes are drain how many minutes to wait for all pods to be evicted
  - name: --node-soak-duration
    type: int
    short-summary: The amount of time (in minutes) to wait after draining a node and before reimaging it and moving on to next node.
  - name: --enable-encryption-at-host
    type: bool
    short-summary: Enable EncryptionAtHost, default value is false.
  - name: --enable-ultra-ssd
    type: bool
    short-summary: Enable UltraSSD, default value is false.
  - name: --enable-fips-image
    type: bool
    short-summary: Use FIPS-enabled OS on agent nodes.
  - name: --snapshot-id
    type: string
    short-summary: The source snapshot id used to create this nodepool.
  - name: --aks-custom-headers
    type: string
    short-summary: Comma-separated key-value pairs to specify custom headers.
  - name: --kubelet-config
    type: string
    short-summary: Path to JSON file containing Kubelet configurations for agent nodes. https://aka.ms/aks/custom-node-config
  - name: --linux-os-config
    type: string
    short-summary: Path to JSON file containing OS configurations for Linux agent nodes. https://aka.ms/aks/custom-node-config
  - name: --message-of-the-day
    type: string
    short-summary: Path to a file containing the desired message of the day. Only valid for linux nodes. Will be written to /etc/motd.
  - name: --host-group-id
    type: string
    short-summary: The fully qualified dedicated host group id used to provision agent node pool.
  - name: --gpu-instance-profile
    type: string
    short-summary: GPU instance profile to partition multi-gpu Nvidia GPUs.
  - name: --allowed-host-ports
    type: string
    short-summary: Expose host ports on the node pool. When specified, format should be a space-separated list of ranges with protocol, eg. `80/TCP 443/TCP 4000-5000/TCP`.
  - name: --asg-ids
    type: string
    short-summary: The IDs of the application security groups to which the node pool's network interface should belong. When specified, format should be a space-separated list of IDs.
  - name: --node-public-ip-tags
    type: string
    short-summary: The ipTags of the node public IPs.
  - name: --crg-id
    type: string
    short-summary: The crg id used to associate the new nodepool with the existed Capacity Reservation Group resource.
  - name: --disable-windows-outbound-nat
    type: bool
    short-summary: Disable Windows OutboundNAT on Windows agent node pool.
  - name: --enable-secure-boot
    type: bool
    short-summary: Enable Secure Boot on agent node pool. Must use VMSS agent pool type.
  - name: --enable-vtpm
    type: bool
    short-summary: Enable vTPM on agent node pool. Must use VMSS agent pool type.
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the agentpool, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new agentpool.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new agentpool to be created, but to prevent updating an existing agentpool. Other values will be ignored.
  - name: --gpu-driver
    type: string
    short-summary: Whether to install driver for GPU node pool. Possible values are "Install" or "None". Default is "Install".

examples:
  - name: Create a nodepool in an existing AKS cluster with ephemeral os enabled.
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --node-osdisk-type Ephemeral --node-osdisk-size 48
  - name: Create a nodepool with EncryptionAtHost enabled.
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --enable-encryption-at-host
  - name: Create a nodepool with UltraSSD enabled.
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --enable-ultra-ssd
  - name: Create a nodepool cluster with a specific OS SKU
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --os-sku Ubuntu
  - name: Create a nodepool with FIPS-enabled OS
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --enable-fips-image
  - name: create a kubernetes cluster with a snapshot id.
    text: az aks nodepool add -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster --kubernetes-version 1.20.9 --snapshot-id "/subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/snapshots/mysnapshot1"
  - name: create a nodepool in an existing AKS cluster with host group id
    text: az aks nodepool add -g MyResourceGroup -n MyNodePool --cluster-name MyMC --host-group-id /subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/hostGroups/myHostGroup --node-vm-size VMSize
  - name: create a nodepool with a Capacity Reservation Group(CRG) ID.
    text: az aks nodepool add -g MyResourceGroup -n MyNodePool --cluster-name MyMC --node-vm-size VMSize --crg-id "/subscriptions/SubID/resourceGroups/ResourceGroupName/providers/Microsoft.ContainerService/CapacityReservationGroups/MyCRGID"
"""

helps['aks nodepool delete'] = """
type: command
short-summary: Delete the agent pool in the managed Kubernetes cluster.
parameters:
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the node pool, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new agentpool.
  - name: --ignore-pdb -i
    type: bool
    short-summary: Delete an existing nodepool without considering Pod Disruption Budget.
examples:
  - name: Delete an agent pool with ignore-pdb
    text: az aks nodepool delete --resource-group MyResourceGroup --cluster-name MyManagedCluster --name nodepool1 --if-match etag --ignore-pdb
"""

helps['aks nodepool get-upgrades'] = """
type: command
short-summary: Get the available upgrade versions for an agent pool of the managed Kubernetes cluster.
examples:
  - name: Get the available upgrade versions for an agent pool of the managed Kubernetes cluster.
    text: az aks nodepool get-upgrades --resource-group MyResourceGroup --cluster-name MyManagedCluster --nodepool-name MyNodePool
    crafted: true
"""

helps['aks nodepool list'] = """
type: command
short-summary: List node pools in the managed Kubernetes cluster. To get list of nodes in the cluster run `kubectl get nodes` command.
"""

helps['aks nodepool scale'] = """
type: command
short-summary: Scale the node pool in a managed Kubernetes cluster.
parameters:
  - name: --node-count -c
    type: int
    short-summary: Number of nodes in the Kubernetes node pool.
"""

helps['aks nodepool show'] = """
type: command
short-summary: Show the details for a node pool in the managed Kubernetes cluster.
"""

helps['aks nodepool update'] = """
type: command
short-summary: Update a node pool properties.
long-summary: Update a node pool to enable/disable cluster-autoscaler or change min-count or max-count. When called with no optional arguments this attempts to move the node pool to its goal state without changing the current node pool configuration. This can be used to move out of a non succeeded state.
parameters:
  - name: --enable-cluster-autoscaler -e
    type: bool
    short-summary: Enable cluster autoscaler.
  - name: --disable-cluster-autoscaler -d
    type: bool
    short-summary: Disable cluster autoscaler.
  - name: --update-cluster-autoscaler -u
    type: bool
    short-summary: Update min-count or max-count for cluster autoscaler.
  - name: --min-count
    type: int
    short-summary: Minimum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [0, 1000] for user nodepool, and [1,1000] for system nodepool.
  - name: --max-count
    type: int
    short-summary: Maximum nodes count used for autoscaler, when "--enable-cluster-autoscaler" specified. Please specify the value in the range of [0, 1000] for user nodepool, and [1,1000] for system nodepool.
  - name: --scale-down-mode
    type: string
    short-summary: "Describe how VMs are added to or removed from nodepools."
  - name: --mode
    type: string
    short-summary: The mode for a node pool which defines a node pool's primary function. If set as "System", AKS prefers system pods scheduling to node pools with mode `System`. Learn more at https://aka.ms/aks/nodepool/mode.
  - name: --max-surge
    type: string
    short-summary: Extra nodes used to speed upgrade. When specified, it represents the number or percent used, eg. 5 or 33%
  - name: --drain-timeout
    type: int
    short-summary: When nodes are drain how many minutes to wait for all pods to be evicted
  - name: --node-soak-duration
    type: int
    short-summary: The amount of time (in minutes) to wait after draining a node and before reimaging it and moving on to next node.
  - name: --node-taints
    type: string
    short-summary: The node taints for the node pool. You can update the existing node taint of a nodepool or create a new node taint for a nodepool. Pass the empty string `""` to remove all taints.
  - name: --labels
    type: string
    short-summary: The node labels for the node pool. See https://aka.ms/node-labels for syntax of labels.
  - name: --aks-custom-headers
    type: string
    short-summary: Comma-separated key-value pairs to specify custom headers.
  - name: --allowed-host-ports
    type: string
    short-summary: Expose host ports on the node pool. When specified, format should be a space-separated list of ranges with protocol, eg. `80/TCP 443/TCP 4000-5000/TCP`.
  - name: --asg-ids
    type: string
    short-summary: The IDs of the application security groups to which the node pool's network interface should belong. When specified, format should be a space-separated list of IDs.
  - name: --os-sku
    type: string
    short-summary: The os-sku of the agent node pool.
  - name: --enable-fips-image
    type: bool
    short-summary: Switch to use FIPS-enabled OS on agent nodes.
  - name: --disable-fips-image
    type: bool
    short-summary: Switch to use non-FIPS-enabled OS on agent nodes.
  - name: --enable-secure-boot
    type: bool
    short-summary: Enable Secure Boot on an existing Trusted Launch enabled agent node pool. Must use VMSS agent pool type.
  - name: --disable-secure-boot
    type: bool
    short-summary: Disable Secure Boot on on an existing Trusted Launch enabled agent node pool.
  - name: --enable-vtpm
    type: bool
    short-summary: Enable vTPM on an existing Trusted Launch enabled agent node pool. Must use VMSS agent pool type.
  - name: --disable-vtpm
    type: bool
    short-summary: Disable vTPM on an existing Trusted Launch enabled agent node pool.
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the node pool, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new agentpool.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new node pool to be created, but to prevent updating an existing node pool. Other values will be ignored.
examples:
  - name: Reconcile the nodepool back to its current state.
    text: az aks nodepool update -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster
  - name: Enable cluster-autoscaler within node count range [1,5]
    text: az aks nodepool update --enable-cluster-autoscaler --min-count 1 --max-count 5 -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster
  - name: Disable cluster-autoscaler for an existing cluster
    text: az aks nodepool update --disable-cluster-autoscaler -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster
  - name: Update min-count or max-count for cluster autoscaler.
    text: az aks nodepool update --update-cluster-autoscaler --min-count 1 --max-count 10 -g MyResourceGroup -n nodepool1 --cluster-name MyManagedCluster
"""

helps['aks nodepool upgrade'] = """
type: command
short-summary: Upgrade the node pool in a managed Kubernetes cluster.
parameters:
  - name: --kubernetes-version -k
    type: string
    short-summary: Version of Kubernetes to upgrade the node pool to, such as "1.16.9".
  - name: --node-image-only
    type: bool
    short-summary: Only upgrade agent pool's node image.
  - name: --max-surge
    type: string
    short-summary: Extra nodes used to speed upgrade. When specified, it represents the number or percent used, eg. 5 or 33% (mutually exclusive with "--node-image-only". See "az aks nodepool update --max-surge" to update max surge before upgrading with "--node-image-only")
  - name: --drain-timeout
    type: int
    short-summary: When nodes are drain how long to wait for all pods to be evicted
  - name: --node-soak-duration
    type: int
    short-summary: The amount of time (in minutes) to wait after draining a node and before reimaging it and moving on to next node.
  - name: --snapshot-id
    type: string
    short-summary: The source snapshot id used to upgrade this nodepool.
  - name: --aks-custom-headers
    type: string
    short-summary: Comma-separated key-value pairs to specify custom headers.
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the node pool, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new agentpool. For upgrade node image version requests this will be ignored.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new node pool to be created, but to prevent updating an existing node pool. Other values will be ignored.
"""

helps['aks nodepool stop'] = """
    type: command
    short-summary: Stop running agent pool in the managed Kubernetes cluster.
    parameters:
        - name: --aks-custom-headers
          type: string
          short-summary: Send custom headers. When specified, format should be Key1=Value1,Key2=Value2
    examples:
        - name: Stop agent pool in the managed cluster
          text: az aks nodepool stop --nodepool-name nodepool1 -g MyResourceGroup --cluster-name MyManagedCluster
"""

helps['aks nodepool start'] = """
    type: command
    short-summary: Start stopped agent pool in the managed Kubernetes cluster.
    parameters:
        - name: --aks-custom-headers
          type: string
          short-summary: Send custom headers. When specified, format should be Key1=Value1,Key2=Value2
    examples:
        - name: Start agent pool in the managed cluster
          text: az aks nodepool start --nodepool-name nodepool1 -g MyResourceGroup --cluster-name MyManagedCluster
"""

helps['aks nodepool operation-abort'] = """
    type: command
    short-summary: Abort last running operation on nodepool.
    examples:
        - name: Abort operation on agent pool
          text: az aks nodepool operation-abort -g myResourceGroup --nodepool-name nodepool1 --cluster-name myAKSCluster
"""

helps['aks operation-abort'] = """
    type: command
    short-summary: Abort last running operation on managed cluster.
    examples:
        - name: Abort operation on managed cluster
          text: az aks operation-abort -g myResourceGroup -n myAKSCluster
"""

helps['aks nodepool delete-machines'] = """
    type: command
    short-summary: Delete specific machines in an agentpool for a managed cluster.
    parameters:
        - name: --machine-names
          type: string array
          short-summary: Space-separated list of machine names from the agent pool to be deleted.
    examples:
        - name: Delete specific machines in an agent pool
          text: az aks nodepool delete-machines -g myResourceGroup --nodepool-name nodepool1 --cluster-name myAKSCluster --machine-names machine1
"""

helps['aks remove-dev-spaces'] = """
type: command
short-summary: Remove Azure Dev Spaces from a managed Kubernetes cluster.
examples:
  - name: Remove Azure Dev Spaces from a managed Kubernetes cluster.
    text: |-
        az aks remove-dev-spaces -g my-aks-group -n my-aks
  - name: Remove Azure Dev Spaces from a managed Kubernetes cluster without prompting.
    text: |-
        az aks remove-dev-spaces -g my-aks-group -n my-aks --yes
"""

helps['aks scale'] = """
type: command
short-summary: Scale the node pool in a managed Kubernetes cluster.
parameters:
  - name: --node-count -c
    type: int
    short-summary: Number of nodes in the Kubernetes node pool.
examples:
  - name: Scale the node pool in a managed Kubernetes cluster. (autogenerated)
    text: az aks scale --name MyManagedCluster --node-count 3 --resource-group MyResourceGroup
    crafted: true
"""

helps['aks show'] = """
type: command
short-summary: Show the details for a managed Kubernetes cluster.
examples:
  - name: Show the details for a managed Kubernetes cluster
    text: az aks show --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks stop'] = """
    type: command
    short-summary: Stop a managed cluster.
    long-summary: This can only be performed on Azure Virtual Machine Scale set backed clusters. Stopping a
        cluster stops the control plane and agent nodes entirely, while maintaining all object and
        cluster state. A cluster does not accrue charges while it is stopped. See `stopping a
        cluster <https://learn.microsoft.com/azure/aks/start-stop-cluster>`_ for more details about
        stopping a cluster.
"""

helps['aks update-credentials'] = """
type: command
short-summary: Update credentials for a managed Kubernetes cluster, like service principal.
parameters:
  - name: --reset-service-principal
    type: bool
    short-summary: Reset service principal for a managed cluster.
  - name: --service-principal
    type: string
    short-summary: Service principal used for authentication to Azure APIs. This argument is required if `--reset-service-principal` is specified.
  - name: --client-secret
    type: string
    short-summary: Secret associated with the service principal. This argument is required if `--service-principal` is specified.
  - name: --reset-aad
    type: string
    short-summary: Reset Azure Active Directory configuration for a managed cluster.
    long-summary: --reset-aad is deprecated. See https://aka.ms/aks/aad-legacy for details.
  - name: --aad-server-app-id
    type: string
    short-summary: The ID of an Azure Active Directory server application. This argument is required if `--reset-aad` is specified.
    long-summary: --aad-server-app-id is deprecated. See https://aka.ms/aks/aad-legacy for details.
  - name: --aad-server-app-secret
    type: string
    short-summary: The secret of an Azure Active Directory server application. This argument is required if `--reset-aad` is specified.
    long-summary: --aad-server-app-secret is deprecated. See https://aka.ms/aks/aad-legacy for details.
  - name: --aad-client-app-id
    type: string
    short-summary: The ID of an Azure Active Directory client application. This argument is required if `--reset-aad` is specified.
    long-summary: --aad-client-app-id is deprecated. See https://aka.ms/aks/aad-legacy for details.
  - name: --aad-tenant-id
    type: string
    short-summary: Tenant ID associated with Azure Active Directory.

examples:
  - name: Update an existing Kubernetes cluster with new service principal.
    text: az aks update-credentials -g MyResourceGroup -n MyManagedCluster --reset-service-principal --service-principal MyNewServicePrincipalID --client-secret MyNewServicePrincipalSecret
"""

helps['aks upgrade'] = """
type: command
short-summary: Upgrade a managed Kubernetes cluster to a newer version.
long-summary: "Kubernetes will be unavailable during cluster upgrades."
parameters:
  - name: --kubernetes-version -k
    type: string
    short-summary: Version of Kubernetes to upgrade the cluster to, such as "1.16.9".
    populator-commands:
      - "`az aks get-upgrades`"
  - name: --control-plane-only
    type: bool
    short-summary: Upgrade the cluster control plane only. If not specified, both control plane AND all node pools will be upgraded.
  - name: --node-image-only
    type: bool
    short-summary: Only upgrade node image for agent pools.
  - name: --enable-force-upgrade
    type: bool
    short-summary: Enable forceUpgrade cluster upgrade settings override.
  - name: --disable-force-upgrade
    type: bool
    short-summary: Disable forceUpgrade cluster upgrade settings override.
  - name: --upgrade-override-until
    type: string
    short-summary: Until when the cluster upgradeSettings overrides are effective.
    long-summary: It needs to be in a valid date-time format that's within the next 30 days. For example, 2023-04-01T13:00:00Z. Note that if --force-upgrade is set to true and --upgrade-override-until is not set, by default it will be set to 3 days from now.
  - name: --k8s-support-plan
    type: string
    short-summary: Choose from "KubernetesOfficial" or "AKSLongTermSupport", with "AKSLongTermSupport" you get 1 extra year of CVE patchs.
  - name: --tier
    type: string
    short-summary: Specify SKU tier for managed clusters. '--tier standard' enables a standard managed cluster service with a financially backed SLA. '--tier free' does not have a financially backed SLA. '--tier premium' is required for '--k8s-support-plan AKSLongTermSupport'.
  - name: --if-match
    type: string
    short-summary: The value provided will be compared to the ETag of the managed cluster, if it matches the operation will proceed. If it does not match, the request will be rejected to prevent accidental overwrites. This must not be specified when creating a new cluster.
  - name: --if-none-match
    type: string
    short-summary: Set to '*' to allow a new cluster to be created, but to prevent updating an existing cluster. Other values will be ignored.

examples:
  - name: Upgrade a managed Kubernetes cluster to a newer version. (autogenerated)
    text: az aks upgrade --kubernetes-version 1.12.6 --name MyManagedCluster --resource-group MyResourceGroup
    crafted: true
"""

helps['aks use-dev-spaces'] = """
type: command
short-summary: Use Azure Dev Spaces with a managed Kubernetes cluster.
parameters:
  - name: --update
    type: bool
    short-summary: Update to the latest Azure Dev Spaces client components.
  - name: --space -s
    type: string
    short-summary: Name of the new or existing dev space to select. Defaults to an interactive selection experience.
  - name: --endpoint -e
    type: string
    short-summary: The endpoint type to be used for a Azure Dev Spaces controller. See https://aka.ms/azds-networking for more information.
examples:
  - name: Use Azure Dev Spaces with a managed Kubernetes cluster, interactively selecting a dev space.
    text: |-
        az aks use-dev-spaces -g my-aks-group -n my-aks
  - name: Use Azure Dev Spaces with a managed Kubernetes cluster, updating to the latest Azure Dev Spaces \
            client components and selecting a new or existing dev space 'my-space'.
    text: |-
        az aks use-dev-spaces -g my-aks-group -n my-aks --update --space my-space
  - name: Use Azure Dev Spaces with a managed Kubernetes cluster, selecting a new or existing dev space \
            'develop/my-space' without prompting for confirmation.
    text: |-
        az aks use-dev-spaces -g my-aks-group -n my-aks -s develop/my-space -y
  - name: Use Azure Dev Spaces with a managed Kubernetes cluster with a private endpoint.
    text: |-
        az aks use-dev-spaces -g my-aks-group -n my-aks -e private
"""

helps['aks wait'] = """
type: command
short-summary: Wait for a managed Kubernetes cluster to reach a desired state.
long-summary: If an operation on a cluster was interrupted or was started with `--no-wait`, use this command to wait for it to complete.
examples:
  - name: Wait for a cluster to be upgraded, polling every minute for up to thirty minutes.
    text: |-
        az aks wait -g MyResourceGroup -n MyManagedCluster --updated --interval 60 --timeout 1800
  - name: Wait for a managed Kubernetes cluster to reach a desired state (autogenerated)
    text: az aks wait --created --interval 60 --name MyManagedCluster --resource-group MyResourceGroup --timeout 1800
    crafted: true
"""

helps['aks nodepool wait'] = """
type: command
short-summary: Wait for a node pool to reach a desired state.
long-summary: If an operation on a node pool was interrupted or was started with `--no-wait`, use this command to wait for it to complete.
examples:
  - name: Wait for a node pool to reach a desired state, polling every minute for up to thirty minutes.
    text: |-
        az aks nodepool wait --created --interval 60 --cluster-name MyManagedCluster --resource-group MyResourceGroup --nodepool-name MyNodePool --timeout 1800
"""

helps['aks nodepool snapshot wait'] = """
type: command
short-summary: Wait for a nodepool snapshot to reach a desired state.
long-summary: If an operation on a nodepool snapshot was interrupted or was started with `--no-wait`, use this command to wait for it to complete.
examples:
  - name: Wait for a node pool to reach a desired state, polling every minute for up to thirty minutes.
    text: |-
        az aks nodepool snapshot wait --created --interval 60 --name MySnapshot --resource-group MyResourceGroup --timeout 1800
"""

helps['aks rotate-certs'] = """
    type: command
    short-summary: Rotate certificates and keys on a managed Kubernetes cluster
    long-summary: Kubernetes will be unavailable during cluster certificate rotation.
"""

helps['aks command'] = """
    type: group
    short-summary: See detail usage in 'az aks command invoke', 'az aks command result'.
"""

helps['aks command invoke'] = """
    type: command
    short-summary: Run a shell command (with kubectl, helm) on your aks cluster, support attaching files as well.
    parameters:
        - name: --command -c
          type: string
          short-summary: Command or shell script you want to run.
        - name: --file -f
          type: string
          short-summary: Files to be used by the command, use '.' to attach the current folder.
"""

helps['aks command result'] = """
    type: command
    short-summary: Fetch result from previously triggered 'aks command invoke'.
    parameters:
        - name: --command-id -i
          type: string
          short-summary: commandId returned from 'aks command invoke'.
"""

helps['aks snapshot'] = """
    type: group
    short-summary: Commands to manage nodepool snapshots.
"""

helps['aks snapshot show'] = """
    type: command
    short-summary: Show the details of a nodepool snapshot.
"""

helps['aks snapshot list'] = """
    type: command
    short-summary: List nodepool snapshots.
"""

helps['aks snapshot create'] = """
    type: command
    short-summary: Create a nodepool snapshot.
    parameters:
        - name: --nodepool-id
          type: string
          short-summary: The source nodepool id from which to create this snapshot.
        - name: --tags
          type: string
          short-summary: The tags of the snapshot.
        - name: --aks-custom-headers
          type: string
          short-summary: Send custom headers. When specified, format should be Key1=Value1,Key2=Value2
"""

helps['aks snapshot delete'] = """
    type: command
    short-summary: Delete a nodepool snapshot.
"""

helps['aks snapshot wait'] = """
type: command
short-summary: Wait for a nodepool snapshot to reach a desired state.
long-summary: If an operation on a nodepool snapshot was interrupted or was started with `--no-wait`, use this command to wait for it to complete.
"""

helps['aks nodepool snapshot'] = """
    type: group
    short-summary: Commands to manage nodepool snapshots.
"""

helps['aks nodepool snapshot show'] = """
    type: command
    short-summary: Show the details of a nodepool snapshot.
"""

helps['aks nodepool snapshot list'] = """
    type: command
    short-summary: List nodepool snapshots.
"""

helps['aks nodepool snapshot update'] = """
    type: command
    short-summary: Update tags on a snapshot of a nodepool.
    examples:
        - name: Update tags on a nodepool snapshot.
          text: az aks nodepool snapshot update -g MyResourceGroup -n snapshot1 --tags "foo=bar" "key1=val1"
        - name: Clear tags on a nodepool snapshot.
          text: az aks nodepool snapshot update -g MyResourceGroup -n snapshot1 --tags ""
"""

helps['aks nodepool snapshot create'] = """
    type: command
    short-summary: Create a nodepool snapshot.
    parameters:
        - name: --nodepool-id
          type: string
          short-summary: The source nodepool id from which to create this snapshot.
        - name: --tags
          type: string
          short-summary: The tags of the snapshot.
        - name: --aks-custom-headers
          type: string
          short-summary: Send custom headers. When specified, format should be Key1=Value1,Key2=Value2
    examples:
        - name: Create a nodepool snapshot.
          text: az aks nodepool snapshot create -g MyResourceGroup -n snapshot1 --nodepool-id "/subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/managedClusters/akscluster1/agentPools/nodepool1"
        - name: Create a nodepool snapshot with custom tags.
          text: az aks nodepool snapshot create -g MyResourceGroup -n snapshot1 --nodepool-id "/subscriptions/00000/resourceGroups/AnotherResourceGroup/providers/Microsoft.ContainerService/managedClusters/akscluster1/agentPools/nodepool1" --tags "foo=bar" "key1=val1"
"""

helps['aks nodepool snapshot delete'] = """
    type: command
    short-summary: Delete a nodepool snapshot.
"""

helps['aks oidc-issuer'] = """
    type: group
    short-summary: oidc issuer related commands
"""
helps['aks oidc-issuer rotate-signing-keys'] = """
    type: command
    short-summary: Rotate oidc issuer service account signing keys
"""

helps['aks trustedaccess'] = """
    type: group
    short-summary: Commands to manage trusted access security features.
"""

helps['aks trustedaccess role'] = """
    type: group
    short-summary: Commands to manage trusted access roles.
"""

helps['aks trustedaccess role list'] = """
    type: command
    short-summary: List trusted access roles.
"""

helps['aks trustedaccess rolebinding'] = """
    type: group
    short-summary: Commands to manage trusted access role bindings.
"""

helps['aks trustedaccess rolebinding list'] = """
    type: command
    short-summary: List all the trusted access role bindings.
"""

helps['aks trustedaccess rolebinding show'] = """
    type: command
    short-summary: Get the specific trusted access role binding according to binding name.
    parameters:
        - name: --name -n
          type: string
          short-summary: Specify the role binding name.
"""

helps['aks trustedaccess rolebinding create'] = """
    type: command
    short-summary: Create a new trusted access role binding.
    parameters:
        - name: --name -n
          type: string
          short-summary: Specify the role binding name.
        - name: --roles
          type: string
          short-summary: Specify the comma-separated roles.
        - name: --source-resource-id
          type: string
          short-summary: Specify the source resource id of the binding.

    examples:
        - name: Create a new trusted access role binding
          text: az aks trustedaccess rolebinding create -g myResourceGroup --cluster-name myCluster -n bindingName --source-resource-id /subscriptions/0000/resourceGroups/myResourceGroup/providers/Microsoft.Demo/samples --roles Microsoft.Demo/samples/reader,Microsoft.Demo/samples/writer
"""

helps['aks trustedaccess rolebinding update'] = """
    type: command
    short-summary: Update a trusted access role binding.
    parameters:
        - name: --name -n
          type: string
          short-summary: Specify the role binding name.
        - name: --roles
          type: string
          short-summary: Specify the comma-separated roles.
"""

helps['aks trustedaccess rolebinding delete'] = """
    type: command
    short-summary: Delete a trusted access role binding according to name.
    parameters:
        - name: --name -n
          type: string
          short-summary: Specify the role binding name.
"""

helps['aks mesh'] = """
    type: group
    short-summary: Commands to manage Azure Service Mesh.
    long-summary: A group of commands to manage Azure Service Mesh in given cluster.
"""

helps['aks mesh enable'] = """
    type: command
    short-summary: Enable Azure Service Mesh.
    long-summary: This command enables Azure Service Mesh in given cluster.
    parameters:
      - name: --revision
        type: string
        short-summary: Azure Service Mesh revision to install.
      - name: --key-vault-id
        type: string
        short-summary: The Azure Keyvault id with plugin CA info.
      - name: --ca-cert-object-name
        type: string
        short-summary: Intermediate cert object name in the Azure Keyvault.
      - name: --ca-key-object-name
        type: string
        short-summary: Intermediate key object name in the Azure Keyvault.
      - name: --cert-chain-object-name
        type: string
        short-summary: Cert chain object name in the Azure Keyvault.
      - name: --root-cert-object-name
        type: string
        short-summary: Root cert object name in the Azure Keyvault.
    examples:
      - name: Enable Azure Service Mesh with selfsigned CA.
        text: az aks mesh enable --resource-group MyResourceGroup --name MyManagedCluster
      - name: Enable Azure Service Mesh with plugin CA.
        text: az aks mesh enable --resource-group MyResourceGroup --name MyManagedCluster --key-vault-id /subscriptions/00000/resourceGroups/foo/providers/Microsoft.KeyVault/vaults/foo --ca-cert-object-name my-ca-cert --ca-key-object-name my-ca-key --cert-chain-object-name my-cert-chain --root-cert-object-name my-root-cert
"""

helps['aks mesh disable'] = """
    type: command
    short-summary: Disable Azure Service Mesh.
    long-summary: This command disables Azure Service Mesh in given cluster.
"""

helps['aks mesh enable-ingress-gateway'] = """
    type: command
    short-summary: Enable an Azure Service Mesh ingress gateway.
    long-summary: This command enables an Azure Service Mesh ingress gateway in given cluster.
    parameters:
      - name: --ingress-gateway-type
        type: string
        short-summary: Specify the type of ingress gateway.
        long-summary: Allowed values are "External" which is backed by a load balancer with an external IP address; "Internal" which is backed by a load balancer with an internal IP address.
    examples:
      - name: Enable an internal ingress gateway.
        text: az aks mesh enable-ingress-gateway --resource-group MyResourceGroup --name MyManagedCluster --ingress-gateway-type Internal
"""

helps['aks mesh disable-ingress-gateway'] = """
    type: command
    short-summary: Disable an Azure Service Mesh ingress gateway.
    long-summary: This command disables an Azure Service Mesh ingress gateway in given cluster.
    parameters:
      - name: --ingress-gateway-type
        type: string
        short-summary: Specify the type of ingress gateway.
        long-summary: Allowed values are "External" which is backed by a load balancer with an external IP address, "Internal" which is backed by a load balancer with an internal IP address.
    examples:
      - name: Disable an internal ingress gateway.
        text: az aks mesh disable-ingress-gateway --resource-group MyResourceGroup --name MyManagedCluster --ingress-gateway-type Internal
"""

helps['aks mesh get-revisions'] = """
    type: command
    short-summary: Discover available Azure Service Mesh revisions and their compatibility.
    long-summary: This command lists available Azure Service Mesh revisions and their compatibility information for the given location.
    examples:
      - name: Discover Azure Service Mesh revisions.
        text: az aks mesh get-revisions --location westus2
        crafted: true
"""

helps['aks mesh get-upgrades'] = """
    type: command
    short-summary: Discover available Azure Service Mesh upgrades.
    long-summary: This command lists available Azure Service Mesh upgrades for the mesh revision installed on the cluster.
    examples:
      - name: Discover Azure Service Mesh upgrades.
        text: az aks mesh get-upgrades --resource-group MyResourceGroup --name MyManagedCluster
"""

helps['aks mesh upgrade start'] = """
    type: command
    short-summary: Initiate Azure Service Mesh upgrade.
    long-summary: This command initiates upgrade of Azure Service Mesh to the specified revision.
    parameters:
      - name: --revision
        type: string
        short-summary: Azure Service Mesh revision to upgrade to.
    examples:
      - name: Initiate Azure Service Mesh upgrade.
        text: az aks mesh upgrade start --resource-group MyResourceGroup --name MyManagedCluster --revision asm-1-18
"""

helps['aks mesh upgrade'] = """
    type: group
    short-summary: Commands to manage the upgrades for Azure Service Mesh.
    long-summary: A group of commands to manage the upgrades for Azure Service Mesh in given cluster.
"""

helps['aks mesh upgrade complete'] = """
    type: command
    short-summary: Complete Azure Service Mesh upgrade.
    long-summary: This command completes Azure Service Mesh canary upgrade by removing the previous revision.
    examples:
      - name: Complete Azure Service Mesh upgrade.
        text: az aks mesh upgrade complete --resource-group MyResourceGroup --name MyManagedCluster
"""

helps['aks mesh upgrade rollback'] = """
    type: command
    short-summary: Rollback Azure Service Mesh upgrade.
    long-summary: This command rolls back Azure Service Mesh upgrade to the previous stable revision.
    examples:
      - name: Rollback Azure Service Mesh upgrade.
        text: az aks mesh upgrade rollback --resource-group MyResourceGroup --name MyManagedCluster
"""

helps['aks approuting'] = """
    type: group
    short-summary: Commands to manage App Routing addon.
    long-summary: A group of commands to manage App Routing in given cluster.
"""

helps['aks approuting enable'] = """
    type: command
    short-summary: Enable App Routing.
    long-summary: This command enables App Routing in given cluster.
    parameters:
      - name: --enable-kv
        type: bool
        short-summary: Enable the keyvault secrets provider.
        long-summary: This optional flag enables the keyvault-secrets-provider addon in given cluster. This is required for most App Routing use-cases.
      - name: --attach-kv
        type: string
        short-summary: Attach a keyvault id to access secrets and certificates.
        long-summary: This optional flag attaches a keyvault id to access secrets and certificates.
"""

helps['aks approuting disable'] = """
    type: command
    short-summary: Disable App Routing addon.
    long-summary: This command disables App Routing in given cluster.
"""

helps['aks approuting update'] = """
    type: command
    short-summary: Update App Routing addon.
    long-summary: This command is used to update keyvault id in App Routing addon.
    parameters:
      - name: --attach-kv
        type: string
        short-summary: Attach a keyvault id to access secrets and certificates.
        long-summary: This optional flag attaches a keyvault id to access secrets and certificates.
      - name: --enable-kv
        type: bool
        short-summary: Enable the keyvault secrets provider addon.
        long-summary: This optional flag enables the keyvault-secrets-provider addon in given cluster. This is required for most App Routing use-cases.
"""

helps['aks approuting zone'] = """
    type: group
    short-summary: Commands to manage App Routing DNS Zones.
    long-summary: A group of commands to manage App Routing DNS zones in given cluster.
"""

helps['aks approuting zone add'] = """
    type: command
    short-summary: Add DNS Zone(s) to App Routing.
    long-summary: This command adds multiple DNS zone resource IDs to App Routing.
    parameters:
      - name: --ids
        type: string
        short-summary: Comma-separated list of DNS zone resource IDs to add to App Routing.
      - name: --attach-zones
        type: bool
        short-summary: Grant DNS zone Contributor permissions on all zone IDs specified in --ids.
"""

helps['aks approuting zone delete'] = """
    type: command
    short-summary: Delete DNS Zone(s) from App Routing.
    long-summary: This command deletes DNS zone resource IDs from App Routing in given cluster.
    parameters:
      - name: --ids
        type: string
        short-summary: Comma-separated list of DNS zone resource IDs to delete from App Routing.
"""

helps['aks approuting zone update'] = """
    type: command
    short-summary: Replace DNS Zone(s) in App Routing.
    long-summary: This command replaces the DNS zone resource IDs used in App Routing.
    parameters:
      - name: --ids
        type: string
        short-summary: Comma-separated list of DNS zone resource IDs to replace in App Routing.
      - name: --attach-zones
        type: bool
        short-summary: Grant DNS zone Contributor permissions on all zone IDs specified in --ids.
"""

helps['aks approuting zone list'] = """
    type: command
    short-summary: List DNS Zone IDs in App Routing.
    long-summary: This command lists the DNS zone resources used in App Routing.
"""

helps['aks machine'] = """
   type: group
   short-summary: Get information about machines in a nodepool of a managed clusters
"""
helps['aks machine list'] = """
   type: command
   short-summary: Get information about IP Addresses, Hostname for all machines in an agentpool
   parameters:
       - name: --cluster-name
         type: string
         short-summary: Name of the managed cluster
       - name: --nodepool-name
         type: string
         short-summary: Name of the agentpool of a managed cluster
   exmaples:
       - name: Get information about IP Addresses, Hostname for all machines in an agentpool
         text: az aks machine list --cluster-name <clusterName> --nodepool-name <apName>
"""
helps['aks machine show'] = """
   type: command
   short-summary: Show IP Addresses, Hostname for a specific machine in an agentpool for a managedcluster.
   parameters:
       - name: --cluster-name
         type: string
         short-summary: Name of the managed cluster
       - name: --nodepool-name
         type: string
         short-summary: Name of the agentpool of a managed cluster
       - name: --machine-name
         type: string
         short-summary: Name of the machine in the agentpool of a managed cluster
   exmaples:
       - name: Get IP Addresses, Hostname for a specific machine in an agentpool
         text: az aks machine show --cluster-name <clusterName> --nodepool-name <apName> --machine-name <machineName>
"""
