

class WebConsole {

  constructor() {
    this.messageContainer = document.getElementById("out");
    this.renderedMessages = 0;
    this._stickToEnd = true;
    this.scrollStickGapSize = 60;
    document.addEventListener('scroll', this._onScroll.bind(this), false);
    document.addEventListener('mousedown', this._onMouseDown.bind(this), false);
    this.rootGroup = new Group(this.messageContainer, null);
    this.currentGroup = this.rootGroup;
    this.maxRenderedCount = 2000;
  }

  /**
   * @param {number} count
   */
  setMaxRenderedCount(count) {
    this.maxRenderedCount = count;
  }

  /**
   * @return {!WebConsole}
   */
  static instance() {
    if (!WebConsole._instance)
      WebConsole._instance = new WebConsole();
    return WebConsole._instance;
  }

  /**
   * @param {boolean} value
   */
  setStickToEnd(value) {
    this._stickToEnd = value;
    setTimeout(() => {
      this._stickToEnd = value;
      if (value) {
        this.scrollDown();
      }
    },0);
  }

  /**
   * @param {!Event} event
   */
  _onScroll(event) {
    let bottom = document.scrollingElement.scrollHeight - window.innerHeight;
    let curY = document.scrollingElement.scrollTop;
    let stick = bottom - curY < this.scrollStickGapSize;
    if (stick !== this._stickToEnd) {
      this._stickToEnd = stick;
      WebConsole._notifyStickToEndChange(stick);
    }
  }

  /**
   * @param {!Event} event
   */
  _onMouseDown(event) {
    let clickY = document.scrollingElement.scrollTop + event.clientY;
    if (this._stickToEnd && document.scrollingElement.scrollHeight - clickY > this.scrollStickGapSize) {
      this._stickToEnd = false;
      WebConsole._notifyStickToEndChange(this._stickToEnd);
    }
  }

  /**
   * @param {boolean} state
   */
  static _notifyStickToEndChange(state) {
    callJVM("updateStickToEnd", [state]);
  }

  scrollDown() {
    const scrollingElement = this.scrollingElement();
    scrollingElement.scrollTop = scrollingElement.scrollHeight;
  }

  scrollingElement() {
    return document.scrollingElement || document.body;
  }

  increaseLastMessageRepeatCount() {
    if (this.lastBlock.repeatCounter) {
      this.lastBlock.repeatCounter.increase()
    } else {
      this.lastBlock.repeatCounter = new RepeatCounter();
      this.lastBlock.classList.add("repeated-message");
      this.lastBlock.insertAdjacentElement('beforebegin', this.lastBlock.repeatCounter.container);
    }
  }

  /**
   * @param {string} text
   * @param {boolean} caseSensitive
   */
  findText(text, caseSensitive) {
    return findText("#out", text, caseSensitive);
  }

  findNext() {
    findNext();
  }

  findPrev() {
    findPrev();
  }

  softWrap(state) {
    document.getElementById("out").style.whiteSpace = state === true ? "pre-wrap" : "no-wrap";
  }

  clear() {
    let root = this.messageContainer.parentNode;
    this.messageContainer.remove();
    this.messageContainer = createElement("div");
    this.messageContainer.id = "out";
    root.appendChild(this.messageContainer);
    this.lastBlock = undefined;
    this.rootGroup = new Group(this.messageContainer, null);
    this.currentGroup = this.rootGroup;
    this.renderedMessages = 0;
  }

  startTrace() {
    let currentBlock = this.getCurrentBlock();
    currentBlock.classList.add("trace");
    currentBlock.classList.add("collapsed");
    let groupContainer = createElement("div", "group-container");
    let preview = createElement("span", "preview");

    currentBlock.insertBefore(preview, WebConsole.isIcon(currentBlock.firstChild)
                                       ? currentBlock.firstChild.nextSibling
                                       : currentBlock.firstChild);

    while (preview.nextSibling) {
      let nextSibling = preview.nextSibling;
      nextSibling.remove();
      preview.appendChild(nextSibling);
    }

    currentBlock.appendChild(groupContainer);
    currentBlock.onclick = (event) => {
      toggle_visibility(currentBlock)
    };

    this.lastBlock = groupContainer;
    groupContainer._parent = currentBlock;
  }

  static isIcon(element) {
    return element.classList.contains("result-icon");
  }

  endTrace() {
    let groupContainer = this.getCurrentBlock();
    this.lastBlock = groupContainer._parent;
  }

  startGroup(groupName, collapsed) {
    let state = collapsed ? "collapsed" : "expanded";
    let block = this.getCurrentBlock();
    block.classList.remove("message");
    block.classList.add("group");
    block.classList.add(state);

    let title = createElement("span", "preview");
    title.appendChild(document.createTextNode(groupName));
    let groupContainer = createElement("div", "group-container");
    block.appendChild(title);
    block.appendChild(groupContainer);
    let clickHandler = (event) => {
      event.stopPropagation();
      toggle_visibility(block);
    };
    title.onclick = clickHandler;
    block.onclick = (event) => {
      if (event.target !== event.currentTarget) return;
      clickHandler(event);
    };

    this.currentGroup = new Group(groupContainer, this.currentGroup);
  }

  endGroup() {
    if (this.currentGroup.parent) {
      this.currentGroup = this.currentGroup.parent;
    }
  }

  startMessage(type, level, source) {
    this._addMessage(new Message(type, level, source));
  }

  print(token) {
    let currentBlock = this.getCurrentBlock();
    let newNode;
    if (token.type.endsWith("-link")) {
      newNode = createTextNode(token);
      newNode.onclick = (e) => {
        e.stopPropagation();
        callJVM("navigate", [token.id]);
      };

      if (token.type === "message-link") {
        newNode.classList.add(...token.styleClasses);
        currentBlock.insertAdjacentElement("beforebegin", newNode);
        return;
      }
    } else if (token.type === "tree") {
      newNode = new TreeView(token, this).rootElement();
    } else {
      newNode = createTextNode(token);
      if (token.iconURL != null) {
        let icon = WebConsole._createDynamicIcon(token, "node-icon");
        newNode.insertAdjacentElement("afterbegin", icon);
      }
    }

    newNode.classList.add(...token.styleClasses);
    currentBlock.appendChild(newNode);
  }

  /**
   * @returns {Element}
   */
  getCurrentBlock() {
    if (!this.lastBlock) {
      this._addMessage(new Message());
    }
    return this.lastBlock;
  }

  static _createIcon(...styles) {
    return createElement("span", "icon", ...styles);
  }

  static _createDynamicIcon(valueMessage, ...styles) {
    let iconElement = WebConsole._createIcon(...styles)
    iconElement.style.backgroundImage = 'url(' + valueMessage.iconURL + ')';
    return iconElement;
  }

  /**
   * @param {Message} message
   * @private
   */
  _addMessage(message) {
    this.lastBlock = message.content;
    this.currentGroup.add(message);

    // groups counted as one rendered message
    if (this.rootGroup === this.currentGroup) {
      this.renderedMessages++;
    }
    if (this.renderedMessages > this.maxRenderedCount) {
      this.hideMessages();
    }
  }

  hideMessages() {
    if (this.renderedMessages <= this.maxRenderedCount) return;

    const singleRun = this.renderedMessages - this.maxRenderedCount;
    let messagesHolder;
    if (this.messageContainer.firstChild.class instanceof MessagesHolder
        && this.messageContainer.firstChild.class.savedMessages.length < this.maxRenderedCount) {
      messagesHolder = this.messageContainer.firstChild.class;
    } else {
      messagesHolder = new MessagesHolder();
      this.messageContainer.insertBefore(messagesHolder.root, this.messageContainer.firstChild);
    }

    let currentMessage = this.messageContainer.firstChild.nextSibling;
    let next = currentMessage.nextSibling;
    for (let i = 0; i < singleRun && next; i++) {
        currentMessage.remove();
        this.renderedMessages--;
        messagesHolder.savedMessages.push(currentMessage);
        currentMessage = next;
        next = currentMessage.nextSibling;
    }
  }
}

class Message {
  constructor(type, level, source) {
    this.content = createElement("div", "message");
    this.root = createElement("div", "message-wrapper");
    this.root.appendChild(this.content);

    if (level === 'level-error' || level === 'level-warning' || level === 'level-info') {
      this.setIcon(WebConsole._createIcon("result-icon"));
    }
    else if (type === 'EVAL_IN') {
      this.root.classList.add("message-input");
      this.setIcon(WebConsole._createIcon("prompt-in", "result-icon"));
    } else if (type === 'EVAL_OUT') {
      this.root.classList.add("message-result");
      this.setIcon(WebConsole._createIcon("prompt-out", "result-icon"));
    }
    this.root.classList.add(level);
    this.root.classList.add(source);
  }

  setIcon(icon) {
    this.icon = icon;
    this.content.appendChild(icon);
  }

}

class Group {
  /**
   * @param {!Element} container
   * @param {?Group} parent
   */
  constructor(container, parent) {
    this.container = container;
    this.parent = parent;
  }

  /**
   * @param {!Message} message
   */
  add(message) {
    this.container.appendChild(message.root);
    message.root.group = this;
  }

}

class MessagesHolder {
  constructor() {
    this.root = createElement("div", "saved-messages");
    this.root.appendChild(document.createTextNode("Show previous logs"));
    this.root.addEventListener("click", this.expand.bind(this));
    this.savedMessages = [];
    this.root.class = this;

  }

  expand() {
    let parent = this.root.parentElement;
    let anchor = this.root;
    for (let message of this.savedMessages) {
      parent.insertBefore(message, anchor);
    }
    this.savedMessages = [];
    this.root.remove();
  }

}

class RepeatCounter {

  constructor() {
    this.count = 2;
    this.container = createElement("label", "repeat-counter");
    this.container.textContent = this.count;
  }

  increase() {
    this.count++;
    this.container.textContent = this.count;
  }

}
