---- SchemaAges ----
select pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:schema_txid))) as schema_age, pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:sources_txid))) as sources_age
;


---- ServerStartupTime ----
select round(extract(epoch from pg_postmaster_start_time() at time zone 'UTC')) as startup_time;
;


---- DbAge ----
select pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:db_age))) as database_age
;


---- IsSuperUser ----
select usesuper
from pg_catalog.pg_user
where usename = current_user
;


---- ListDatabases ----
select N.oid::bigint as id,
       datname as name,
       D.description,
       pg_catalog.pg_get_userbyid(N.datdba) as "owner"
from pg_catalog.pg_database N
  left join pg_catalog.pg_shdescription D on N.oid = D.objoid
where not datistemplate
order by case when datname = pg_catalog.current_database() then -1::bigint else N.oid::bigint end
;


---- ListSchemas ----
select N.oid::bigint as id,
       N.xmin as state_number,
       nspname as name,
       D.description,
       pg_catalog.pg_get_userbyid(N.nspowner) as "owner"
from pg_catalog.pg_namespace N
  left join pg_catalog.pg_description D on N.oid = D.objoid
#INTROSPECTSYSTEMSCHEMAS: #: where N.nspname not like 'pg_toast%' and N.nspname not like 'pg_temp%' #.
order by case when nspname = current_schema then -1::bigint else N.oid::bigint end
;


---- RetrieveRoles ----
select R.oid::bigint as role_id,
       R.rolname as role_name,
       R.rolsuper is_super,
       R.rolinherit is_inherit,
       R.rolcreaterole can_createrole,
       R.rolcreatedb can_createdb,
       R.rolcanlogin can_login,
       R.rolconnlimit conn_limit,
       R.rolvaliduntil valid_until,
       R.rolconfig config,
       D.description
from pg_catalog.pg_roles R
  left join pg_catalog.pg_shdescription D on D.objoid = R.oid
;


---- StateNumber ----
select pg_catalog.int8in(pg_catalog.xidout(L.transactionid))
from pg_catalog.pg_locks L
where L.transactionid is not null
order by pg_catalog.age(L.transactionid) desc
limit 1
;


---- CurrentXid ----
select pg_catalog.txid_current()::varchar::bigint as current_txid
;


---- CheckSchemaChanges ----
with saved_age as (select greatest(pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))), pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:src_tx)))) as "value")
select pg_catalog.age(C.last_tx) <= (case when SA."value" < 0 then 2147483647 else SA."value" end) as has_changes from
(
  select X.last_tx as last_tx
  from (
    (
    select xmin as last_tx
      from pg_catalog.pg_type
      where typnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select E.xmin as last_tx
      from pg_catalog.pg_exttable E
        join pg_catalog.pg_class C on E.reloid = C.oid
      where C.relnamespace = :schema_id::oid
      order by pg_catalog.age(E.xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_class
      where relnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_proc
      where pronamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select xmin as last_tx
      from pg_catalog.pg_operator
      where oprnamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select A.xmin as last_tx
      from pg_catalog.pg_attribute A join pg_catalog.pg_class K on K.oid = A.attrelid
      where K.relnamespace = :schema_id::oid
      order by pg_catalog.age(A.xmin)
      limit 1
    )
    union all
    #V50:
    (
    select E.xmin as last_tx
      from pg_catalog.pg_enum E join pg_catalog.pg_type T on T.oid = E.enumtypid
      where T.typnamespace = :schema_id::oid
      order by pg_catalog.age(E.xmin)
      limit 1
    )
    union all
    #.
    (
    select xmin as last_tx
      from pg_catalog.pg_constraint
      where connamespace = :schema_id::oid
      order by pg_catalog.age(xmin)
      limit 1
    )
    union all
    (
    select IX.xmin as last_tx
      from pg_catalog.pg_index IX,
           pg_catalog.pg_class IC
      where IC.oid = IX.indrelid
        and IC.relnamespace = :schema_id::oid
        order by pg_catalog.age(IX.xmin)
        limit 1
    )
    union all
    (
    select RU.xmin as last_tx
      from pg_catalog.pg_rewrite RU,
           pg_catalog.pg_class RC
      where RC.oid = RU.ev_class
        and RC.relnamespace = :schema_id::oid
        order by pg_catalog.age(RU.xmin)
        limit 1
    )
    union all
    (
    select TG.xmin as last_tx
      from pg_catalog.pg_trigger TG,
           pg_catalog.pg_class TC
      where TC.oid = TG.tgrelid
        and TC.relnamespace = :schema_id::oid
        order by pg_catalog.age(TG.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_class C on C.oid = D.objoid
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_constraint C on C.oid = D.objoid
      where C.connamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_trigger T on T.oid = D.objoid
        join pg_catalog.pg_class C on C.oid = T.tgrelid
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_rewrite R on R.oid = D.objoid
        join pg_catalog.pg_class C on C.oid = R.ev_class
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_proc P on P.oid = D.objoid
      where P.pronamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_type T on T.oid = D.objoid
      where T.typnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select D.xmin as last_tx
      from pg_catalog.pg_description D
        join pg_catalog.pg_operator O on O.oid = D.objoid
      where O.oprnamespace = :schema_id::oid
        order by pg_catalog.age(D.xmin)
        limit 1
    )
    union all
    (
    select PR.xmin as last_tx
      from pg_catalog.pg_partition_rule PR
        join pg_catalog.pg_partition P on PR.paroid = P.oid
        join pg_catalog.pg_class C on P.parrelid = C.oid
      where C.relnamespace = :schema_id::oid
        order by pg_catalog.age(PR.xmin)
        limit 1
    )
  ) X
  order by pg_catalog.age(X.last_tx)
  limit 1
) C, saved_age SA
;


---- ListExistentSchemaComments ----
select D.objoid id, pg_catalog.array_agg(D.objsubid) sub_ids
from pg_catalog.pg_description D
  join pg_catalog.pg_class C on D.objoid = C.oid
where C.relnamespace = :schema_id::oid and C.relkind != 'c'
group by D.objoid
union all
select T.oid id, pg_catalog.array_agg(D.objsubid)
from pg_catalog.pg_description D
  join pg_catalog.pg_type T on T.oid = D.objoid or T.typrelid = D.objoid
  left join pg_catalog.pg_class C on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid and (C.relkind = 'c' or C.relkind is null)
group by T.oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_constraint C on D.objoid = C.oid
where C.connamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_trigger T on T.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = T.tgrelid
where C.relnamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_rewrite R on R.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = R.ev_class
where C.relnamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_proc P on P.oid = D.objoid
where P.pronamespace = :schema_id::oid
union all
select D.objoid id, array[D.objsubid]
from pg_catalog.pg_description D
  join pg_catalog.pg_operator O on O.oid = D.objoid
where O.oprnamespace = :schema_id::oid
;


---- ListExistentDatabaseComments ----
select D.objoid id
from pg_catalog.pg_description D
where D.classoid in (
  'pg_catalog.pg_extension'::regclass
)
;


---- RetrieveSchemaComments ----
select D.objoid id, case when C.relstorage = 'x' then 'e' else C.relkind end as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_class C on D.objoid = C.oid
where C.relnamespace = :schema_id::oid and C.relkind != 'c'
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select T.oid id, 'T' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_type T on T.oid = D.objoid or T.typrelid = D.objoid
  left join pg_catalog.pg_class C on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid and (C.relkind = 'c' or C.relkind is null)
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select D.objoid id, pg_catalog.translate(C.contype, 'pufc', 'kkxz') as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_constraint C on D.objoid = C.oid
where C.connamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select D.objoid id, 't' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_trigger T on T.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = T.tgrelid
where C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select D.objoid id, 'R' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_rewrite R on R.oid = D.objoid
  join pg_catalog.pg_class C on C.oid = R.ev_class
where C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select D.objoid id, 'F' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_proc P on P.oid = D.objoid
where P.pronamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
union all
select D.objoid id, 'O' as kind, D.objsubid sub_id, D.description
from pg_catalog.pg_description D
  join pg_catalog.pg_operator O on O.oid = D.objoid
where O.oprnamespace = :schema_id::oid
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))


---- RetrieveDatabaseComments ----
select D.objoid id, case
    when 'pg_catalog.pg_extension'::regclass = classoid then 'E'
  end as kind,
  D.objsubid sub_id, D.description
from pg_catalog.pg_description D
where classoid in (
  'pg_catalog.pg_extension'::regclass
)
#INC> and pg_catalog.age(D.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))


---- ListExistentClasses ----
select distinct oid
from pg_catalog.pg_type
where typnamespace = :schema_id::oid
union
select distinct C.oid
from pg_catalog.pg_class C
     left outer join pg_catalog.pg_partition_rule PR on C.oid = PR.parchildrelid
where C.relnamespace = :schema_id::oid
    and PR.oid is null
union
select distinct oid
from pg_catalog.pg_proc
where pronamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_operator
where oprnamespace = :schema_id::oid
union
select distinct oid
from pg_catalog.pg_constraint
where connamespace = :schema_id::oid
union
select distinct IX.indexrelid
from pg_catalog.pg_index IX,
     pg_catalog.pg_class IC
where IC.oid = IX.indrelid
  and IC.relnamespace = :schema_id::oid
union
select distinct RU.oid
from pg_catalog.pg_rewrite RU,
     pg_catalog.pg_class RC
where RC.oid = RU.ev_class
  and RC.relnamespace = :schema_id::oid
  and not RU.rulename = '_RETURN'
union
select distinct TG.oid
from pg_catalog.pg_trigger TG,
     pg_catalog.pg_class TC
where TC.oid = TG.tgrelid
  and TC.relnamespace = :schema_id::oid
union
select distinct PR.oid
from pg_catalog.pg_partition P
  join pg_catalog.pg_class C on P.parrelid = C.oid
  join pg_catalog.pg_partition_rule PR on P.oid = PR.paroid
where C.relnamespace = :schema_id::oid
;


---- ListDatabaseExistentClasses ----
select distinct oid
from pg_catalog.pg_extension
;


---- RetrieveStates ----
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_namespace
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_type
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_class
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_proc
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_operator
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_constraint
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_rewrite
where oid in (OIDS)
union
select distinct oid, pg_catalog.int8in(pg_catalog.xidout(xmin))
from pg_catalog.pg_trigger
where oid in (OIDS)
;


---- RetrieveSequences ----
with T as (
            select oid as sequence_id,
                 relname as sequence_name,
                 pg_catalog.pg_get_userbyid(relowner) as "owner",
                 xmin as sequence_state_number
                 from pg_catalog.pg_class
                   where relnamespace = :schema_id::oid
                   and relkind = 'S'
                   #INC> and pg_catalog.age(xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
          )
select sequence_id,
       sequence_name,
       sequence_state_number,
       "owner"
from T
;


---- RetrieveDataTypes ----
select T.oid as type_id,
       T.xmin as type_state_number,
       T.typname as type_name,
       T.typtype as type_sub_kind,
       T.typrelid as class_id,
       T.typbasetype as base_type_id,
       case when T.typtype in ('c','e') then null
            else pg_catalog.format_type(T.typbasetype, T.typtypmod) end as type_def,
       T.typdefault as default_expression,
       T.typnotnull as mandatory,
       pg_catalog.pg_get_userbyid(T.typowner) as "owner"
from pg_catalog.pg_type T
         left outer join pg_catalog.pg_class C
             on T.typrelid = C.oid
where T.typnamespace = :schema_id::oid
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
  and (T.typtype in ('d','e') or C.relkind = 'c'::"char" or T.typtype = 'b' and T.typelem = 0)
order by 1
;


---- RetrieveDataTypeChecks ----
select T.oid::bigint as type_id,
       C.oid::bigint as constraint_id,
       C.xmin as constraint_state_number,
       C.conname as constraint_name,
       C.consrc as predicate
from pg_catalog.pg_type T,
     pg_catalog.pg_constraint C
where T.typnamespace = :schema_id::oid
  and T.typtype = 'd'::"char"
  and T.oid = C.contypid
  #INC> and (pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) or pg_catalog.age(C.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))))
order by constraint_name
;


---- RetrieveTables ----
select T.relkind as table_kind,
       T.relname as table_name,
       T.oid as table_id,
       T.xmin as table_state_number,
       T.relhasoids as table_with_oids,
       T.relnamespace as tablespace_id,
       T.reloptions as options,
       (select '{' || pg_catalog.array_to_string(pg_catalog.array_agg(inhparent), ',') || '}' from (select inhparent::bigint from pg_catalog.pg_inherits where T.oid = inhrelid order by inhseqno) as bts) as ancestors,
       (select '{' || pg_catalog.array_to_string(pg_catalog.array_agg(inhrelid), ',') || '}' from (select inhrelid::bigint from pg_catalog.pg_inherits where T.oid = inhparent order by inhrelid) as its) as successors,
       pg_catalog.pg_get_userbyid(T.relowner) as "owner"
from pg_catalog.pg_class T
 left outer join pg_catalog.pg_partition_rule PR on T.oid = PR.parchildrelid
where T.relnamespace = :schema_id::oid
  and T.relkind in ('r', 'm', 'v', 'f', 'p')
  and T.relstorage <> 'x'
  and PR.oid is null
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
order by table_kind, table_id
;


---- RetrieveStructAttributes ----
with types      as (select distinct
                           T.oid as type_id,
                           C.oid as class_id
                    from pg_catalog.pg_type T
                    join pg_catalog.pg_class C
                      on T.typrelid = C.oid
                    join pg_catalog.pg_attribute A
                      on C.oid = A.attrelid
                    where T.typnamespace = :schema_id::oid
                      #INC> and pg_catalog.age(A.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
                      and T.typtype = 'c'::"char"
                      and C.relkind = 'c'::"char"),
     attributes as (select A.attrelid as class_id,
                           A.attnum   as att_position,
                           A.attname  as att_name,
                           pg_catalog.format_type(A.atttypid, A.atttypmod) as att_type,
                           A.attisdropped as att_dropped
                    from pg_catalog.pg_attribute A)
select type_id, att_position, att_name, att_type, att_dropped
from types natural join attributes
order by type_id::bigint, att_position
;


---- RetrieveEnumLabels ----
with types as (select distinct T.oid as type_id
               from pg_catalog.pg_type T
               join pg_catalog.pg_enum E
                 on T.oid = E.enumtypid
               where T.typnamespace = :schema_id::oid
                 and T.typtype = 'e'::"char"
               #INC> and pg_catalog.age(E.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
               )
select T.type_id as enum_id,
       pg_catalog.array_agg(E.enumlabel::varchar) as enum_labels
from types T
join pg_catalog.pg_enum E
  on T.type_id = E.enumtypid
group by T.type_id
;


---- RetrieveColumns ----
with T as ( select distinct T.oid as table_id, T.relname as table_name
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = :schema_id::oid
              and T.relkind in ('r', 'm', 'v', 'f', 'p')
              #INC> and (pg_catalog.age(A.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) or pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))))
              and A.attrelid = T.oid )
select T.table_id,
       C.attnum as column_position,
       C.attname as column_name,
       C.xmin as column_state_number,
       C.atttypmod as type_mod,
       C.attndims as dimensions_number,
       pg_catalog.format_type(C.atttypid, C.atttypmod) as type_spec,
       C.atttypid as type_id,
       C.attnotnull as mandatory,
       D.adsrc as column_default_expression,
       not C.attislocal as column_is_inherited,
       C.attisdropped as column_is_dropped
from T
  inner join pg_catalog.pg_attribute C on T.table_id = C.attrelid
  left outer join pg_catalog.pg_attrdef D on (C.attrelid, C.attnum) = (D.adrelid, D.adnum)
where attnum > 0
order by table_id, attnum
;


---- RetrieveDistributionKeys ----
select DP.localoid as table_id,
       DP.attrnums as distribution_keys
from pg_catalog.gp_distribution_policy DP
         inner join pg_catalog.pg_class T on t.oid = DP.localoid
where T.relnamespace = :schema_id::oid
  and T.relkind = 'r'
;


---- RetrieveIndices ----
with tab as ( select relnamespace as schema_id,
                     oid as table_id,
                     relkind as table_kind
              from pg_catalog.pg_class
              where relnamespace = :schema_id::oid
                and relkind in ('r','m','v','p')),
    ind_head as ( select indrelid as table_id,
                         indexrelid as index_id,
                         indisunique as is_unique,
                         indisprimary as is_primary,
                         indkey as column_positions,
                         indclass as column_operators,
                         #V50> indoption as column_options,
                         pg_catalog.pg_get_expr(indpred, indrelid) as condition,
                         case when indexprs is not null
                             then pg_catalog.array_to_string(array(
                                      select case when indkey[k] = 0
                                             then chr(27) || pg_catalog.pg_get_indexdef(indexrelid, k + 1, true)
                                             else pg_catalog.pg_get_indexdef(indexrelid, k + 1, true)
                                             end
                                      from #V50: pg_catalog.generate_subscripts(indkey, 1)
                                           #:  pg_catalog.generate_series(1, pg_catalog.array_upper(indkey, 1)) #.
                                           as k
                                      order by k),
                                  chr(28))
                             else null end as expressions
                  from pg_catalog.pg_index ),
    ind_stor as ( select cls.relnamespace as schema_id,
                         cls.oid as index_id,
                         cls.xmin as state_number,
                         cls.relname as index_name
                         #V50>, am.amcanorder as can_order
                  from pg_catalog.pg_class cls
                    #V50> join pg_catalog.pg_am am on (cls.relam = am.oid)
                  where cls.relkind = 'i'
                    #INC: and pg_catalog.age(cls.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) #.)
select table_id, table_kind,
       index_name, index_id, state_number,
       is_unique, is_primary,
       column_positions, column_operators, #V50: column_options, #.
       condition, expressions #V50:, can_order #.
from tab natural join ind_head natural join ind_stor
;


---- RetrieveConstraints ----
with T as ( select oid as table_id,
                   relkind as table_kind,
                   xmin as table_state_id
            from pg_catalog.pg_class
            where relkind in ('r','v','f','p')
              and relnamespace = :schema_id::oid ),
     C as ( select conrelid as table_id,
                   oid::bigint as con_id,
                   xmin as con_state_id,
                   conname as con_name,
                   contype as con_kind,
                   conkey as con_columns,
                   confrelid as ref_table_id,
                   confkey as ref_columns,
                   condeferrable as is_deferrable,
                   condeferred as is_init_deferred,
                   confupdtype as on_update,
                   confdeltype as on_delete,
                   consrc as con_expression
            from pg_catalog.pg_constraint
            where contype in ('p','u','f','c')
              and connamespace = :schema_id::oid )
select table_id,
       table_kind,
       con_id,
       pg_catalog.int8in(pg_catalog.xidout(con_state_id)) as con_state_id,
       con_name,
       con_kind,
       con_columns,
       ref_table_id,
       is_deferrable,
       is_init_deferred,
       on_update,
       on_delete,
       con_expression,
       ref_columns
from T natural join C
#INC> where pg_catalog.age(table_state_id) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) or pg_catalog.age(con_state_id) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
order by pg_catalog.strpos('pufc', con_kind)
;


---- RetrieveUsedArgumentTypes ----
with schema_procs as (select prorettype, proargtypes, proallargtypes
                      from pg_catalog.pg_proc
                      where pronamespace = :schema_id::oid
                        #INC: and pg_catalog.age(xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) #. ),
     schema_opers as (select oprleft, oprright, oprresult
                      from pg_catalog.pg_operator
                      where oprnamespace = :schema_id::oid
                        #INC: and pg_catalog.age(xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) #. ),
     schema_aggregates as (select A.aggtranstype
                           from pg_catalog.pg_aggregate A
                           join pg_catalog.pg_proc P
                             on A.aggfnoid = P.oid
                           where P.pronamespace = :schema_id::oid
                           #INC: and (pg_catalog.age(A.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) or pg_catalog.age(P.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))) #.),
     schema_arg_types as ( select prorettype as type_id
                           from schema_procs
                           union
                           select distinct unnest(proargtypes) as type_id
                           from schema_procs
                           union
                           select distinct unnest(proallargtypes) as type_id
                           from schema_procs
                           union
                           select oprleft as type_id
                           from schema_opers
                           where oprleft is not null
                           union
                           select oprright as type_id
                           from schema_opers
                           where oprright is not null
                           union
                           select oprresult as type_id
                           from schema_opers
                           where oprresult is not null
                           union
                           select aggtranstype::oid as type_id
                           from schema_aggregates
                           )
select type_id, pg_catalog.format_type(type_id, null) as type_spec
from schema_arg_types
where type_id <> 0
;


---- RetrieveRoutines ----
with languages as (select oid as lang_oid, lanname as lang
                   from pg_catalog.pg_language),
     routines as (select proname as r_name,
                         prolang as lang_oid,
                         oid as r_id,
                         xmin as r_state_number,
                         proargnames as arg_names,
                         proargmodes as arg_modes,
                         proargtypes::int[] as in_arg_types,
                         proallargtypes::int[] as all_arg_types,
                         #SUPER> #V50> pg_catalog.pg_get_expr(proargdefaults, 'pg_catalog.pg_proc'::regclass) as arg_defaults,
                         #V50> provariadic as arg_variadic_id,
                         prorettype as ret_type_id,
                         proretset as ret_set,
                         case when proisagg then 'a' else 'f' end as kind,
                         provolatile as volatile_kind,
                         proisstrict as is_strict,
                         prosecdef as is_security_definer,
                         #V50> proconfig as configuration_parameters,
                         #V50> procost as cost,
                         pg_catalog.pg_get_userbyid(proowner) as "owner"
                         #V50> ,prorows as rows
                  from pg_catalog.pg_proc
                  where pronamespace = :schema_id::oid
                    and not proisagg
                    #INC: and pg_catalog.age(xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) #.)
select *
from routines natural join languages
;


---- RetrieveAggregates ----
with procedures as (
  select P.oid as function_id,
         pg_catalog.quote_ident(N.nspname) || '.' || pg_catalog.quote_ident(P.proname) as function_name,
         P.prorettype as return_type
  from pg_catalog.pg_proc P
  join pg_catalog.pg_namespace N
    on P.pronamespace = N.oid
),
operators as (
  select O.oid as operator_id,
         pg_catalog.quote_ident(N.nspname) || '.' || O.oprname as operator_name
  from pg_catalog.pg_operator O
  join pg_catalog.pg_namespace N
    on O.oprnamespace = N.oid
)
select P.oid as aggregate_id,
       P.xmin as state_number,
       P.proname as aggregate_name,
       P.proargnames as arg_names,
       P.proargmodes as arg_modes,
       P.proargtypes::int[] as in_arg_types,
       P.proallargtypes::int[] as all_arg_types,
       A.aggtransfn::oid as transition_function_id,
       PT.function_name::text as transition_function_name,
       A.aggtranstype as transition_type,
       A.aggfinalfn::oid as final_function_id,
       case when A.aggfinalfn::oid = 0 then null else PF.function_name end as final_function_name,
       A.agginitval as initial_value,
       A.aggsortop as sort_operator_id,
       case when A.aggsortop = 0 then null else O.operator_name end as sort_operator_name,
       pg_catalog.pg_get_userbyid(P.proowner) as "owner"
from pg_catalog.pg_aggregate A
join pg_catalog.pg_proc P
  on A.aggfnoid = P.oid
join procedures PT
  on A.aggtransfn::oid = PT.function_id
left outer join procedures PF
  on A.aggfinalfn::oid = PF.function_id
left outer join operators O
  on A.aggsortop = O.operator_id
where P.pronamespace = :schema_id::oid
#INC> and (pg_catalog.age(A.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))) or pg_catalog.age(P.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx))))
order by P.oid
;


---- RetrieveOperators ----
select O.oid as op_id,
       O.xmin as state_number,
       oprname as op_name,
       oprkind as op_kind,
       oprleft as arg_left_type_id,
       oprright as arg_right_type_id,
       oprresult as arg_result_type_id,
       oprcode as func_main,
       pg_catalog.pg_get_userbyid(O.oprowner) as "owner"
from pg_catalog.pg_operator O
where oprnamespace = :schema_id::oid
  #INC> and pg_catalog.age(xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
;


---- RetrieveRules ----
select R.ev_class as table_id,
       R.oid as rule_id,
       R.xmin as rule_state_number,
       R.rulename as rule_name,
       pg_catalog.translate(ev_type,'1234','SUID') as rule_event_code,
       #V50> R.ev_enabled as rule_fire_mode,
       R.is_instead as rule_is_instead
from pg_catalog.pg_rewrite R
where R.ev_class in (select oid
                     from pg_catalog.pg_class
                     where relnamespace = :schema_id::oid)
  #INC> and pg_catalog.age(R.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
  and R.rulename != '_RETURN'::name
order by R.ev_class::bigint, ev_type
;


---- RetrieveTriggers ----
select T.tgrelid as table_id,
       T.oid as trigger_id,
       T.xmin as trigger_state_number,
       T.tgname as trigger_name,
       T.tgfoid as function_id,
       T.tgtype & 31 as bits,
       T.tgenabled as trigger_fire_mode
from pg_catalog.pg_trigger T
where T.tgrelid in (select oid
                    from pg_catalog.pg_class
                    where relnamespace = :schema_id::oid)
  #INC> and pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
  and not T.tgisconstraint
;


---- RetrieveViewSources ----
select distinct
       T.relkind as view_kind,
       T.oid as view_id,
       pg_catalog.pg_get_viewdef(T.oid) as source_text
from pg_catalog.pg_class T
  join pg_catalog.pg_namespace N on T.relnamespace = N.oid
  left outer join pg_catalog.pg_attribute A on (A.attrelid = T.oid)
where N.oid = :schema_id::oid
  and T.relkind in ('m','v')
  #INCSRC> and (pg_catalog.age(T.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:src_tx))) or pg_catalog.age(A.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:src_tx))))
;


---- RetrieveRuleSources ----
with A as ( select oid as table_id, case when relstorage = 'x' then 'e' else relkind end as table_kind
            from pg_catalog.pg_class
            where relnamespace = :schema_id::oid
              and relkind in ('r', 'm', 'v', 'f', 'p') )
select table_kind,
       table_id,
       R.oid as rule_id,
       pg_catalog.pg_get_ruledef(R.oid) as source_text
from A join pg_catalog.pg_rewrite R
        on A.table_id = R.ev_class
where R.rulename != '_RETURN'::name
  #INC> and pg_catalog.age(R.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:src_tx)))
;


---- RetrieveFunctionSources ----
with system_languages as ( select oid as lang
                           from pg_catalog.pg_language
                           where lanname in ('c','internal') )
select PC.oid as id,
       #V50:
       pg_catalog.pg_get_function_arguments(PC.oid) as arguments_def,
       pg_catalog.pg_get_function_result(PC.oid) as result_def,
       #:
       (select distinct pg_catalog.array_to_string(ARRAY(select (case when arg_names is null then '' else arg_names[k] || ' ' end) || pg_catalog.format_type(T.oid, null)
                                                from pg_catalog.generate_series(1, pg_catalog.array_upper(arg_types, 1)) as k
                                                join pg_catalog.pg_type T ON T.oid = arg_types[k]
                                                order by k), ', ')
          from (select pg_catalog.string_to_array(pg_catalog.textin(pg_catalog.oidvectorout(PGPC.proargtypes)), ' ') arg_types,
                       PGPC.proargnames arg_names
                  from pg_proc PGPC
                  where PGPC.oid = PC.oid) oids_array) as arguments_def,
       (select distinct pg_catalog.format_type(PGT.oid, null)
          from pg_catalog.pg_type PGT
          where PGT.oid = PC.prorettype) as result_def,
      #.
       PC.prosrc as source_text
from pg_catalog.pg_proc PC
where PC.pronamespace = :schema_id::oid
  #INC> and pg_catalog.age(PC.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:src_tx)))
  and not PC.proisagg
  and PC.prolang not in (select lang from system_languages)
  and PC.prosrc is not null
;


---- ListReferencingSchemas ----
select distinct connamespace as schema_id
from pg_catalog.pg_constraint F,
     pg_catalog.pg_class O
where F.contype = 'f'
  and F.confrelid = O.oid
  and O.relnamespace in (SCHEMAIDS)
;


---- RetrieveRelations ----
select 'sequence' as dependent_kind,
       'table' as owner_kind,
       D.objid as dependent_id,
       D.refobjid as owner_id,
       D.refobjsubid as owner_subobject_id
from pg_depend D
  join pg_class C_SEQ on D.objid = C_SEQ.oid
  join pg_class C_TAB on D.refobjid = C_TAB.oid
where C_SEQ.relkind = 'S'
  and C_TAB.relkind = 'r'
  and D.refobjsubid <> 0
  and D.deptype = 'a'
union all
select 'index'       as dependent_kind,
       'key'         as owner_kind,
       D.objid       as dependent_id,
       D.refobjid    as owner_id,
       D.refobjsubid as owner_subobject_id
from pg_catalog.pg_depend D
       join pg_catalog.pg_constraint C on D.refobjid = C.oid
       join pg_catalog.pg_class C_IDX on D.objid = C_IDX.oid
where C_IDX.relkind = 'i'
;


---- RetrieveExtensions ----
select pg_extension.oid as id,
       pg_extension.xmin as state_number,
       extname as name,
       extversion as version,
       extnamespace as schema_id,
       nspname as schema_name,
       array(select target
             from pg_extension_update_paths(extname)
             where source = extversion and path is not null) as available_updates
from pg_catalog.pg_extension
  join pg_namespace on pg_extension.extnamespace = pg_namespace.oid
#INC> where pg_catalog.age(pg_extension.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
;


---- RetrieveExternalTables ----
select E.xmin as state_number,
       E.reloid as id,
       C.relname as name,
       E.writable as writable,
       #V50: E.urilocation #: case when command is null then E.location else null end #. as locations,
       E.command as command,
       #V50: E.execlocation[1] #: case when command is not null then E.location[1] else null end #. as exec_location,
       E.fmttype as "format",
       E.fmtopts as format_options,
       E.encoding as "encoding",
       E.rejectlimit as reject_limit,
       E.rejectlimittype as reject_limit_type
from pg_catalog.pg_exttable E
  join pg_class C on E.reloid = C.oid
  where C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(E.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
;

---- RetrievePartitions ----
select PR.oid as id,
       P.parlevel as level,
       PR.parname as name,
       P.parkind as kind,
       P.parrelid as table_id,
       '{' || pg_catalog.array_to_string(P.paratts, ',') || '}' as columns,
       PR_PARENT.oid as parent_id,
       pg_catalog.pg_get_expr(PR.parlistvalues, 'pg_catalog.pg_partition_rule'::regclass) as values_list,
       pg_catalog.pg_get_expr(PR.parrangestart, 'pg_catalog.pg_partition_rule'::regclass) as start_value,
       pg_catalog.pg_get_expr(PR.parrangeend, 'pg_catalog.pg_partition_rule'::regclass) as end_value,
       pg_catalog.pg_get_expr(PR.parrangeevery, 'pg_catalog.pg_partition_rule'::regclass) as every_value,
       PR.parrangestartincl as start_inclusive,
       PR.parrangeendincl as end_inclusive,
       PR.parisdefault as is_default,
       PR.parruleord as "order"
from pg_catalog.pg_partition P
  join pg_catalog.pg_partition_rule PR on P.oid = PR.paroid
  join pg_class C on P.parrelid = C.oid
  left join pg_catalog.pg_partition_rule PR_PARENT on PR.parparentrule = PR_PARENT.oid
where not P.paristemplate
  and C.relnamespace = :schema_id::oid
#INC> and pg_catalog.age(PR.xmin) <= pg_catalog.age(pg_catalog.xidin(pg_catalog.int8out(:tx)))
order by PR.oid::bigint