/* This file is part of the Chakra project

   Copyright (C) 2012 Lukas Appelhans <boom1992@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEICLIENTPOLKITDOWNLOADER_H
#define AKABEICLIENTPOLKITDOWNLOADER_H

#include <QObject>
#include <QTime>

#include <akabeierror.h>

namespace AkabeiClient {
/**
 * \class PolKitDownloader akabeiclientpolkitdownloader.h "akabeiclientpolkitdownloader.h"
 *
 * \brief This is used to download files from the internet to a specified destination
 *
 * \code
 * PolKitDownloader * downloader = new PolKitDownloader(url, destination, operation, this);
 * connect(downloader, SIGNAL(finished(bool)), SLOT(finished(bool)));
 * connect(downloader, SIGNAL(progressChanged(int)), SLOT(progressChanged(int)));
 * downloader->start();
 * \endcode
 *
 * This class is not thread-safe.
 */
class PolKitDownloader : public QObject
{
Q_OBJECT
public:
    /**
     * Constructs a PolKitDownloader object
     * @param url the url to download the content from
     * @param dest the destination file on disk
     * @param op the operation which invokes this PolKitDownloader (might be 0)
     * @param parent a parent for this class
     */
    PolKitDownloader(const QString &url, const QString &dest, Akabei::Operation * op, bool force = false, QObject * parent = 0);
    virtual ~PolKitDownloader();

    /**
     * Starts the download
     * @param force if true it forces to download the file, otherwise we check the modification time and see if we are already up-to-date
     */
    void start();

    /**
     * @returns the total size of the file to be downloaded
     */
    //qint64 totalBytes() const;
    /**
     * @returns the size of the file which already got downloaded
     */
    //qint64 downloadedBytes() const;
    /**
     * @returns how fast the download is happening (in bytes)
     */
    //ulong speed() const;

    /**
     * @returns the percentage of content which was already downloaded
     */
    //int progress() const;

    /**
     * @returns the estimated time (in seconds)
     */
    //int eta() const;

    /**
     * @returns a list of errors which have occurred (empty if no errors occurred)
     */
    Akabei::Error::List errors() const;

    /**
     * @returns the modification time of the remote file
     */
    //QDateTime modificationTime() const;

Q_SIGNALS:
    /**
     * Gets emitted when the total bytes changed
     * @param totalBytes the new total size
     */
    void totalBytesChanged(qint64 totalBytes);
    /**
     * Gets emitted when the downloaded bytes changed
     * @param downloadedBytes the size which already got downloaded
     */
    void downloadedBytesChanged(qint64 downloadedBytes);
    /**
     * Gets emitted when the download speed changes
     * @param speed the new speed the file is downloading
     */
    void speedChanged(ulong speed);
    /**
     * Gets emitted when the download progress changed
     * @param progress the progress which was made
     */
    void progressChanged(int progress);
    /**
     * Gets emitted when the estimated time changed
     * @param eta the estimated time for the download to finish
     */
    void etaChanged(int eta);
    /**
     * Gets emitted when the download finished
     * @param success true if the download was successful, otherwise false
     */
    void finished(bool success);

private:
    class Private;
    Private * d;

    Q_PRIVATE_SLOT(d, void __k__finished(QString, bool))
    Q_PRIVATE_SLOT(d, void __k__totalBytesChanged(QString, qlonglong))
    Q_PRIVATE_SLOT(d, void __k__downloadedBytesChanged(QString, qlonglong))
    Q_PRIVATE_SLOT(d, void __k__speedChanged(QString, quint64))
    Q_PRIVATE_SLOT(d, void __k__progressChanged(QString, int))
    Q_PRIVATE_SLOT(d, void __k__etaChanged(QString, int))
    Q_PRIVATE_SLOT(d, void __k__errorOccurred(QString, int, QString))
};
}

#endif // AKABEICLIENTPOLKITDOWNLOADER_H
