/* This file is part of the Chakra project
 * It contains methods to return the most used queries, with placeholders for the parameters.

   Copyright (C) 2012 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEIQUERY_H
#define AKABEIQUERY_H

#include <QString>
#include <akabeicore_global.h>

namespace Akabei
{
namespace Queries
{

/**
 * Query to select groups.
 * 
 * @param property the property to compare groups to.
 * @param comparison the comparison method (LIKE, exact comparison...)
 * @param value the property's value.
 */
QString AKABEICORESHARED_EXPORT selectGroups(const QString &, const QString &, const QString &);

/**
 * Query to select groups with more freedom.
 * 
 * @param token a pattern for the name and/or description.
 * @param comparison the comparison method (LIKE, exact comparison...)
 */
QString AKABEICORESHARED_EXPORT selectGroupsNameOrDescription(const QString &, const QString &);

/**
 * Query to select packages.
 * 
 * @param property the property to compare packages to.
 * @param comparison the comparison method (LIKE, exact comparison...)
 * @param value the property's value.
 */
QString AKABEICORESHARED_EXPORT selectPackages(const QString &property, const QString &comparison, const QString &value);

/**
 * Query to select packages with more freedom.
 * 
 * @param token a pattern for the name and/or description.
 * @param comparison the comparison method (LIKE, exact comparison...)
 */
QString AKABEICORESHARED_EXPORT selectPackagesNameOrDescription(const QString &, const QString &);

/**
 * Query to select hooks.
 * 
 * @param property the property to compare hooks to.
 * @param comparison the comparison method (LIKE, exact comparison...)
 * @param value the property's value.
 */
QString AKABEICORESHARED_EXPORT selectHooks(const QString &, const QString &, const QString &);

/**
 * Query to select deltas.
 * 
 * @param property the property to compare deltas to.
 * @param comparison the comparison method (LIKE, exact comparison...)
 * @param value the property's value.
 */
QString AKABEICORESHARED_EXPORT selectDeltas(const QString &, const QString &, const QString &);

/**
 * Query that returns all packages.
 */
QString AKABEICORESHARED_EXPORT allPackages();

/**
 * Query that returns all groups.
 */
QString AKABEICORESHARED_EXPORT allGroups();

/**
 * Query for a package property among those expressed with tables (conflicts, dependencies, provides...).
 * 
 * @param property the property returned.
 * @param table the table from which the property is taken.
 * @param packageId the package ID to distinguish its properties.
 */
QString AKABEICORESHARED_EXPORT packageProperty(const QString &, const QString &, int);

/**
 * Query for the package hooks.
 * 
 * @param packageId the package ID.
 */
QString AKABEICORESHARED_EXPORT packageHooks(int);

/**
 * Query for all the packages with a certain hook.
 * 
 * @param hook the hook's name.
 */
QString AKABEICORESHARED_EXPORT packagesFromHook(const QString &);

/**
 * Query that selects some packages in a group.
 * 
 * @param group the group's name.
 * @param token pattern for package's name and description.
 */
QString AKABEICORESHARED_EXPORT packagesInGroup(const QString &, const QString &);

/**
 * Query that shows us if a package belongs to some group.
 * 
 * @param group the group's name.
 * @param package the package's ID.
 */
QString AKABEICORESHARED_EXPORT belongsGroup(const QString &, const QString &);

/**
 * Query that returns all packages in a group.
 * 
 * @param group the group's name.
 */
QString AKABEICORESHARED_EXPORT allPackagesInGroup(const QString &);

/**
 * Query for all the packages with a given dependency.
 * 
 * @param dependency the dependency as a regular expression.
 */
QString AKABEICORESHARED_EXPORT requiredByPackages(const QString &);

/**
 * Query for some package's dependencies.
 * 
 * @param dependency a regexp's filter for the dependencies.
 * @param comparison the comparison method (LIKE, exact comparison...)
 */
QString AKABEICORESHARED_EXPORT packageDependencies(const QString &, const QString &);

/**
 * Query all packages with at least one replacement.
 */
QString AKABEICORESHARED_EXPORT globalReplaces();

/**
 * Query that begins a transaction.
 */
QString AKABEICORESHARED_EXPORT begin();

/**
 * Query that commits a transaction.
 */
QString AKABEICORESHARED_EXPORT commit();

/**
 * Query that vacuums the database.
 */
QString AKABEICORESHARED_EXPORT vacuum();

/**
 * Query that rollbacks a transaction.
 */
QString AKABEICORESHARED_EXPORT rollback();
}
}

#endif
