/* This file is part of the Chakra project

   Copyright (C) 2011 Lukas Appelhans

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEIHOOK_H
#define AKABEIHOOK_H

#include <QObject>

namespace Akabei {
class HookPrivate;
class Database;
class Package;
/**
 * \class Hook  akabeihook.h "akabeihook.h"
 *
 * \brief This class describes a script which possibly gets executed once in a transaction.
 *
 * A Hook is basically a scriptlet, but on a global scale.
 * Even though multiple packages can request a hook to be executed,
 * it will only be done once at the end of the transaction.
 *
 * For example, we have a bunch of packages wanting to run kbuildsycoca4 to update kde's plugin cache.
 * Since this command is quite expensive it took a lot of time in pacman days to install multiple of
 * those packages as every package was executing the command on itself.
 * Now we just have a hook running kbuildsycoca4 inside the database and all those packages use it. It
 * then gets executed only once at the end of the transaction.
 *
 * This class is thread-safe.
 */
class Hook
{
    Q_DISABLE_COPY(Hook)
    Q_DECLARE_PRIVATE(Hook)
    public:
        virtual ~Hook();

        /**
         * @returns the name of the hook
         */
        QString name() const;

        /**
         * @returns the script which gets executed when running the hook
         */
        QString content() const;

        /**
         * @returns a list of packages which request this hook on transaction
         */
        QList<Akabei::Package*> packages() const;

        /**
         * @returns a path to a file containing the content()
         * A developer can now easily run the script with a \c QProcess object
         */
        QString path();

    private:
        Hook(const QString &name, const QString &content, QList<Akabei::Package*> pkgs);

        HookPrivate * const d_ptr;

        friend class Backend;
        friend class Database;
        friend class DatabasePrivate;
};
}

#endif // AKABEIHOOK_H
