/* This file is part of the Chakra project

   Copyright (C) 2010 Dario Freddi <drf@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEIDATABASE_H
#define AKABEIDATABASE_H

#include <akabeicore_global.h>
#include <akabeipackage.h>
#include <akabeierror.h>

#include <QObject>

namespace Akabei {

class Group;
class Package;
class Delta;
class Hook;

class DatabasePrivate;
/**
 * \class Database  akabeidatabase.h "akabeidatabase.h"
 *
 * \brief This class is used for reading from an SQLite-Database on disk
 *
 * Databases are usually inside the DatabaseDir and are loaded via our
 * Backend class. One can specify which databases to load through the
 * Config::setDatabases() method.
 */
class AKABEICORESHARED_EXPORT Database
{
    Q_DISABLE_COPY(Database)
    Q_DECLARE_PRIVATE(Database)
public:
    Database(QString const& pathToDatabase);
    virtual ~Database();

    /**
     * Reinitializes the database
     */
    void reinit();

    /**
     * @returns if the database is ready to read from disk
     */
    bool isValid() const;
    
    Error error() const;

    /**
     * @returns the path to the database on disk
     */
    QString name() const;

    /**
     * @returns all packages of the database
     * This might not be actual package-tarballs on disk, but just the package information
     * which can be read from the database
     */
    Akabei::Package::List packages();
    /**
     * Searches the database for a specific set of packages.
     * @returns a list of packages which match the given @param token
     * @param type describes what properties are taken into account
     */
    Akabei::Package::List searchPackages(QString const& token, SearchType type = SearchNameAndDescriptionLike);
    /**
     * Searches the database for a specific set of packages.
     * @returns a list of packages which match the given @param token
     * @param type describes what properties are taken into account
     */
    Akabei::Package::List searchPackages(QStringList const& packages, SearchType type = SearchNameAndDescriptionLike);

    /**
     * @returns all groups of the database
     */
    QList<Group*> groups();
    /**
     * Searches the database for a specific set of groups.
     * @returns a list of groups which match the given @param token
     */
    QList<Group*> searchGroups(QString const& token);

    /**
     * Queries the database for a set of packages.
     * @param sql This has to be a real SQL query, e.g. "SELECT * FROM packages".
     * @returns a list of packages which match the query
     */
    Akabei::Package::List queryPackages(QString const& sql);
    /**
     * Queries the database for a set of groups.
     * @param sql This has to be a real SQL query, e.g. "SELECT * FROM groups".
     */
    QList<Group*> queryGroups(QString const& sql);
    /**
     * Queries the database for a set of deltas.
     * @param sql This has to be a real SQL query, e.g. "SELECT * FROM deltas".
     */
    QList<Delta*> queryDeltas(QString const& sql);
    /**
     * Queries the database for a set of hooks.
     * @param sql This has to be a real SQL query, e.g. "SELECT * FROM hooks".
     */
    QList<Hook*> queryHooks(QString const& sql);

private:
    DatabasePrivate* const d_ptr;

    friend class QueryHelper;
    friend class Package;
};
}

#endif // AKABEIDATABASE_H
