/* This file is part of the Chakra project

   Copyright (C) 2010 Dario Freddi <drf@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef AKABEI_AKABEICONFIG_H
#define AKABEI_AKABEICONFIG_H

#include <akabeicore_global.h>
#include <akabeicache.h>

class QString;
class QStringList;
class QDir;
namespace Akabei {
/**
 * \class Config  akabeiconfig.h "akabeiconfig.h"
 *
 * \brief This class just holds certain config values which akabeicore needs to be working.
 *
 * This class is thread-safe.
 */
class AKABEICORESHARED_EXPORT Config
{
    public:
        /**
         * @returns the config singleton
         */
        static Config *instance();
        ~Config();

        /**
         * @returns the root path which akabeicore operates in.
         * In most cases this is going to be "/".
         */
        QString root() const;
        /**
         * @returns the \c QDir object of the root path.
         * This is useful for operations like absoluteFilePath().
         */
        QDir rootDir() const;
        /**
         * Sets a new root path for akabeicore to use.
         * @param root the new root path
         */
        void setRoot(const QString &root);

        /**
         * @returns the dir where all databases are placed in.
         */
        QString databasePath() const;
        /**
         * @returns the \c QDir object of the database path.
         * This is useful for operations like absoluteFilePath().
         */
        QDir databaseDir() const;
        /**
         * Sets a new database path for akabeicore to use.
         * @param path the new database path
         */
        void setDatabasePath(const QString &path);

        /**
         * @returns the dir where all packages/deltas get cached
         */
        QString cachePath() const;
        /**
         * @returns the \c QDir object of the cache path.
         * This is useful for operations like absoluteFilePath().
         */
        QDir cacheDir() const;
        /**
         * Sets a new cache path for akabeicore to use.
         * @param path the new cache path
         */
        void setCachePath(const QString &path);

        /**
         * @returns the databases which Backend is trying to parse
         */
        QStringList databases() const;
        /**
         * Sets the names of the databases the Backend class is trying to parse when initializing.
         * @param dbs a stringlist of database names which are going to be opened by akabeicore
         */
        void setDatabases(const QStringList &dbs);

        /**
         * @returns the list of files not to upgrade
         */
        QStringList noUpgrade() const;
        /**
         * Sets the files not to upgrade
         */
        void setNoUpgrade(const QStringList &files);

        /**
         * @returns the list of files not to extract
         */
        QStringList noExtract() const;
        /**
         * Sets the files not to extract
         * For directories, applies to all the subtree
         */
        void setNoExtract(const QStringList& files);

        /**
         * @returns the list of packages always to be mantained in our cache
         */
        QStringList keepPackages() const;
        /**
         * Sets the packages to be mantained in cache
         */
        void setKeepPackages(const QStringList& pkgs);

        /**
         * @returns the max cache size in MB
         */
        int maxCacheSize() const;
        /**
         * Sets the maximum size allowed for the cache
         */
        void setMaxCacheSize(int size);

        /**
         * @returns the policy to be applied when the maximum cache size is reached
         */
        Cache::MaxSizePolicy cachePolicy() const;
        /**
         * Sets the policy to be applied to the cache
         */
        void setCachePolicy(Cache::MaxSizePolicy policy);

        /**
         * @returns whether to display debugging prints
         */
        bool debug() const;

        /**
         * Sets whether to display debugging prints
         */
        void setDebug(bool debug);
        
        /**
         * @returns if akabei needs to call polkit and its helper app to e.g. extract an archive
         * This will return false if the root dir is owned by the user executing akabei
         */
        bool needsPrivileges() const;
        
        /**
         * Sets if akabei should use polkit and its helper app e.g. to extract an archive.
         * @param privs true if akabei should use polkit
         */
        void setNeedsPrivileges(bool privs);
        
        /**
         * @returns the log file path
         */
        QString logFile() const;
        
        /**
         * Sets the path of the logging file
         * 
         * @param file the log file path
         */
        void setLogFile(const QString &);
        
        /**
         * @returns whether we log through syslog
         */
        bool useSyslog() const;
        
        /**
         * Sets whether akabei will log through syslog.
         * 
         * @param useSyslog true for logging through syslog, false otherwise.
         */
        void setUseSyslog(bool);

    private:
        Config();

        class Private;
        Private * const d;
};

}

#endif // AKABEI_AKABEICONFIG_H
