/*
 * This file is part of the Chakra project
 * Header file for the cache manager

   Copyright (C) 2010 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/
#ifndef AKABEICACHE_H
#define AKABEICACHE_H

#include <QStringList>
#include <QHash>
#include <QDir>
#include <QDateTime>
#include <akabeidebug.h>

#include <akabeibackend.h>

namespace Akabei
{
/**
 * \class Cache akabeicache.h "akabeicache.h"
 *
 * \brief This class represents the package cache.
 *
 * This class is used to manage the cache. It takes care of cleaning unused/old packages from it
 * when configured so and can be used to get information if packages are cached.
 *
 * This class is not thread-safe.
 */
class Cache
{
public:
    /**
    * This enum describes what to do when the cache reaches the maximum size allowed by the user
    */
    enum MaxSizePolicy
    {
        CleanAll,
        CleanOldAndUnused,
        CleanOld,
        CleanUnused,
        CleanNotInstalled
    };

    /**
     * Constructs a cache manager and executes the requested operations.
     *
     * @param cachedir The cache directory.
     * @param keepPackages A list of packages the cache must always maintain.
     * @param operations A list of operations to perform.
     */
    Cache(const QString &cachedir, const QString &databasedir, const QStringList &keepPackages);

    /**
     * Constructs a cache manager.
     */
    Cache();

    ~Cache();

    /**
     * Checks whether a file is in the cache.
     *
     * @param filename filename.
     * @returns true if file is present, false otherwise.
     */
    bool isPackageInCache(const QString &filename) const;

    /**
     * Find a file in the cache.
     *
     * @param filename filename.
     * @returns the file path even if the file does not exist
     */
    QString getPathInCache(const QString &filename) const;

    /**
     * Find package files in cache. Useful when you don't care about the version.
     *
     * @param pkg the name of the package to find.
     * @returns a list of files corresponding to the available versions of the desired package.
     */
    QStringList getPackagesByName(const QString &pkg) const;

    /**
     * Copy a new file in cache.
     *
     * @param filename the file path.
     * @param newName the new filename in cache.
     * @returns true if successful, false otherwise.
     */
    bool writePackage(const QString &path, const QString &newName);

    /**
     * Cleans the cache with the given policy
     * @see errorString
     */
    void cleanCache(MaxSizePolicy policy);

    /**
     * Error message.
     *
     * @returns a description if some error occurred, an empty string otherwise.
     */
    QString errorString() const;

private:
    class Private;
    Private *d;
};

}

#endif