/*
 * <SQLite3 C++ interface: header file>

   Copyright (C) 2011 Lisa Vitolo <shainer@chakra-project.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/
#ifndef _SQLITE_CONNECTION_H
#define _SQLITE_CONNECTION_H

#include "AbstractSQLiteConnection.h"

/*
 * If there's a God of programming somewhere,
 * please forgive these sinners, for they have lost the way
 */
#include <sqlite3.h>

/*
 * Instead, these are rather good people indeed
 */
#include <QString>
#include <QMap>
#include <QList>
#include <QVariant>
#include <QByteArray>

/**
 * @class SQLiteConnection
 * @brief manages a connection with a database using the sqlite C library
 *
 * This class is not thread-safe.
 */
class SQLiteConnection : public AbstractSQLiteConnection
{
    typedef QVariant Value;
    typedef QVariantMap Row;

public:
    /**
     * Constructs an object and establishes the connection
     *
     * @param name the database filename
     * @param readonly true if read-only access, false for read-write
     */
    SQLiteConnection(QString const& name, bool readonly);

    /**
     * Constructs an object without establishing any connection
     */
    SQLiteConnection();

    ~SQLiteConnection();

    /**
     * Establishes a new connection with a database
     *
     * @param name the database filename
     * @param readonly true if read-only access, false for read-write
     */
    virtual void connectToDB(QString const&, bool) override;

    /**
     * Executes a query on the database
     *
     * @param q the query string
     * @returns a SQLiteResource object with the result, if present
     */
    virtual SQLiteResource query(const QString &q) override;

    /**
     * Executes multiple queries on the database
     *
     * @param q the query
     * @param bindings the bindings we're going to use
     *                 this is pair of values, the key is a string determining the binding name, the value is a stringlist which contains all the values
     */
    virtual void query(const QString &q, const QVariantMap &bindings) override;

    /**
     * Executes a query on the database
     *
     * @param q the query string
     * @returns a QList<Row> object with the result, if present
     */
    QList<Row> rowQuery(const QString &q);

    /**
     * If the table interested by the last query as an autoincremented primary key,
     * gets the last value assigned to the key
     */
    virtual qint64 getLastRowId() override;

    /**
     * Creates a "macro" (fake name) that maps into an integer, a QString, or a QByteArray
     * The next query can contain the defined macros in place of the actual values.
     * @note All bindings stay valid only for the subsequent call of query().
     */
    virtual void bind(QString const&, int) override;
    virtual void bind(QString const&, QString const&) override;
    virtual void bind(QString const&, QByteArray const&) override;

private:
    // Disable copy constructor and default constructor
    SQLiteConnection(SQLiteConnection const&);

    class Private;
    Private *d;
};

#endif
