/*
** Copyright (C) 2001-2025 Carnegie Mellon University
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.24
**
** Copyright 2025 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM25-0915
**
** @OPENSOURCE_LICENSE_END@
*/
#ifndef _RWCOUNT_H
#define _RWCOUNT_H
#ifdef __cplusplus
extern "C" {
#endif

#include <silk/silk.h>

RCSIDENTVAR(rcsID_RWCOUNT_H, "$SiLK: rwcount.h c7d6bb438741 2025-01-17 20:52:04Z mthomas $");

#include <silk/rwrec.h>
#include <silk/sksite.h>
#include <silk/skstream.h>
#include <silk/utils.h>

/*
 *  rwcount.h
 *
 *    Header file for the rwcount utility.
 *
 *
 *    rwcount's minimum precision is 1 millisecond.  Due to the large byte
 *    counts sometimes encountered in rwcount, using a smaller precision opens
 *    the potential that the bytes/time ratio extends outside the precision of
 *    a double (precision is about 52 bits).
 *
 *    To change the precision used in rwcount, globally replace 'Milli' with
 *    either 'Micro' or 'Nano', change SKTIMESTAMP_MILLI appropriately, and
 *    adjust the precision used in in appUsageLong().
 *
 *    To enable easy changes of the precision, the code uses complex looking
 *    constructs these instead of saying 1000.
 *
 *    skIntervalGetMilli(skIntervalSetFromSeconds(1))
 *
 *    The code also references "fractional seconds" instead of using the more
 *    specific milliseconds.
 */


/* DEFINES AND TYPEDEFS */

/* minimum bin size */
#define BINSIZE_MIN       skIntervalSetFromMilli(0, 1)

/* default size of bins */
#define BINSIZE_DEFAULT   skIntervalSetFromSeconds(30)

/* bin loading schemata */
typedef enum {
    LOAD_MEAN=0, LOAD_START, LOAD_END, LOAD_MIDDLE,
    LOAD_DURATION, LOAD_MAXIMUM, LOAD_MINIMUM
} bin_load_scheme_enum_t;

#define MAX_LOAD_SCHEME LOAD_MINIMUM

#define DEFAULT_LOAD_SCHEME LOAD_DURATION


/* Values to use for the start_time and end_time to denote that they
 * are not set */
#define RWCO_UNINIT_START 0
#define RWCO_UNINIT_END   INT64_MAX


/* counting data structure */
typedef struct count_bin_st {
    double bytes;
    double pkts;
    double flows;
} count_bin_t;

typedef struct count_data_st {
    /* size of each bin, in fractional seconds */
    uint64_t    size;
    /* total number of bins that are allocated */
    uint64_t    count;
    /* time on the first bin, in UNIX epoch fractional seconds */
    uint64_t    window_min;
    /* one fractional second after the final bin, in UNIX epoch fractional
     * seconds */
    uint64_t    window_max;
    /* subset of the window representing the range of dates to print, in UNIX
     * epoch fractional seconds where
     *
     * window_min <= start_time < end_time <= window_max */
    uint64_t    start_time;
    uint64_t    end_time;

    /* the data */
    count_bin_t *data;
} count_data_t;


typedef struct count_flags_st {
    /* how to label timestamps */
    uint32_t    timeflags;

    /* bin loading scheme */
    bin_load_scheme_enum_t  load_scheme;

    /* delimiter between columns */
    char        delimiter;

    /* when non-zero, print row label with bin's index value */
    unsigned    label_index         :1;

    /* when non-zero, do not print column titles */
    unsigned    no_titles           :1;

    /* when non-zero, suppress the final delimiter */
    unsigned    no_final_delimiter  :1;

    /* when non-zero, do not print bins with zero counts */
    unsigned    skip_zeroes         :1;

    /* when non-zero, do not print column titles */
    unsigned    no_columns          :1;
} count_flags_t;


/* FUNCTIONS */

void
appSetup(
    int                 argc,
    char              **argv);
void
appTeardown(
    void);
FILE *
getOutputHandle(
    void);


/* VARIABLES */

extern sk_options_ctx_t *optctx;

/* the data */
extern count_data_t bins;

/* flags */
extern count_flags_t flags;

#ifdef __cplusplus
}
#endif
#endif /* _RWCOUNT_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
