/*
** Copyright (C) 2008-2025 Carnegie Mellon University
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.24
**
** Copyright 2025 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM25-0915
**
** @OPENSOURCE_LICENSE_END@
*/

/*
**  cutmatch.c
**
**    A plug-in to be loaded by rwcut to display the values that
**    rwmatch encodes in the NextHopIP field.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: cutmatch.c c7d6bb438741 2025-01-17 20:52:04Z mthomas $");

#include <silk/skplugin.h>
#include <silk/rwrec.h>


/* DEFINES AND TYPEDEFS */

#define FIELD_WIDTH 10

/* Plugin protocol version */
#define PLUGIN_API_VERSION_MAJOR 1
#define PLUGIN_API_VERSION_MINOR 0


/* FUNCTION DECLARATIONS */

static skplugin_err_t
recToText(
    const rwRec            *rwrec,
    char                   *text_value,
    size_t                  text_size,
    void            UNUSED(*idx),
    void           UNUSED(**extra));


/* FUNCTION DEFINITIONS */

/* the registration function called by skplugin.c */
skplugin_err_t
SKPLUGIN_SETUP_FN(
    uint16_t            major_version,
    uint16_t            minor_version,
    void        UNUSED(*pi_data))
{
    skplugin_field_t *field = NULL;
    skplugin_err_t rv;
    skplugin_callbacks_t regdata;

    /* Check API version */
    rv = skpinSimpleCheckVersion(major_version, minor_version,
                                 PLUGIN_API_VERSION_MAJOR,
                                 PLUGIN_API_VERSION_MINOR,
                                 skAppPrintErr);
    if (rv != SKPLUGIN_OK) {
        return rv;
    }

    /* register the field to use for rwcut */
    memset(&regdata, 0, sizeof(regdata));
    regdata.column_width = FIELD_WIDTH;
    regdata.rec_to_text  = recToText;
    rv = skpinRegField(&field, "match", NULL, &regdata, NULL);
    if (SKPLUGIN_OK != rv) {
        return rv;
    }
    if (field) {
        rv = skpinSetFieldTitle(field, "<->Match#");
    }

    return rv;
}


/*
 *  status = recToText(rwrec, text_val, text_len, &index);
 *
 *    Given the SiLK Flow record 'rwrec', fill 'text_val', a buffer of
 *    'text_len' characters, with the appropriate direction and match
 *    number.
 */
static skplugin_err_t
recToText(
    const rwRec            *rwrec,
    char                   *text_value,
    size_t                  text_size,
    void            UNUSED(*idx),
    void           UNUSED(**extra))
{
    static const char *match_dir[] = {"->", "<-"};
    uint32_t nhip;
    uint32_t match_count;

    nhip = rwRecGetNhIPv4(rwrec);
    match_count = 0x00FFFFFF & nhip;
    if (match_count) {
        snprintf(text_value, text_size, ("%s%8" PRIu32),
                 match_dir[(nhip & 0xFF000000) ? 1 : 0],
                 match_count);
    } else {
        snprintf(text_value, text_size, "%-10s",
                 match_dir[(nhip & 0xFF000000) ? 1 : 0]);
    }

    return SKPLUGIN_OK;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
