/*
 * Decompiled with CFR 0.152.
 */
package io.github.dsheirer.source.tuner.usb;

import io.github.dsheirer.buffer.INativeBuffer;
import io.github.dsheirer.buffer.INativeBufferFactory;
import io.github.dsheirer.sample.Listener;
import io.github.dsheirer.source.SourceException;
import io.github.dsheirer.source.tuner.ITunerErrorListener;
import io.github.dsheirer.source.tuner.TunerController;
import io.github.dsheirer.source.tuner.TunerType;
import io.github.dsheirer.source.tuner.manager.TunerManager;
import io.github.dsheirer.util.ThreadPool;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.usb4java.Context;
import org.usb4java.Device;
import org.usb4java.DeviceDescriptor;
import org.usb4java.DeviceHandle;
import org.usb4java.DeviceList;
import org.usb4java.LibUsb;
import org.usb4java.Transfer;
import org.usb4java.TransferCallback;

public abstract class USBTunerController
extends TunerController {
    private Logger mLog = LoggerFactory.getLogger(USBTunerController.class);
    private static final int USB_INTERFACE = 0;
    private static final int USB_CONFIGURATION = 1;
    private static final int USB_BULK_TRANSFER_BUFFER_POOL_SIZE = 8;
    protected static final byte USB_BULK_TRANSFER_ENDPOINT = -127;
    private static final long USB_BULK_TRANSFER_TIMEOUT_MS = 2000L;
    protected int mBus;
    protected String mPortAddress;
    private Context mDeviceContext = new Context();
    private Device mDevice;
    private DeviceHandle mDeviceHandle;
    private DeviceDescriptor mDeviceDescriptor;
    private TransferManager mTransferManager = new TransferManager();
    private UsbEventProcessor mEventProcessor = new UsbEventProcessor();
    private AtomicBoolean mStreaming = new AtomicBoolean();
    private boolean mRunning = false;
    private int mAnomalousTransfersDetected = 0;

    public USBTunerController(int bus, String portAddress, ITunerErrorListener tunerErrorListener) {
        super(tunerErrorListener);
        this.mBus = bus;
        this.mPortAddress = portAddress;
    }

    public USBTunerController(int bus, String portAddress, long minimum, long maximum, int halfBandwidth, double usablePercent, ITunerErrorListener tunerErrorListener) {
        this(bus, portAddress, tunerErrorListener);
        this.setMinimumFrequency(minimum);
        this.setMaximumFrequency(maximum);
        this.setMiddleUnusableHalfBandwidth(halfBandwidth);
        this.setUsableBandwidthPercentage(usablePercent);
    }

    @Override
    public abstract TunerType getTunerType();

    protected abstract INativeBufferFactory getNativeBufferFactory();

    protected abstract int getTransferBufferSize();

    protected abstract void deviceStart() throws SourceException;

    protected abstract void deviceStop();

    @Override
    public final void start() throws SourceException {
        if (this.mDeviceContext == null) {
            throw new SourceException("Device cannot be reused once it has been shutdown");
        }
        int status = LibUsb.init((Context)this.mDeviceContext);
        if (status != 0) {
            throw new SourceException("Can't initialize libusb library - " + LibUsb.errorName((int)status));
        }
        this.mDevice = this.findDevice();
        if (this.mDevice == null) {
            throw new SourceException("Couldn't find USB device at bus [" + this.mBus + "] port [" + this.mPortAddress + "]");
        }
        this.mDeviceDescriptor = new DeviceDescriptor();
        status = LibUsb.getDeviceDescriptor((Device)this.mDevice, (DeviceDescriptor)this.mDeviceDescriptor);
        if (status != 0) {
            this.mDeviceDescriptor = null;
            throw new SourceException("Can't obtain tuner's device descriptor - " + LibUsb.errorName((int)status));
        }
        this.mDeviceHandle = new DeviceHandle();
        status = LibUsb.open((Device)this.mDevice, (DeviceHandle)this.mDeviceHandle);
        LibUsb.unrefDevice((Device)this.mDevice);
        if (status == -3) {
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            this.mLog.error("Access to USB tuner denied - (windows) reinstall zadig driver or (linux) blacklist driver and/or check udev rules");
            throw new SourceException("access denied - if using linux, blacklist the default driver and/or install udev rules");
        }
        if (status != 0) {
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            this.mLog.error("Can't open USB tuner - check driver or Linux udev rules");
            throw new SourceException("Can't open USB tuner - reinstall driver? - " + LibUsb.errorName((int)status));
        }
        status = LibUsb.kernelDriverActive((DeviceHandle)this.mDeviceHandle, (int)0);
        if (status == 1 && (status = LibUsb.detachKernelDriver((DeviceHandle)this.mDeviceHandle, (int)0)) != 0) {
            this.mLog.error("Unable to detach kernel driver for USB tuner device - bus:" + this.mBus + " port:" + this.mPortAddress);
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            throw new SourceException("Can't detach kernel driver");
        }
        status = LibUsb.setConfiguration((DeviceHandle)this.mDeviceHandle, (int)1);
        if (status == -6) {
            this.mLog.error("Unable to set USB configuration on tuner - device is busy (in use by another application)");
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            throw new SourceException("USB tuner is in-use by another application");
        }
        if (status != 0) {
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            throw new SourceException("Can't set configuration (ie reset) on the USB tuner - " + LibUsb.errorName((int)status));
        }
        status = LibUsb.claimInterface((DeviceHandle)this.mDeviceHandle, (int)0);
        if (status == -6) {
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            throw new SourceException("USB tuner is in-use by another application");
        }
        if (status != 0) {
            this.mDeviceHandle = null;
            this.mDeviceDescriptor = null;
            throw new SourceException("Can't claim interface on USB tuner - " + LibUsb.errorName((int)status));
        }
        this.mRunning = true;
        try {
            this.deviceStart();
        }
        catch (Exception se) {
            this.mRunning = false;
            throw se;
        }
    }

    @Override
    public final void stop() {
        this.mRunning = false;
        Thread t = new Thread(() -> {
            this.stopStreaming();
            this.mNativeBufferBroadcaster.clear();
            this.deviceStop();
        });
        t.start();
        try {
            t.join(500L);
            if (t.isAlive()) {
                this.mLog.info("Tuner shutdown exceeded 500ms - forcing shutdown");
                t.interrupt();
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.mTransferManager.freeTransfers();
        if (this.mDeviceHandle != null) {
            LibUsb.releaseInterface((DeviceHandle)this.mDeviceHandle, (int)0);
            LibUsb.close((DeviceHandle)this.mDeviceHandle);
            this.mDeviceHandle = null;
            this.mDevice = null;
            this.mDeviceDescriptor = null;
        }
        LibUsb.exit((Context)this.mDeviceContext);
        this.mDeviceContext = null;
    }

    private void startStreaming() {
        if (this.mStreaming.compareAndSet(false, true)) {
            try {
                this.prepareStreaming();
                List<Transfer> transfers = this.mTransferManager.getTransfers();
                this.mEventProcessor.start();
                this.mTransferManager.setAutoResubmitTransfers(true);
                this.mTransferManager.submitTransfers(transfers);
            }
            catch (SourceException se) {
                this.mLog.error("Error starting streaming on USB tuner", (Throwable)se);
            }
        }
    }

    protected void prepareStreaming() {
    }

    private void stopStreaming() {
        if (this.mStreaming.compareAndSet(true, false)) {
            this.mTransferManager.setAutoResubmitTransfers(false);
            this.mEventProcessor.stop();
            this.mTransferManager.cancelTransfers();
            this.mEventProcessor.handleFinalEvents();
            this.streamingCleanup();
        }
    }

    protected void streamingCleanup() {
    }

    private Device findDevice() throws SourceException {
        Device foundDevice = null;
        DeviceList deviceList = new DeviceList();
        int count = LibUsb.getDeviceList((Context)this.mDeviceContext, (DeviceList)deviceList);
        if (count >= 0) {
            for (Device device : deviceList) {
                int bus = LibUsb.getBusNumber((Device)device);
                int port = LibUsb.getPortNumber((Device)device);
                if (port > 0) {
                    String portAddress = TunerManager.getPortAddress(device);
                    if (this.mBus == bus && this.mPortAddress != null && this.mPortAddress.equals(portAddress)) {
                        foundDevice = device;
                        continue;
                    }
                    LibUsb.unrefDevice((Device)device);
                    continue;
                }
                LibUsb.unrefDevice((Device)device);
            }
        }
        LibUsb.freeDeviceList((DeviceList)deviceList, (boolean)false);
        if (foundDevice != null) {
            return foundDevice;
        }
        throw new SourceException("LibUsb couldn't discover USB device [" + this.mBus + ":" + this.mPortAddress + "] from device list" + (String)(count < 0 ? " - error: " + LibUsb.errorName((int)count) : ""));
    }

    protected Device getDevice() {
        return this.mDevice;
    }

    protected Context getDeviceContext() {
        return this.mDeviceContext;
    }

    protected DeviceDescriptor getDeviceDescriptor() {
        return this.mDeviceDescriptor;
    }

    protected DeviceHandle getDeviceHandle() {
        return this.mDeviceHandle;
    }

    protected boolean hasDeviceHandle() {
        return this.getDeviceHandle() != null;
    }

    protected boolean isRunning() {
        return this.mRunning;
    }

    @Override
    public void addBufferListener(Listener<INativeBuffer> listener) {
        if (this.isRunning()) {
            this.mBufferListenerLock.lock();
            try {
                boolean hasExistingListeners = this.hasBufferListeners();
                super.addBufferListener(listener);
                if (!hasExistingListeners) {
                    this.startStreaming();
                }
            }
            finally {
                this.mBufferListenerLock.unlock();
            }
        }
    }

    @Override
    public void removeBufferListener(Listener<INativeBuffer> listener) {
        this.mBufferListenerLock.lock();
        try {
            super.removeBufferListener(listener);
            if (!this.hasBufferListeners()) {
                this.stopStreaming();
            }
        }
        finally {
            this.mBufferListenerLock.unlock();
        }
    }

    class TransferManager
    implements TransferCallback {
        private List<Transfer> mAvailableTransfers;
        private LinkedTransferQueue<Transfer> mInProgressTransfers = new LinkedTransferQueue();
        private boolean mAutoResubmitTransfers = false;
        private int mTransferErrorCount = 0;
        private List<Transfer> mErrorTransfers = new ArrayList<Transfer>();
        private int mResubmitFailureLogCount = 0;

        TransferManager() {
        }

        private List<Transfer> getTransfers() throws SourceException {
            if (this.mAvailableTransfers == null) {
                this.mAvailableTransfers = new ArrayList<Transfer>();
                for (int x = 0; x < 8; ++x) {
                    Transfer transfer = LibUsb.allocTransfer();
                    if (transfer == null) {
                        throw new SourceException("Couldn't allocate USB transfer buffer - out of memory");
                    }
                    ByteBuffer buffer = ByteBuffer.allocateDirect(USBTunerController.this.getTransferBufferSize());
                    LibUsb.fillBulkTransfer((Transfer)transfer, (DeviceHandle)USBTunerController.this.mDeviceHandle, (byte)-127, (ByteBuffer)buffer, (TransferCallback)this, (Object)("Transfer Buffer " + x), (long)2000L);
                    this.mAvailableTransfers.add(transfer);
                }
            }
            return this.mAvailableTransfers;
        }

        private void setAutoResubmitTransfers(boolean resubmit) {
            this.mAutoResubmitTransfers = resubmit;
        }

        private void submitTransfers(List<Transfer> transfers) {
            for (Transfer transfer : transfers) {
                this.submitTransfer(transfer);
            }
        }

        private void submitTransfer(Transfer transfer) {
            int status = LibUsb.submitTransfer((Transfer)transfer);
            if (status == 0) {
                this.mInProgressTransfers.add(transfer);
                if (!this.mErrorTransfers.isEmpty()) {
                    Transfer toResubmit = this.mErrorTransfers.remove(0);
                    int resubmitStatus = LibUsb.submitTransfer((Transfer)toResubmit);
                    if (resubmitStatus == 0) {
                        this.mInProgressTransfers.add(transfer);
                        if (this.mErrorTransfers.size() >= this.mAvailableTransfers.size() / 2) {
                            USBTunerController.this.mLog.info("Successfully resubmitted previous error USB transfer buffer.  Current transfer buffer status (error queue/total available) [" + this.mErrorTransfers.size() + "/" + this.mAvailableTransfers.size() + "]");
                        }
                    } else {
                        this.mErrorTransfers.add(transfer);
                        ++this.mTransferErrorCount;
                        if (this.mErrorTransfers.size() >= this.mAvailableTransfers.size() / 2) {
                            USBTunerController.this.mLog.error("Attempt to resubmit previous error USB transfer buffer failed with status [" + LibUsb.errorName((int)resubmitStatus) + "] - this may be temporary and has happened [" + this.mTransferErrorCount + "] times so far.  Transfer buffer holding queue (error/total) [" + this.mErrorTransfers.size() + "/" + this.mAvailableTransfers.size() + "]");
                        }
                    }
                }
            } else {
                this.mErrorTransfers.add(transfer);
                ++this.mTransferErrorCount;
                if (this.mErrorTransfers.size() >= this.mAvailableTransfers.size() / 2) {
                    USBTunerController.this.mLog.error("Attempt to submit USB transfer buffer failed with status [" + LibUsb.errorName((int)status) + "] - this may be temporary and has happened [" + this.mTransferErrorCount + "] time(s) so far.  Transfer buffer holding queue (error/total) [" + this.mErrorTransfers.size() + "/" + this.mAvailableTransfers.size() + "]");
                }
            }
            if (this.mErrorTransfers.size() >= this.mAvailableTransfers.size()) {
                USBTunerController.this.mLog.error("Maximum USB transfer buffer errors reached - transfer buffers exhausted - shutting down USB tuner");
                ThreadPool.CACHED.submit(() -> USBTunerController.this.setErrorMessage("USB Error - Transfer Buffers Exhausted"));
            }
        }

        private void cancelTransfers() {
            for (Transfer transfer : this.mInProgressTransfers) {
                LibUsb.cancelTransfer((Transfer)transfer);
            }
            for (Transfer transfer : this.mErrorTransfers) {
                LibUsb.cancelTransfer((Transfer)transfer);
            }
        }

        private void freeTransfers() {
            if (this.mAvailableTransfers != null) {
                for (Transfer transfer : this.mAvailableTransfers) {
                    try {
                        LibUsb.freeTransfer((Transfer)transfer);
                    }
                    catch (Exception e) {
                        USBTunerController.this.mLog.error("Error releasing allocated USB transfer buffer during tuner shutdown: " + e.getLocalizedMessage());
                    }
                }
                this.mAvailableTransfers.clear();
                this.mAvailableTransfers = null;
            }
        }

        public void processTransfer(Transfer transfer) {
            this.mInProgressTransfers.remove(transfer);
            switch (transfer.status()) {
                case 0: 
                case 1: 
                case 2: 
                case 3: 
                case 4: {
                    int transferLength = transfer.actualLength();
                    if (transferLength > 0) {
                        this.dispatchTransfer(transfer);
                    }
                    transfer.buffer().rewind();
                    if (!this.mAutoResubmitTransfers) break;
                    this.submitTransfer(transfer);
                    break;
                }
                default: {
                    transfer.buffer().rewind();
                    if (!this.mAutoResubmitTransfers) break;
                    ThreadPool.CACHED.submit(() -> USBTunerController.this.setErrorMessage("LibUsb Transfer Error - stopping device - status [" + transfer.status() + "] - " + LibUsb.errorName((int)transfer.status())));
                }
            }
        }

        private void dispatchTransfer(Transfer transfer) {
            INativeBuffer nativeBuffer = USBTunerController.this.getNativeBufferFactory().getBuffer(transfer.buffer(), System.currentTimeMillis());
            USBTunerController.this.mNativeBufferBroadcaster.broadcast(nativeBuffer);
        }
    }

    class UsbEventProcessor
    implements Runnable {
        private Thread mThread;
        private boolean mProcessing = false;

        UsbEventProcessor() {
        }

        public void start() {
            if (this.mThread == null) {
                this.mProcessing = true;
                this.mThread = new Thread(this);
                this.mThread.setName("sdrtrunk USB tuner - bus [" + USBTunerController.this.mBus + "] port [" + USBTunerController.this.mPortAddress + "]");
                this.mThread.setPriority(10);
                this.mThread.start();
            }
        }

        public void stop() {
            this.mProcessing = false;
            try {
                this.mThread.join(1000L);
            }
            catch (Exception e) {
                USBTunerController.this.mLog.error("Error stopping LibUsb event processing thread - " + e.getLocalizedMessage());
            }
            this.mThread = null;
        }

        public void handleFinalEvents() {
            try {
                LibUsb.handleEventsTimeout((Context)USBTunerController.this.mDeviceContext, (long)50L);
            }
            catch (Throwable throwable) {
                USBTunerController.this.mLog.error("Error while processing stop-streaming LibUsb timeout events", throwable);
            }
        }

        @Override
        public void run() {
            this.mProcessing = true;
            while (this.mProcessing) {
                try {
                    LibUsb.handleEventsTimeout((Context)USBTunerController.this.mDeviceContext, (long)250L);
                }
                catch (Throwable throwable) {
                    USBTunerController.this.mLog.error("Error while processing LibUsb timeout events", throwable);
                }
            }
        }
    }
}

