/*
 * enumNG.h
 * @authors: Fabian Angelstorf, Franziska Juckel
 * @copyright: Horst Goertz Institute for IT-Security, Ruhr-University Bochum
 *
 * Reads the createConfig generated by createNG as well as the level files
 * using the nGramReader. Based on the settings and the level read, enumNG
 * performs the ordered password enumeration.
 */

#ifndef ENUM_3G_H_
#define ENUM_3G_H_

#include "common.h"
#include "cmdlineEnumNG.h"
#include "stdint.h"

// === Defines ===
#define DEFAULT_OUTPUT_CYCLE_FACTOR 1000
#define DEFAULT_MAX_ATTEMPTS 1000000000
#define MAX_MAX_ATTEMPTS 1000000000000000
/*
 *  These variables are needed for the attackSimulator functions
 *  and are therefore declared as extern.
 */
extern uint64_t glbl_attemptsCount; // couting attempts done
extern uint64_t glbl_createdLengths[MAX_PASSWORD_LENGTH]; // counting the lengths of the created PWs

// === public functions ===

/*
 * initializes all global parameters, setting them to their default value
 * !! this function must be called before any other operation !!
 */
void initialize ();

/*
 *  prints all Error-Messages (if any), clears the allocated memory of the
 *  global variables and ends the application
 *  the char* exit_msg is printed out on the command line
 *  !! this function is set via atexit() !!
 */
void exit_routine ();

/*
 * Evaluates given command line arguments
 * there has to be at least 1 argument: the input filename
 * additional arguments are evaluated in this method an the
 * corresponding parameters are set
 * returns TRUE, if the evaluation was successful
 */
bool evaluate_arguments (struct gengetopt_args_info *args_info);

/*
 * Reads any needed input file using the nGramIO-functions and
 * sets all needed variables accordingly.
 * Returns TRUE on success and FALSE if something went wrong.
 */
bool apply_settings ();
bool sort_ngrams ();

/*
 *  main process: runs the password enumeration
 *  Generates levelChains using the generate_levelChain_live function.
 *  Once a levelChain is found, the actual passwords are generated
 *  based on the current levelChain. The amount of passwords generated
 *  equals the global variable @attemptsMax.
 *  run_enumeration_fixedLenghts() works just like run_enumeration(), but
 *  for a fixed length only (set by argument -L)
 */
void run_enumeration ();
void run_enumeration_fixedLenghts ();
void run_enumeration_optimizedLengths ();

/*
 * Generates the next levelChain based on the given @levelChain, with
 * a @length and a level of @levelMax.
 * To generate a levelChain with different length or level, reset each value
 * of @levelChain to the minimum.
 * The new levelChain is stored in @levelChain.
 * ATTENTION: don't forget the length of the initalProb! To generate
 * passwords based on 3grams (with a initialProb size of 2), each levelChain
 * must be 1 char shorter then the actual password, since initialProb uses
 * one integer from the levelChain, but 2 chars in the password!
 */
bool getNext_levelChain (int levelChain[MAX_PASSWORD_LENGTH], // next levelChain
                         int length,  // length for the next levelChain
                         int levelMax,  // sum level for the next levelChain
                         bool newChain);  // new chain? or create one on based the given levelChain?

/*
 * Generates the first chars of a new Password. The used initialProbs
 * have a level equal to @iP_level. Calls the intern function
 * enumerate_password_recursivly to generate the rest of each password.
 * Returns FALSE if as many passwords as glbl_attemptsMax have been created.
 */
bool enumerate_password (int levelChain[MAX_PASSWORD_LENGTH], // levelChain specifying the level of each char in the passwords
                         int lengthMax);  // length of the passwords to be generated

/*
 * Creates a new result folder under the subfolder "results". The created
 * folder is named after the current date and time ("Year-Mon-Day_Hou.Min").
 * If the folder all ready exist, an incrementing integer is appended.
 * After creating the folder, the basic result file "createdPWs.txt" is
 * created and assigned to the global file pointer "glbl_FP_generatedPasswords".
 * If any of those operation fails, the application is aborted!
 */
void create_resultFolder ();

/*
 * Prints the by arguments selected mode as well as the output and input filenames
 * to the given file pointer @fp.
 */
void print_settings_enumNG (FILE * fp);

/*
 * Prints the results of the current run through if the verboseMode is active
 * to the given file pointer @fp.
 */
void print_report_enumNG (FILE * fp);

/*
 * Prints a log file under.
 * Returns TRUE if the log file creation was successful, FALSE if not.
 */
bool print_log ();

bool boost_allHints ();

bool boostingMode ();

/*
 * Handles signal "ctrl + c" for application abortion
 */
void sigint_handler (int s);

#endif /* ENUM_3G_H_ */
