'''
Generators that repeate stuff.

Modified by Jose Miguel Esparza to use in Malybuzz fuzzer

@author: Michael Eddington
@version: $Id: repeater.py 279 2007-05-08 01:21:58Z meddingt $
'''

#
# Copyright 2007 Jose Miguel Esparza
# Copyright (c) 2005-2007 Michael Eddington
# Copyright (c) 2004-2005 IOActive Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy 
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights 
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
# copies of the Software, and to permit persons to whom the Software is 
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in	
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#

# Authors:
#   Michael Eddington (mike@phed.org)

# $Id: repeater.py 279 2007-05-08 01:21:58Z meddingt $

import static, sys
from Peach import generator, group

#__all__ = ['Repeater','RepeaterOverflow']

class Repeater(generator.Generator):
	'''
	Will repeat a value (generated by a Generator) by round count.  Can be 
	used for basic buffer overflow testing.
	
	Example:
	
		>>> gen = Repeater(None, String("A"), 3)
		>>> gen.getValue()
		A
		>>> gen.next()
		>>> gen.getValue()
		AA
		>>> gen.next()
		>>> gen.getValue()
		AAA
	
	Example:
	
		>>> gen = Repeater(None, Static("Peach "), 5, 3)
		>>> gen.getValue()
		Peach
		>>> gen.next()
		>>> gen.getValue()
		Peach Peach Peach Peach Peach
		>>> gen.next()
		>>> gen.getValue()
		Peach Peach Peach Peach Peach Peach Peach Peach Peach Peach
		
	
	'''
	
	_incrementor = None
	_roundCount = 1
	_generator = None
	_maxSteps = -1
	_ret = ''
	
	def __init__(self, group, generator, incrementor = 1, maxSteps = -1, startStep = None):
		'''
		@type	group: Group
		@param	group: Group this generator belongs to
		@type	generator: Generator
		@param	generator: Generator to repeate
		@type	incrementor: number
		@param	incrementor: Multiplier against round count
		@type	maxSteps: number
		@param	maxSteps: Maximum repeates
		@type	startSteps: number
		@param	startSteps: Start at this step
		'''
		self._generator = generator
		self._incrementor = incrementor
		self.setGroup(group)
		self._maxSteps = maxSteps
		self._startStep = startStep
		
		if self._startStep != None:
			self._roundCount = self._startStep
	
	def next(self):
		self._roundCount+=1
		if self._maxSteps != -1 and self._roundCount > self._maxSteps:
			self._roundCount -= 1
			raise generator.GeneratorCompleted("Peach.repeater.Repeater")
	
	def getRawValue(self):
		# Hah, this is much faster then the old way!
		self._ret += str(self._generator.getValue()) * self._incrementor
		#for i in range(self._roundCount*self._incrementor):
		#	ret += self._generator.getValue()
		return self._ret
	
	def getGenerator(self):
		'''
		Get Generator who's value we will repeat.
		
		@rtype: Generator
		@return: Generator we are repeating
		'''
		return self._generator
	def setGenerator(self, generator):
		'''
		Set Generator who's value we will repeat.
		
		@type	generator: Generator
		@param	generator: Generator to repeate
		'''
		self._generator = generator
	
	def reset(self):
		self._roundCount = 1
		self._ret = ''
		
		if self._startStep != None:
			self._roundCount = self._startStep
		
		self._generator.reset()
	
	def unittest():
		g = group.Group()
		r = Repeater(g, static.Static('A'), 1, 10)
		
		try:
			while g.next():
				print r.getValue()
		except group.GroupCompleted:
			pass
	unittest = staticmethod(unittest)


class RepeaterGI(generator.Generator):
	'''
	Will repeat a value (generated by a Generator) by multiplier (generator).
	
	Example:
	
		Repeater(None, String("A"), BadUnsignedNumbers(None))
	
	Would produce a string of A's the length of each number returned by
	BadUnsignedNumbers.
	
	'''
	
	_incrementor = None
	_roundCount = 1
	_generator = None
	
	def __init__(self, group, generator, incrementor):
		'''
		@type	group: Group
		@param	group: Group this generator belongs to
		@type	generator: Generator
		@param	generator: Generator to repeate
		@type	incrementor: Generator
		@param	incrementor: Multiplier against round count
		'''
		self._generator = generator
		self._incrementor = incrementor
		self.setGroup(group)
	
	def next(self):
		self._roundCount+=1
		self._incrementor.next()
	
	def getRawValue(self):
		try:
			ret = str(self._generator.getValue()) * int(self._incrementor.getValue())
		
		except OverflowError:
			# Integer overflow exception.  Oh well, we tried!
			ret = self._generator.getValue()
		
		except MemoryError:
			ret = self._generator.getValue()
			#print "RepeaterGI: MemoryError! Value is %d long multiplier is %d." % (
			#	len(str(ret)), int(self._incrementor.getValue()))
			
		return ret
	
	def reset(self):
		self._roundCount = 1
		self._incrementor.reset()
		self._generator.reset()
	
def unittest():
	Repeater.unittest()


class RepeaterOverflow(Repeater):
	'''
	Class to generate chains of data with the specified character and an usual buffer length (power of 2) +/- offset 
	'''
	
	# Power base to obtain a length next to the normal buffer lengths
	_base = 2	
	_firstExponent = None
	_lastExponent = None
	_actualExponent = None
	_roundCount = 0
	_steps = 0
	_mul = 0
	_baseLength = 0
	_offset = 20
	_ret = ''
	_indexValue = 0
	
	def __init__(self, group, generator, firstExponent = 5, lastExponent = 13, maxSteps = -1):
		'''
		@type	group: Group
		@param	group: Group this generator belongs to
		@type	generator: Generator
		@param	generator: Generator to repeate
		@type	firstExponent: integer
		@param	firstExponent: initial exponent of the power to obtain the chain length
		@type	lastExponent: integer
		@param	lastExponent: last exponent of the power to obtain the chain length
		@type	maxSteps: integer
		@param	maxSteps: Maximum repeates
		'''
		self._generator = generator
		self._firstExponent = firstExponent
		self._actualExponent = firstExponent
		self._lastExponent = lastExponent
		self.setGroup(group)
		self._maxSteps = maxSteps
		
	def next(self):
		self._roundCount+=1
		self._steps+=1
		if self._maxSteps != -1 and self._steps > self._maxSteps:
			raise generator.GeneratorCompleted("Peach.repeater.RepeaterOverflow")
		if self._roundCount == self._offset*2:
			if self._baseLength+self._offset == self._base**self._lastExponent:
				raise generator.GeneratorCompleted("Peach.repeater.RepeaterOverflow")
			self._roundCount = 0
			if self._actualExponent < 10:
				self._actualExponent += 1
			else:
				self._mul += 1


	
	def getRawValue(self):
		value = self._generator.getValue()
		if self._roundCount == 0:
			if self._actualExponent >= 10:
				self._baseLength = self._base**self._actualExponent+(1024*self._mul)-self._offset
			else:
				self._baseLength = self._base**self._actualExponent-self._offset
			if self._baseLength < 0:
				sys.exit("Error: Negative value generating overflow test")
			self._ret = value*(self._baseLength/len(value))
		else:
			self._ret += value[self._indexValue]
			if self._indexValue == len(value)-1:
				self._indexValue = 0
			else:
				self._indexValue += 1
		return self._ret




# ############################################################################

import inspect, pyclbr

def RunUnit(obj, clsName):
	print "Unittests for: %s..." % clsName,
	cnt = 0
	try:
		while True:
			s = obj.getValue()
			obj.next()
			cnt+=1
			
	except GeneratorCompleted:
		print "%d tests found." % cnt

if __name__ == "__main__":
	print "\n -- Running A Quick Unittest for %s --\n" % __file__
	mod = inspect.getmodulename(__file__)
	for clsName in pyclbr.readmodule(mod):
		cls = globals()[clsName]
		if str(cls).find('__main__') > -1 and hasattr(cls, 'next') and hasattr(cls, 'getValue'):
			try:
				RunUnit(cls(), clsName)
			except TypeError:
				pass

# end
