#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["UrlDisclosure"]

from .. import WebVulnerability

from posixpath import split


#------------------------------------------------------------------------------
class UrlDisclosure(WebVulnerability):
    """
    URL Disclosure.

    These are URLs that are accessible but not linked from the web site itself.
    It may indicate a poor attempt at concealing information.
    For example:
    - Backup files: http://www.example.com/ **index.php.old**
    - Alternative file names: http://www.example.com/ **index4.php**
    - Remnants of deployment: http://www.example.com/ **build.xml**
    - Poorly configured servers: http://www.example.com/ **error_log**
    - Forgotten server files: http://www.example.com/ **server-status**

    Remove any sensitive information that may have been left behind. If it's
    not possible to remove it, block access to it from the HTTP server.
    """

    DEFAULTS = WebVulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "informational"
    DEFAULTS["cwe"]        = "CWE-200"
    DEFAULTS["cvss_base"]  = "2.9"
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/Information_Leakage",
    )


    #--------------------------------------------------------------------------
    @property
    def discovered_path(self):
        """
        Discovered part of the URL.

        >>> from golismero.api.data.resource.url import URL
        >>> from golismero.api.data.vulnerability.information_disclosure.url_disclosure import UrlDisclosure
        >>> url = Url('http://www.example.com/path/to_the/file/index.php.old')
        >>> url_disclosure = UrlDisclosure(url)
        >>> url_disclosure
        <UrlDisclosure url='http://www.example.com/path/to_the/file/index.php.old'>
        >>> url_disclosure.discovered
        'index.php.old'

        :returns: Discovered part of the URL.
        :rtype: str
        """
        return split(self.url.parsed_url.path)[1]
