// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { isTokenCredential } from "@azure/core-auth";
import { isNodeLike } from "@azure/core-util";
import { isPipelineLike, newPipeline } from "./Pipeline.js";
import { DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS, DEFAULT_HIGH_LEVEL_CONCURRENCY, FILE_MAX_SIZE_BYTES, FILE_RANGE_MAX_SIZE_BYTES, URLConstants, } from "./utils/constants.js";
import { appendToURLPath, setURLParameter, truncatedISO8061Date, extractConnectionStringParts, getShareNameAndPathFromUrl, appendToURLQuery, httpAuthorizationToString, setURLPath, setURLQueries, EscapePath, ConvertInternalResponseOfListFiles, ConvertInternalResponseOfListHandles, assertResponse, removeEmptyString, asSharePermission, parseOctalFileMode, toOctalFileMode, } from "./utils/utils.common.js";
import { Credential } from "@azure/storage-common";
import { StorageSharedKeyCredential } from "@azure/storage-common";
import { AnonymousCredential } from "@azure/storage-common";
import { tracingClient } from "./utils/tracing.js";
import { StorageClient } from "./StorageClient.js";
import { FileDownloadResponse } from "./FileDownloadResponse.js";
import { rangeToString } from "./Range.js";
import { fileAttributesToString, fileCreationTimeToString, fileLastWriteTimeToString, validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions, validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions, toShareProtocolsString, toShareProtocols, fileChangeTimeToString, } from "./models.js";
import { Batch } from "./utils/Batch.js";
import { BufferScheduler } from "./utils/BufferScheduler.js";
import { fsStat, fsCreateReadStream, readStreamToLocalFile, streamToBuffer, } from "./utils/utils.js";
import { randomUUID } from "@azure/core-util";
import { generateFileSASQueryParameters, generateFileSASQueryParametersInternal, } from "./FileSASSignatureValues.js";
/**
 * A ShareClient represents a URL to the Azure Storage share allowing you to manipulate its directories and files.
 */
export class ShareClient extends StorageClient {
    /**
     * Share operation context provided by protocol layer.
     */
    context;
    _name;
    shareClientConfig;
    /**
     * The name of the share
     */
    get name() {
        return this._name;
    }
    constructor(urlOrConnectionString, credentialOrPipelineOrShareName, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    options) {
        let pipeline;
        let url;
        if (isPipelineLike(credentialOrPipelineOrShareName)) {
            // (url: string, pipeline: Pipeline)
            url = urlOrConnectionString;
            pipeline = credentialOrPipelineOrShareName;
        }
        else if (credentialOrPipelineOrShareName instanceof Credential ||
            isTokenCredential(credentialOrPipelineOrShareName)) {
            // (url: string, credential?: Credential, options?: ShareClientOptions)
            url = urlOrConnectionString;
            pipeline = newPipeline(credentialOrPipelineOrShareName, options);
        }
        else if (!credentialOrPipelineOrShareName &&
            typeof credentialOrPipelineOrShareName !== "string") {
            // (url: string, credential?: Credential, options?: ShareClientOptions)
            // The second parameter is undefined. Use anonymous credential.
            url = urlOrConnectionString;
            pipeline = newPipeline(new AnonymousCredential(), options);
        }
        else if (credentialOrPipelineOrShareName &&
            typeof credentialOrPipelineOrShareName === "string") {
            // (connectionString: string, name: string, options?: ShareClientOptions)
            const extractedCreds = extractConnectionStringParts(urlOrConnectionString);
            const name = credentialOrPipelineOrShareName;
            if (extractedCreds.kind === "AccountConnString") {
                if (isNodeLike) {
                    const sharedKeyCredential = new StorageSharedKeyCredential(extractedCreds.accountName, extractedCreds.accountKey);
                    url = appendToURLPath(extractedCreds.url, name);
                    pipeline = newPipeline(sharedKeyCredential, options);
                }
                else {
                    throw new Error("Account connection string is only supported in Node.js environment");
                }
            }
            else if (extractedCreds.kind === "SASConnString") {
                url = appendToURLPath(extractedCreds.url, name) + "?" + extractedCreds.accountSas;
                pipeline = newPipeline(new AnonymousCredential(), options);
            }
            else {
                throw new Error("Connection string must be either an Account connection string or a SAS connection string");
            }
        }
        else {
            throw new Error("Expecting non-empty strings for name parameter");
        }
        super(url, pipeline);
        this._name = getShareNameAndPathFromUrl(this.url).shareName;
        this.shareClientConfig = options;
        this.context = this.storageClientContext.share;
    }
    /**
     * Creates a new ShareClient object identical to the source but with the specified snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base share.
     *
     * @param snapshot - The snapshot timestamp.
     * @returns A new ShareClient object identical to the source but with the specified snapshot timestamp
     */
    withSnapshot(snapshot) {
        return new ShareClient(setURLParameter(this.url, URLConstants.Parameters.SHARE_SNAPSHOT, snapshot.length === 0 ? undefined : snapshot), this.pipeline, this.shareClientConfig);
    }
    /**
     * Creates a new share under the specified account. If the share with
     * the same name already exists, the operation fails.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-share
     *
     * @param options - Options to Share Create operation.
     * @returns Response data for the Share Create operation.
     */
    async create(options = {}) {
        return tracingClient.withSpan("ShareClient-create", options, async (updatedOptions) => {
            return assertResponse(await this.context.create({
                ...updatedOptions,
                ...this.shareClientConfig,
                enabledProtocols: toShareProtocolsString(updatedOptions.protocols),
            }));
        });
    }
    /**
     * Creates a new share under the specified account. If the share with
     * the same name already exists, it is not changed.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-share
     *
     * @param options -
     */
    async createIfNotExists(options = {}) {
        return tracingClient.withSpan("ShareClient-createIfNotExists", options, async (updatedOptions) => {
            try {
                const res = await this.create(updatedOptions);
                return {
                    succeeded: true,
                    ...res,
                };
            }
            catch (e) {
                if (e.details?.errorCode === "ShareAlreadyExists") {
                    return {
                        succeeded: false,
                        ...e.response?.parsedHeaders,
                        _response: e.response,
                    };
                }
                throw e;
            }
        });
    }
    /**
     * Creates a {@link ShareDirectoryClient} object.
     *
     * @param directoryName - A directory name
     * @returns The ShareDirectoryClient object for the given directory name.
     */
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-subclients */
    getDirectoryClient(directoryName) {
        return new ShareDirectoryClient(appendToURLPath(this.url, EscapePath(directoryName)), this.pipeline, this.shareClientConfig);
    }
    /**
     * Gets the directory client for the root directory of this share.
     * Note that the root directory always exists and cannot be deleted.
     *
     * @readonly A new ShareDirectoryClient object for the root directory.
     */
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-subclients */
    get rootDirectoryClient() {
        return this.getDirectoryClient("");
    }
    /**
     * Creates a new subdirectory under this share.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Create operation.
     * @returns Directory creation response data and the corresponding directory client.
     */
    async createDirectory(directoryName, options = {}) {
        return tracingClient.withSpan("ShareClient-createDirectory", options, async (updatedOptions) => {
            const directoryClient = this.getDirectoryClient(directoryName);
            const directoryCreateResponse = await directoryClient.create(updatedOptions);
            return {
                directoryClient,
                directoryCreateResponse,
            };
        });
    }
    /**
     * Removes the specified empty sub directory under this share.
     * Note that the directory must be empty before it can be deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Delete operation.
     * @returns Directory deletion response data.
     */
    async deleteDirectory(directoryName, options = {}) {
        return tracingClient.withSpan("ShareClient-deleteDirectory", options, async (updatedOptions) => {
            const directoryClient = this.getDirectoryClient(directoryName);
            return directoryClient.delete(updatedOptions);
        });
    }
    /**
     * Creates a new file or replaces a file under the root directory of this share.
     * Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param fileName -
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns File creation response data and the corresponding file client.
     */
    async createFile(fileName, size, options = {}) {
        return tracingClient.withSpan("ShareClient-createFile", options, async (updatedOptions) => {
            const directoryClient = this.rootDirectoryClient;
            const fileClient = directoryClient.getFileClient(fileName);
            const fileCreateResponse = await fileClient.create(size, updatedOptions);
            return {
                fileClient,
                fileCreateResponse,
            };
        });
    }
    /**
     * Removes a file under the root directory of this share from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code `SharingViolation`
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400
     * (`InvalidQueryParameterValue`)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param directoryName -
     * @param fileName -
     * @param options - Options to File Delete operation.
     * @returns Promise<FileDeleteResponse> File Delete response data.
     */
    async deleteFile(fileName, options = {}) {
        return tracingClient.withSpan("ShareClient-deleteFile", options, async (updatedOptions) => {
            const directoryClient = this.rootDirectoryClient;
            const fileClient = directoryClient.getFileClient(fileName);
            return fileClient.delete(updatedOptions);
        });
    }
    /**
     * Returns true if the Azrue share resource represented by this client exists; false otherwise.
     *
     * NOTE: use this function with care since an existing share might be deleted by other clients or
     * applications. Vice versa new shares might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    async exists(options = {}) {
        return tracingClient.withSpan("ShareClient-exists", options, async (updatedOptions) => {
            try {
                await this.getProperties(updatedOptions);
                return true;
            }
            catch (e) {
                if (e.statusCode === 404) {
                    return false;
                }
                throw e;
            }
        });
    }
    /**
     * Returns all user-defined metadata and system properties for the specified
     * share.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-share-properties
     *
     * WARNING: The `metadata` object returned in the response will have its keys in lowercase, even if
     * they originally contained uppercase characters. This differs from the metadata keys returned by
     * the `listShares` method of {@link ShareServiceClient} using the `includeMetadata` option, which
     * will retain their original casing.
     *
     * @returns Response data for the Share Get Properties operation.
     */
    async getProperties(options = {}) {
        return tracingClient.withSpan("ShareClient-getProperties", options, async (updatedOptions) => {
            const res = assertResponse(await this.context.getProperties(updatedOptions));
            return {
                ...res,
                ...this.shareClientConfig,
                protocols: toShareProtocols(res.enabledProtocols),
            };
        });
    }
    /**
     * Marks the specified share for deletion. The share and any directories or files
     * contained within it are later deleted during garbage collection.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-share
     *
     * @param options - Options to Share Delete operation.
     * @returns Response data for the Share Delete operation.
     */
    async delete(options = {}) {
        return tracingClient.withSpan("ShareClient-delete", options, async (updatedOptions) => {
            return assertResponse(await this.context.delete({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Marks the specified share for deletion if it exists. The share and any directories or files
     * contained within it are later deleted during garbage collection.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-share
     *
     * @param options -
     */
    async deleteIfExists(options = {}) {
        return tracingClient.withSpan("ShareClient-deleteIfExists", options, async (updatedOptions) => {
            try {
                const res = await this.delete(updatedOptions);
                return {
                    succeeded: true,
                    ...res,
                };
            }
            catch (e) {
                if (e.details?.errorCode === "ShareNotFound" ||
                    e.details?.errorCode === "ShareSnapshotNotFound") {
                    return {
                        succeeded: false,
                        ...e.response?.parsedHeaders,
                        _response: e.response,
                    };
                }
                throw e;
            }
        });
    }
    /**
     * Sets one or more user-defined name-value pairs for the specified share.
     *
     * If no option provided, or no metadata defined in the option parameter, the share
     * metadata will be removed.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-share-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed.
     * @param option - Options to Share Set Metadata operation.
     * @returns Response data for the Share Set Metadata operation.
     */
    async setMetadata(metadata, options = {}) {
        return tracingClient.withSpan("ShareClient-setMetadata", options, async (updatedOptions) => {
            return assertResponse(await this.context.setMetadata({
                ...updatedOptions,
                ...this.shareClientConfig,
                metadata,
            }));
        });
    }
    /**
     * Gets the permissions for the specified share. The permissions indicate
     * whether share data may be accessed publicly.
     *
     * WARNING: JavaScript Date will potential lost precision when parsing start and expiry string.
     * For example, new Date("2018-12-31T03:44:23.8827891Z").toISOString() will get "2018-12-31T03:44:23.882Z".
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/get-share-acl
     *
     * @param option - Options to Share Get Access Policy operation.
     * @returns Response data for the Share Get Access Policy operation.
     */
    async getAccessPolicy(options = {}) {
        return tracingClient.withSpan("ShareClient-getAccessPolicy", options, async (updatedOptions) => {
            const response = assertResponse(await this.context.getAccessPolicy({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
            const res = {
                _response: response._response,
                date: response.date,
                etag: response.etag,
                lastModified: response.lastModified,
                requestId: response.requestId,
                signedIdentifiers: [],
                version: response.version,
            };
            for (const identifier of response) {
                let accessPolicy = undefined;
                if (identifier.accessPolicy) {
                    accessPolicy = {
                        permissions: identifier.accessPolicy.permissions,
                    };
                    if (identifier.accessPolicy.expiresOn) {
                        accessPolicy.expiresOn = new Date(identifier.accessPolicy.expiresOn);
                    }
                    if (identifier.accessPolicy.startsOn) {
                        accessPolicy.startsOn = new Date(identifier.accessPolicy.startsOn);
                    }
                }
                res.signedIdentifiers.push({
                    accessPolicy,
                    id: identifier.id,
                });
            }
            return res;
        });
    }
    /**
     * Sets the permissions for the specified share. The permissions indicate
     * whether directories or files in a share may be accessed publicly.
     *
     * When you set permissions for a share, the existing permissions are replaced.
     * If no shareAcl provided, the existing share ACL will be
     * removed.
     *
     * When you establish a stored access policy on a share, it may take up to 30 seconds to take effect.
     * During this interval, a shared access signature that is associated with the stored access policy will
     * fail with status code 403 (Forbidden), until the access policy becomes active.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-share-acl
     *
     * @param shareAcl - Array of signed identifiers, each having a unique Id and details of access policy.
     * @param option - Options to Share Set Access Policy operation.
     * @returns Response data for the Share Set Access Policy operation.
     */
    async setAccessPolicy(shareAcl, options = {}) {
        return tracingClient.withSpan("ShareClient-setAccessPolicy", options, async (updatedOptions) => {
            const acl = [];
            for (const identifier of shareAcl || []) {
                acl.push({
                    accessPolicy: {
                        expiresOn: identifier.accessPolicy?.expiresOn
                            ? truncatedISO8061Date(identifier.accessPolicy.expiresOn)
                            : undefined,
                        permissions: identifier.accessPolicy?.permissions,
                        startsOn: identifier.accessPolicy?.startsOn
                            ? truncatedISO8061Date(identifier.accessPolicy.startsOn)
                            : undefined,
                    },
                    id: identifier.id,
                });
            }
            return assertResponse(await this.context.setAccessPolicy({
                ...updatedOptions,
                ...this.shareClientConfig,
                shareAcl: acl,
            }));
        });
    }
    /**
     * Creates a read-only snapshot of a share.
     *
     * @param options - Options to Share Create Snapshot operation.
     * @returns Response data for the Share Create Snapshot operation.
     */
    async createSnapshot(options = {}) {
        return tracingClient.withSpan("ShareClient-createSnapshot", options, async (updatedOptions) => {
            return assertResponse(await this.context.createSnapshot({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Sets quota for the specified share.
     *
     * @deprecated Use {@link ShareClient.setProperties} instead.
     *
     * @param quotaInGB - Specifies the maximum size of the share in gigabytes
     * @param option - Options to Share Set Quota operation.
     * @returns Response data for the Share Get Quota operation.
     */
    async setQuota(quotaInGB, options = {}) {
        return tracingClient.withSpan("ShareClient-setQuota", options, async (updatedOptions) => {
            return assertResponse(await this.context.setProperties({
                ...updatedOptions,
                ...this.shareClientConfig,
                quota: quotaInGB,
            }));
        });
    }
    /**
     * Sets properties of the share.
     *
     * @param option - Options to Share Set Properties operation.
     * @returns Response data for the Share Set Properties operation.
     */
    async setProperties(options = {}) {
        return tracingClient.withSpan("ShareClient-setProperties", options, async (updatedOptions) => {
            return assertResponse(await this.context.setProperties({
                ...options,
                ...this.shareClientConfig,
                quota: options.quotaInGB,
                tracingOptions: updatedOptions.tracingOptions,
            }));
        });
    }
    /**
     * Retrieves statistics related to the share.
     *
     * @param option - Options to Share Get Statistics operation.
     * @returns Response data for the Share Get Statistics operation.
     */
    async getStatistics(options = {}) {
        return tracingClient.withSpan("ShareClient-getStatistics", options, async (updatedOptions) => {
            const response = assertResponse(await this.context.getStatistics({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
            const GBBytes = 1024 * 1024 * 1024;
            return { ...response, shareUsage: Math.ceil(response.shareUsageBytes / GBBytes) };
        });
    }
    /**
     * Creates a file permission (a security descriptor) at the share level.
     * The created security descriptor can be used for the files/directories in the share.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-permission
     *
     * @param options - Options to Share Create Permission operation.
     * @param filePermission - File permission described in the SDDL
     */
    async createPermission(filePermission, options = {}) {
        return tracingClient.withSpan("ShareClient-createPermission", options, async (updatedOptions) => {
            return assertResponse(await this.context.createPermission(asSharePermission(filePermission), {
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Gets the Security Descriptor Definition Language (SDDL) for a given file permission key
     * which indicates a security descriptor.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-permission
     *
     * @param options - Options to Share Create Permission operation.
     * @param filePermissionKey - File permission key which indicates the security descriptor of the permission.
     */
    async getPermission(filePermissionKey, options = {}) {
        return tracingClient.withSpan("ShareClient-getPermission", options, async (updatedOptions) => {
            return assertResponse(await this.context.getPermission(filePermissionKey, {
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Get a {@link ShareLeaseClient} that manages leases on the file.
     *
     * @param proposeLeaseId - Initial proposed lease Id.
     * @returns A new ShareLeaseClient object for managing leases on the file.
     */
    getShareLeaseClient(proposeLeaseId) {
        return new ShareLeaseClient(this, proposeLeaseId);
    }
    /**
     * Only available for ShareClient constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the SAS when the client is initialized with a shared key credential");
        }
        const sas = generateFileSASQueryParameters({
            shareName: this.name,
            ...options,
        }, this.credential).toString();
        return appendToURLQuery(this.url, sas);
    }
    /**
     * Only available for ShareClient constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options*/
    generateSasStringToSign(options) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the SAS when the client is initialized with a shared key credential");
        }
        return generateFileSASQueryParametersInternal({
            shareName: this.name,
            ...options,
        }, this.credential).stringToSign;
    }
}
/**
 * A ShareDirectoryClient represents a URL to the Azure Storage directory allowing you to manipulate its files and directories.
 */
export class ShareDirectoryClient extends StorageClient {
    /**
     * context provided by protocol layer.
     */
    context;
    _shareName;
    _path;
    _name;
    shareClientConfig;
    /**
     * The share name corresponding to this directory client
     */
    get shareName() {
        return this._shareName;
    }
    /**
     * The full path of the directory
     */
    get path() {
        return this._path;
    }
    /**
     * The name of the directory
     */
    get name() {
        return this._name;
    }
    constructor(url, credentialOrPipeline, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options = {}) {
        let pipeline;
        if (isPipelineLike(credentialOrPipeline)) {
            pipeline = credentialOrPipeline;
        }
        else if (credentialOrPipeline instanceof Credential ||
            isTokenCredential(credentialOrPipeline)) {
            pipeline = newPipeline(credentialOrPipeline, options);
        }
        else {
            // The second parameter is undefined. Use anonymous credential.
            pipeline = newPipeline(new AnonymousCredential(), options);
        }
        super(url, pipeline);
        ({
            baseName: this._name,
            shareName: this._shareName,
            path: this._path,
        } = getShareNameAndPathFromUrl(this.url));
        this.shareClientConfig = options;
        this.context = this.storageClientContext.directory;
    }
    /**
     * Creates a new directory under the specified share or parent directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param options - Options to Directory Create operation.
     * @returns Response data for the Directory  operation.
     */
    async create(options = {}) {
        if (!options.fileAttributes) {
            options = validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions(options);
        }
        return tracingClient.withSpan("ShareDirectoryClient-create", options, async (updatedOptions) => {
            const rawResponse = await this.context.create({
                ...updatedOptions,
                fileChangeOn: fileChangeTimeToString(updatedOptions.changeTime),
                fileCreatedOn: fileCreationTimeToString(updatedOptions.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(updatedOptions.lastWriteTime),
                fileAttributes: updatedOptions.fileAttributes
                    ? fileAttributesToString(updatedOptions.fileAttributes)
                    : undefined,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                ...this.shareClientConfig,
            });
            const wrappedRes = {
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    fileType: rawResponse.nfsFileType,
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                },
            };
            return assertResponse(wrappedRes);
        });
    }
    /**
     * Creates a new directory under the specified share or parent directory if it does not already exists.
     * If the directory already exists, it is not modified.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param options -
     */
    async createIfNotExists(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-createIfNotExists", options, async (updatedOptions) => {
            try {
                const res = await this.create(updatedOptions);
                return {
                    succeeded: true,
                    ...res,
                };
            }
            catch (e) {
                if (e.details?.errorCode === "ResourceAlreadyExists") {
                    return {
                        succeeded: false,
                        ...e.response?.parsedHeaders,
                        _response: e.response,
                    };
                }
                throw e;
            }
        });
    }
    /**
     * Sets properties on the directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-directory-properties
     *
     * @param DirectoryProperties - Directory properties. If no values are provided,
     *                                            existing values will be preserved.
     */
    async setProperties(properties = {}) {
        properties = validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(properties);
        return tracingClient.withSpan("ShareDirectoryClient-setProperties", properties, async (updatedOptions) => {
            const rawResponse = await this.context.setProperties({
                ...updatedOptions,
                fileChangeOn: fileChangeTimeToString(updatedOptions.changeTime),
                fileCreatedOn: fileCreationTimeToString(updatedOptions.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(updatedOptions.lastWriteTime),
                fileAttributes: updatedOptions.fileAttributes
                    ? fileAttributesToString(updatedOptions.fileAttributes)
                    : undefined,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                },
            });
        });
    }
    /**
     * Creates a ShareDirectoryClient object for a sub directory.
     *
     * @param subDirectoryName - A subdirectory name
     * @returns The ShareDirectoryClient object for the given subdirectory name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleGetDirectoryClient
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const shareClient = serviceClient.getShareClient(shareName);
     * const directoryClient = shareClient.getDirectoryClient(directoryName);
     * await directoryClient.create();
     * ```
     */
    getDirectoryClient(subDirectoryName) {
        return new ShareDirectoryClient(appendToURLPath(this.url, EscapePath(subDirectoryName)), this.pipeline, this.shareClientConfig);
    }
    /**
     * Creates a new subdirectory under this directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Create operation.
     * @returns Directory create response data and the corresponding DirectoryClient instance.
     */
    async createSubdirectory(directoryName, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-createSubdirectory", options, async (updatedOptions) => {
            const directoryClient = this.getDirectoryClient(directoryName);
            const directoryCreateResponse = await directoryClient.create(updatedOptions);
            return {
                directoryClient,
                directoryCreateResponse,
            };
        });
    }
    /**
     * Removes the specified empty sub directory under this directory.
     * Note that the directory must be empty before it can be deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param directoryName -
     * @param options - Options to Directory Delete operation.
     * @returns Directory deletion response data.
     */
    async deleteSubdirectory(directoryName, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-deleteSubdirectory", options, async (updatedOptions) => {
            const directoryClient = this.getDirectoryClient(directoryName);
            return directoryClient.delete(updatedOptions);
        });
    }
    /**
     * Creates a new file or replaces a file under this directory. Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param fileName -
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns File creation response data and the corresponding file client.
     */
    async createFile(fileName, size, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-createFile", options, async (updatedOptions) => {
            const fileClient = this.getFileClient(fileName);
            const fileCreateResponse = await fileClient.create(size, updatedOptions);
            return {
                fileClient,
                fileCreateResponse,
            };
        });
    }
    /**
     * Removes the specified file under this directory from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param fileName - Name of the file to delete
     * @param options - Options to File Delete operation.
     * @returns File deletion response data.
     */
    async deleteFile(fileName, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-deleteFile", options, async (updatedOptions) => {
            const fileClient = this.getFileClient(fileName);
            return fileClient.delete(updatedOptions);
        });
    }
    /**
     * Creates a {@link ShareFileClient} object.
     *
     * @param fileName - A file name.
     * @returns A new ShareFileClient object for the given file name.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-subclients */
    getFileClient(fileName) {
        return new ShareFileClient(appendToURLPath(this.url, EscapePath(fileName)), this.pipeline, this.shareClientConfig);
    }
    /**
     * Returns true if the specified directory exists; false otherwise.
     *
     * NOTE: use this function with care since an existing directory might be deleted by other clients or
     * applications. Vice versa new directories might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    async exists(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-exists", options, async (updatedOptions) => {
            try {
                await this.getProperties({ ...updatedOptions, ...this.shareClientConfig });
                return true;
            }
            catch (e) {
                if (e.statusCode === 404) {
                    return false;
                }
                throw e;
            }
        });
    }
    /**
     * Returns all system properties for the specified directory, and can also be used to check the
     * existence of a directory. The data returned does not include the files in the directory or any
     * subdirectories.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-directory-properties
     *
     * @param options - Options to Directory Get Properties operation.
     * @returns Response data for the Directory Get Properties operation.
     */
    async getProperties(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-getProperties", options, async (updatedOptions) => {
            const rawResponse = await this.context.getProperties({
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    fileType: rawResponse.nfsFileType,
                },
            });
        });
    }
    /**
     * Removes the specified empty directory. Note that the directory must be empty before it can be
     * deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param options - Options to Directory Delete operation.
     * @returns Response data for the Directory Delete operation.
     */
    async delete(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-delete", options, async (updatedOptions) => {
            return assertResponse(await this.context.delete({ ...updatedOptions, ...this.shareClientConfig }));
        });
    }
    /**
     * Removes the specified empty directory if it exists. Note that the directory must be empty before it can be
     * deleted.
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-directory
     *
     * @param options -
     */
    async deleteIfExists(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-deleteIfExists", options, async (updatedOptions) => {
            try {
                const res = await this.delete(updatedOptions);
                return {
                    succeeded: true,
                    ...res,
                };
            }
            catch (e) {
                if (e.details?.errorCode === "ResourceNotFound" ||
                    e.details?.errorCode === "ParentNotFound") {
                    return {
                        succeeded: false,
                        ...e.response?.parsedHeaders,
                        _response: e.response,
                    };
                }
                throw e;
            }
        });
    }
    /**
     * Updates user defined metadata for the specified directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-directory-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed
     * @param options - Options to Directory Set Metadata operation.
     * @returns Response data for the Directory Set Metadata operation.
     */
    async setMetadata(metadata, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-setMetadata", options, async (updatedOptions) => {
            return assertResponse(await this.context.setMetadata({
                ...updatedOptions,
                metadata,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Returns an AsyncIterableIterator for {@link DirectoryListFilesAndDirectoriesSegmentResponse} objects
     *
     * @param marker - A string value that identifies the portion of
     *                          the list of files and directories to be returned with the next listing operation. The
     *                          operation returns the ContinuationToken value within the response body if the
     *                          listing operation did not return all files and directories remaining to be listed
     *                          with the current page. The ContinuationToken value can be used as the value for
     *                          the marker parameter in a subsequent call to request the next page of list
     *                          items. The marker value is opaque to the client.
     * @param options - Options to list files and directories operation.
     */
    async *iterateFilesAndDirectoriesSegments(marker, options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        let listFilesAndDirectoriesResponse;
        do {
            listFilesAndDirectoriesResponse = await this.listFilesAndDirectoriesSegment(marker, options);
            marker = listFilesAndDirectoriesResponse.continuationToken;
            yield await listFilesAndDirectoriesResponse;
        } while (marker);
    }
    /**
     * Returns an AsyncIterableIterator for file and directory items
     *
     * @param options - Options to list files and directories operation.
     */
    async *listFilesAndDirectoriesItems(options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        let marker;
        for await (const listFilesAndDirectoriesResponse of this.iterateFilesAndDirectoriesSegments(marker, options)) {
            for (const file of listFilesAndDirectoriesResponse.segment.fileItems) {
                yield { kind: "file", ...file };
            }
            for (const directory of listFilesAndDirectoriesResponse.segment.directoryItems) {
                yield { kind: "directory", ...directory };
            }
        }
    }
    /**
     * Returns an async iterable iterator to list all the files and directories
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the files and directories in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const item of directoryClient.listFilesAndDirectories()) {
     *   if (item.kind === "directory") {
     *     console.log(`${i} - directory\t: ${item.name}`);
     *   } else {
     *     console.log(`${i} - file\t: ${item.name}`);
     *   }
     *   i++;
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * const iter = directoryClient.listFilesAndDirectories();
     * let { value, done } = await iter.next();
     * while (!done) {
     *   if (value.kind === "directory") {
     *     console.log(`${i} - directory\t: ${value.name}`);
     *   } else {
     *     console.log(`${i} - file\t: ${value.name}`);
     *   }
     *   ({ value, done } = await iter.next());
     *   i++;
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const response of directoryClient
     *   .listFilesAndDirectories()
     *   .byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const item of response.segment.directoryItems) {
     *     console.log(`\tdirectory: ${item.name}`);
     *   }
     *   for (const item of response.segment.fileItems) {
     *     console.log(`\tfile: ${item.name}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListFilesAndDirectories_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let iterator = directoryClient.listFilesAndDirectories().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const item of response.segment.directoryItems) {
     *   console.log(`\tdirectory: ${item.name}`);
     * }
     *
     * for await (const item of response.segment.fileItems) {
     *   console.log(`\tfile: ${item.name}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = directoryClient
     *   .listFilesAndDirectories()
     *   .byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const item of response.segment.directoryItems) {
     *   console.log(`\tdirectory: ${item.name}`);
     * }
     *
     * for await (const item of response.segment.fileItems) {
     *   console.log(`\tfile: ${item.name}`);
     * }
     * ```
     *
     * @param options - Options to list files and directories operation.
     * @returns An asyncIterableIterator that supports paging.
     */
    listFilesAndDirectories(options = {}) {
        const include = [];
        if (options.includeTimestamps) {
            include.push("Timestamps");
        }
        if (options.includeEtag) {
            include.push("Etag");
        }
        if (options.includeAttributes) {
            include.push("Attributes");
        }
        if (options.includePermissionKey) {
            include.push("PermissionKey");
        }
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        const updatedOptions = {
            ...options,
            ...(include.length > 0 ? { include: include } : {}),
        };
        // AsyncIterableIterator to iterate over files and directories
        const iter = this.listFilesAndDirectoriesItems(updatedOptions);
        return {
            /**
             * The next method, part of the iteration protocol
             */
            async next() {
                return iter.next();
            },
            /**
             * The connection to the async iterator, part of the iteration protocol
             */
            [Symbol.asyncIterator]() {
                return this;
            },
            /**
             * Return an AsyncIterableIterator that works a page at a time
             */
            byPage: (settings = {}) => {
                return this.iterateFilesAndDirectoriesSegments(removeEmptyString(settings.continuationToken), {
                    maxResults: settings.maxPageSize,
                    ...updatedOptions,
                });
            },
        };
    }
    /**
     * Returns a list of files or directories under the specified share or directory. It lists the
     * contents only for a single level of the directory hierarchy.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-directories-and-files
     *
     * @param marker - A string value that identifies the portion of the list to be returned with the next list operation.
     * @param options - Options to Directory List Files and Directories Segment operation.
     * @returns Response data for the Directory List Files and Directories operation.
     */
    async listFilesAndDirectoriesSegment(marker, options = {}) {
        if (options.prefix === "") {
            options.prefix = undefined;
        }
        return tracingClient.withSpan("ShareDirectoryClient-listFilesAndDirectoriesSegment", options, async (updatedOptions) => {
            const rawResponse = assertResponse(await this.context.listFilesAndDirectoriesSegment({
                ...updatedOptions,
                marker,
                ...this.shareClientConfig,
            }));
            const wrappedResponse = {
                ...ConvertInternalResponseOfListFiles(rawResponse),
                _response: {
                    ...rawResponse._response,
                    parsedBody: ConvertInternalResponseOfListFiles(rawResponse._response.parsedBody),
                }, // _response is made non-enumerable
            };
            return wrappedResponse;
        });
    }
    /**
     * Returns an AsyncIterableIterator for {@link DirectoryListHandlesResponse}
     *
     * @param marker - A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options - Options to list handles operation.
     */
    async *iterateHandleSegments(marker, options = {}) {
        let listHandlesResponse;
        if (!!marker || marker === undefined) {
            do {
                listHandlesResponse = await this.listHandlesSegment(marker, options);
                marker = listHandlesResponse.continuationToken;
                yield await listHandlesResponse;
            } while (marker);
        }
    }
    /**
     * Returns an AsyncIterableIterator for handles
     *
     * @param options - Options to list handles operation.
     */
    async *listHandleItems(options = {}) {
        let marker;
        for await (const listHandlesResponse of this.iterateHandleSegments(marker, options)) {
            if (listHandlesResponse.handleList) {
                for (const handle of listHandlesResponse.handleList) {
                    yield handle;
                }
            }
        }
    }
    /**
     * Returns an async iterable iterator to list all the handles.
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the handles in pages.
     *
     * Example using `for await` syntax:
     *
     * ```ts snippet:ReadmeSampleListHandles
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * for await (const handle of directoryClient.listHandles()) {
     *   console.log(`Handle: ${handle.handleId}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```ts snippet:ReadmeSampleListHandles_Iterator
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const handleIter = directoryClient.listHandles();
     * let { value, done } = await handleIter.next();
     * while (!done) {
     *   console.log(`Handle: ${value.handleId}`);
     *   ({ value, done } = await handleIter.next());
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```ts snippet:ReadmeSampleListHandles_ByPage
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let i = 1;
     * for await (const response of directoryClient.listHandles().byPage({ maxPageSize: 20 })) {
     *   console.log(`Page ${i++}:`);
     *   for (const handle of response.handleList || []) {
     *     console.log(`\thandle: ${handle.handleId}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```ts snippet:ReadmeSampleListHandles_Continuation
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * let iterator = directoryClient.listHandles().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * for await (const handle of response.handleList || []) {
     *   console.log(`\thandle: ${handle.handleId}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     * iterator = directoryClient.listHandles().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * for await (const handle of response.handleList || []) {
     *   console.log(`\thandle: ${handle.handleId}`);
     * }
     * ```
     *
     * @param options - Options to list handles operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listHandles(options = {}) {
        // an AsyncIterableIterator to iterate over handles
        const iter = this.listHandleItems(options);
        return {
            /**
             * The next method, part of the iteration protocol
             */
            async next() {
                return iter.next();
            },
            /**
             * The connection to the async iterator, part of the iteration protocol
             */
            [Symbol.asyncIterator]() {
                return this;
            },
            /**
             * Return an AsyncIterableIterator that works a page at a time
             */
            byPage: (settings = {}) => {
                return this.iterateHandleSegments(removeEmptyString(settings.continuationToken), {
                    maxResults: settings.maxPageSize,
                    ...options,
                });
            },
        };
    }
    /**
     * Lists handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-handles
     *
     * @param marker - Optional. A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options -
     */
    async listHandlesSegment(marker, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-listHandlesSegment", options, async (updatedOptions) => {
            marker = marker === "" ? undefined : marker;
            const response = assertResponse(await this.context.listHandles({
                ...updatedOptions,
                marker,
                ...this.shareClientConfig,
            }));
            // TODO: Protocol layer issue that when handle list is in returned XML
            // response.handleList is an empty string
            if (response.handleList === "") {
                response.handleList = undefined;
            }
            const wrappedResponse = {
                ...ConvertInternalResponseOfListHandles(response),
                _response: {
                    ...response._response,
                    parsedBody: ConvertInternalResponseOfListHandles(response._response.parsedBody),
                },
            };
            return wrappedResponse;
        });
    }
    /**
     * Force close all handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param marker - Optional. A string value that identifies the position of handles that will
     *                          be closed with the next force close handles operation.
     *                          The operation returns a marker value within the response
     *                          body if there are more handles to close. The marker value
     *                          may then be used in a subsequent call to close the next set of handles.
     * @param options -
     */
    async forceCloseHandlesSegment(marker, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-forceCloseHandlesSegment", options, async (updatedOptions) => {
            if (marker === "") {
                marker = undefined;
            }
            const rawResponse = assertResponse(await this.context.forceCloseHandles("*", {
                ...updatedOptions,
                marker,
                ...this.shareClientConfig,
            }));
            return {
                ...rawResponse,
                closedHandlesCount: rawResponse.numberOfHandlesClosed ?? 0,
                closeFailureCount: rawResponse.numberOfHandlesFailedToClose ?? 0,
            };
        });
    }
    /**
     * Force close all handles for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param options -
     */
    async forceCloseAllHandles(options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-forceCloseAllHandles", options, async (updatedOptions) => {
            let handlesClosed = 0;
            let numberOfHandlesFailedToClose = 0;
            let marker = "";
            do {
                const response = await this.forceCloseHandlesSegment(marker, updatedOptions);
                marker = response.marker;
                if (response.closedHandlesCount) {
                    handlesClosed += response.closedHandlesCount;
                }
                if (response.closeFailureCount) {
                    numberOfHandlesFailedToClose += response.closeFailureCount;
                }
            } while (marker);
            return {
                closedHandlesCount: handlesClosed,
                closeFailureCount: numberOfHandlesFailedToClose,
            };
        });
    }
    /**
     * Force close a specific handle for a directory.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param aborter - Create a new Aborter instance with Aborter.none or Aborter.timeout(),
     *                          goto documents of Aborter for more examples about request cancellation
     * @param handleId - Specific handle ID, cannot be asterisk "*".
     *                          Use forceCloseHandlesSegment() to close all handles.
     * @param options -
     */
    async forceCloseHandle(handleId, options = {}) {
        return tracingClient.withSpan("ShareDirectoryClient-forceCloseHandle", options, async (updatedOptions) => {
            if (handleId === "*") {
                throw new RangeError(`Parameter handleID should be a specified handle ID. Use forceCloseHandlesSegment() to close all handles.`);
            }
            const rawResponse = await this.context.forceCloseHandles(handleId, {
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            const response = rawResponse;
            response.closedHandlesCount = rawResponse.numberOfHandlesClosed || 0;
            response.closeFailureCount = rawResponse.numberOfHandlesFailedToClose || 0;
            return response;
        });
    }
    /**
     * Renames a directory.
     * This API only supports renaming a directory in the same share.
     *
     * @param destinationPath - Specifies the destination path to rename to. The path will be encoded to put into a URL to specify the destination.
     * @param options - Options for the renaming operation.
     * @returns Response data for the file renaming operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleRenameDirectory
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const destinationPath = "<destination path>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * await directoryClient.rename(destinationPath);
     * ```
     */
    async rename(destinationPath, options = {}) {
        const split = destinationPath.split("?");
        let destinationUrl;
        if (split.length === 2) {
            const pathOnly = EscapePath(split[0]);
            const renameDestination = `/${this.shareName}/${pathOnly}`;
            destinationUrl = setURLPath(this.url, renameDestination);
            destinationUrl = setURLQueries(destinationUrl, split[1]);
        }
        else if (split.length === 1) {
            const pathOnly = EscapePath(destinationPath);
            const renameDestination = `/${this.shareName}/${pathOnly}`;
            destinationUrl = setURLPath(this.url, renameDestination);
        }
        else {
            throw new RangeError("Destination path should not contain more than one query string");
        }
        const destDirectory = new ShareDirectoryClient(destinationUrl, this.pipeline, this.shareClientConfig);
        return tracingClient.withSpan("ShareDirectoryClient-rename", options, async (updatedOptions) => {
            const response = assertResponse(await destDirectory.context.rename(this.url, {
                ...updatedOptions,
                sourceLeaseAccessConditions: updatedOptions.sourceLeaseAccessConditions
                    ? {
                        sourceLeaseId: updatedOptions.sourceLeaseAccessConditions.leaseId,
                    }
                    : undefined,
                destinationLeaseAccessConditions: updatedOptions.destinationLeaseAccessConditions
                    ? {
                        destinationLeaseId: updatedOptions.destinationLeaseAccessConditions.leaseId,
                    }
                    : undefined,
                ...this.shareClientConfig,
            }));
            return {
                destinationDirectoryClient: destDirectory,
                directoryRenameResponse: response,
            };
        });
    }
}
/**
 * A ShareFileClient represents a URL to an Azure Storage file.
 */
export class ShareFileClient extends StorageClient {
    /**
     * context provided by protocol layer.
     */
    context;
    _shareName;
    _path;
    _name;
    shareClientConfig;
    /**
     * The share name corresponding to this file client
     */
    get shareName() {
        return this._shareName;
    }
    /**
     * The full path of the file
     */
    get path() {
        return this._path;
    }
    /**
     * The name of the file
     */
    get name() {
        return this._name;
    }
    constructor(url, credentialOrPipeline, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options) {
        let pipeline;
        if (isPipelineLike(credentialOrPipeline)) {
            pipeline = credentialOrPipeline;
        }
        else if (credentialOrPipeline instanceof Credential ||
            isTokenCredential(credentialOrPipeline)) {
            pipeline = newPipeline(credentialOrPipeline, options);
        }
        else {
            // The second parameter is undefined. Use anonymous credential.
            pipeline = newPipeline(new AnonymousCredential(), options);
        }
        super(url, pipeline);
        ({
            baseName: this._name,
            shareName: this._shareName,
            path: this._path,
        } = getShareNameAndPathFromUrl(this.url));
        this.shareClientConfig = options;
        this.context = this.storageClientContext.file;
    }
    /**
     * Creates a new ShareFileClient object identical to the source but with the specified share snapshot timestamp.
     * Provide "" will remove the snapshot and return a URL to the base ShareFileClient.
     *
     * @param shareSnapshot - The share snapshot timestamp.
     * @returns A new ShareFileClient object identical to the source but with the specified share snapshot timestamp.
     */
    withShareSnapshot(shareSnapshot) {
        return new ShareFileClient(setURLParameter(this.url, URLConstants.Parameters.SHARE_SNAPSHOT, shareSnapshot.length === 0 ? undefined : shareSnapshot), this.pipeline, this.shareClientConfig);
    }
    /**
     * Creates a new file or replaces a file. Note it only initializes the file with no content.
     * @see https://learn.microsoft.com/rest/api/storageservices/create-file
     *
     * @param size - Specifies the maximum size in bytes for the file, up to 4 TB.
     * @param options - Options to File Create operation.
     * @returns Response data for the File Create  operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    async create(size, options = {}) {
        if (size < 0 || size > FILE_MAX_SIZE_BYTES) {
            throw new RangeError(`File size must >= 0 and < ${FILE_MAX_SIZE_BYTES}.`);
        }
        options = validateAndSetDefaultsForFileAndDirectoryCreateCommonOptions(options);
        options.fileHttpHeaders = options.fileHttpHeaders || {};
        return tracingClient.withSpan("ShareFileClient-create", options, async (updatedOptions) => {
            const rawResponse = await this.context.create(size, {
                ...updatedOptions,
                fileChangeOn: fileChangeTimeToString(updatedOptions.changeTime),
                fileCreatedOn: fileCreationTimeToString(updatedOptions.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(updatedOptions.lastWriteTime),
                fileAttributes: updatedOptions.fileAttributes
                    ? fileAttributesToString(updatedOptions.fileAttributes)
                    : undefined,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                nfsFileType: updatedOptions.posixProperties?.fileType,
                ...this.shareClientConfig,
            });
            const wrappedRes = {
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    fileType: rawResponse.nfsFileType,
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                },
            };
            return assertResponse(wrappedRes);
        });
    }
    /**
     * Reads or downloads a file from the system, including its metadata and properties.
     *
     * * In Node.js, data returns in a Readable stream `readableStreamBody`
     * * In browsers, data returns in a promise `contentAsBlob`
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file
     *
     * @param offset - From which position of the file to download, greater than or equal to 0
     * @param count - How much data to be downloaded, greater than 0. Will download to the end when undefined
     * @param options - Options to File Download operation.
     * @returns Response data for the File Download operation.
     *
     * Example usage (Node.js):
     *
     * ```ts snippet:ReadmeSampleDownloadFileAndConvertToString_Node
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const fileName = "<file name>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .rootDirectoryClient.getFileClient(fileName);
     *
     * // Get file content from position 0 to the end
     * // In Node.js, get downloaded data by accessing downloadFileResponse.readableStreamBody
     * const downloadFileResponse = await fileClient.download();
     * if (downloadFileResponse.readableStreamBody) {
     *   const buffer = await streamToBuffer(downloadFileResponse.readableStreamBody);
     *   console.log(`Downloaded file content: ${buffer.toString()}`);
     * }
     *
     * // [Node.js only] A helper method used to read a Node.js readable stream into a Buffer
     * async function streamToBuffer(readableStream: NodeJS.ReadableStream): Promise<Buffer> {
     *   return new Promise((resolve, reject) => {
     *     const chunks: Buffer[] = [];
     *     readableStream.on("data", (data) => {
     *       chunks.push(data instanceof Buffer ? data : Buffer.from(data));
     *     });
     *     readableStream.on("end", () => {
     *       resolve(Buffer.concat(chunks));
     *     });
     *     readableStream.on("error", reject);
     *   });
     * }
     * ```
     *
     * Example usage (browsers):
     *
     * ```ts snippet:ReadmeSampleDownloadFileAndConvertToString_Browser
     * import { ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account name>";
     * const sas = "<service Shared Access Signature Token>";
     *
     * const serviceClient = new ShareServiceClient(`https://${account}.file.core.windows.net?${sas}`);
     *
     * const shareName = "<share name>";
     * const fileName = "<file name>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .rootDirectoryClient.getFileClient(fileName);
     *
     * // Get file content from position 0 to the end
     * // In browsers, get downloaded data by accessing downloadFileResponse.blobBody
     * const downloadFileResponse = await fileClient.download(0);
     * if (downloadFileResponse.blobBody) {
     *   console.log(`Downloaded file content: ${(await downloadFileResponse.blobBody).text()}`);
     * }
     * ```
     */
    async download(offset = 0, count, options = {}) {
        return tracingClient.withSpan("ShareFileClient-download", options, async (updatedOptions) => {
            if (updatedOptions.rangeGetContentMD5 && offset === 0 && count === undefined) {
                throw new RangeError(`rangeGetContentMD5 only works with partial data downloading`);
            }
            const downloadFullFile = offset === 0 && !count;
            const rawResponse = await this.context.download({
                ...updatedOptions,
                requestOptions: {
                    onDownloadProgress: isNodeLike ? undefined : updatedOptions.onProgress, // for Node.js, progress is reported by RetriableReadableStream
                },
                range: downloadFullFile ? undefined : rangeToString({ offset, count }),
                ...this.shareClientConfig,
            });
            const res = assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
            // Return browser response immediately
            if (!isNodeLike) {
                return res;
            }
            // We support retrying when download stream unexpected ends in Node.js runtime
            // Following code shouldn't be bundled into browser build, however some
            // bundlers may try to bundle following code and "FileReadResponse.ts".
            // In this case, "FileDownloadResponse.browser.ts" will be used as a shim of "FileDownloadResponse.ts"
            // The config is in package.json "browser" field
            if (updatedOptions.maxRetryRequests === undefined || updatedOptions.maxRetryRequests < 0) {
                // TODO: Default value or make it a required parameter?
                updatedOptions.maxRetryRequests = DEFAULT_MAX_DOWNLOAD_RETRY_REQUESTS;
            }
            if (res.contentLength === undefined) {
                throw new RangeError(`File download response doesn't contain valid content length header`);
            }
            return new FileDownloadResponse(res, async (start) => {
                const updatedDownloadOptions = {
                    range: rangeToString({
                        count: offset + res.contentLength - start,
                        offset: start,
                    }),
                };
                // Debug purpose only
                // console.log(
                //   `Read from internal stream, range: ${
                //     chunkDownloadOptions.range
                //   }, options: ${JSON.stringify(chunkDownloadOptions)}`
                // );
                const downloadRes = await this.context.download({
                    ...updatedOptions,
                    ...updatedDownloadOptions,
                    ...this.shareClientConfig, // TODO: confirm whether this is needed
                });
                if (!(downloadRes.etag === res.etag)) {
                    throw new Error("File has been modified concurrently");
                }
                return downloadRes.readableStreamBody;
            }, offset, res.contentLength, {
                maxRetryRequests: updatedOptions.maxRetryRequests,
                onProgress: updatedOptions.onProgress,
            });
        });
    }
    /**
     * Returns true if the specified file exists; false otherwise.
     *
     * NOTE: use this function with care since an existing file might be deleted by other clients or
     * applications. Vice versa new files might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    async exists(options = {}) {
        return tracingClient.withSpan("ShareFileClient-exists", options, async (updatedOptions) => {
            try {
                await this.getProperties(updatedOptions);
                return true;
            }
            catch (e) {
                if (e.statusCode === 404) {
                    return false;
                }
                throw e;
            }
        });
    }
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the file. It does not return the content of the file.
     * @see https://learn.microsoft.com/rest/api/storageservices/get-file-properties
     *
     * @param options - Options to File Get Properties operation.
     * @returns Response data for the File Get Properties operation.
     */
    async getProperties(options = {}) {
        return tracingClient.withSpan("ShareFileClient-getProperties", options, async (updatedOptions) => {
            const rawResponse = await this.context.getProperties({
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    fileType: rawResponse.nfsFileType,
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
        });
    }
    /**
     * Sets properties on the file.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param properties - File properties. For file HTTP headers(e.g. Content-Type),
     *                                       if no values are provided, existing HTTP headers will be removed.
     *                                       For other file properties(e.g. fileAttributes), if no values are provided,
     *                                       existing values will be preserved.
     */
    async setProperties(properties = {}) {
        properties = validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(properties);
        properties.fileHttpHeaders = properties.fileHttpHeaders || {};
        return tracingClient.withSpan("ShareFileClient-setProperties", properties, async (updatedOptions) => {
            const rawResponse = await this.context.setHttpHeaders({
                ...updatedOptions,
                fileChangeOn: fileChangeTimeToString(updatedOptions.changeTime),
                fileCreatedOn: fileCreationTimeToString(updatedOptions.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(updatedOptions.lastWriteTime),
                fileAttributes: updatedOptions.fileAttributes
                    ? fileAttributesToString(updatedOptions.fileAttributes)
                    : undefined,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
        });
    }
    /**
     * Removes the file from the storage account.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param options - Options to File Delete operation.
     * @returns Response data for the File Delete operation.
     */
    async delete(options = {}) {
        return tracingClient.withSpan("ShareFileClient-delete", options, async (updatedOptions) => {
            return assertResponse(await this.context.delete({ ...updatedOptions, ...this.shareClientConfig }));
        });
    }
    /**
     * Removes the file from the storage account if it exists.
     * When a file is successfully deleted, it is immediately removed from the storage
     * account's index and is no longer accessible to clients. The file's data is later
     * removed from the service during garbage collection.
     *
     * Delete File will fail with status code 409 (Conflict) and error code SharingViolation
     * if the file is open on an SMB client.
     *
     * Delete File is not supported on a share snapshot, which is a read-only copy of
     * a share. An attempt to perform this operation on a share snapshot will fail with 400 (InvalidQueryParameterValue)
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/delete-file2
     *
     * @param options -
     */
    async deleteIfExists(options = {}) {
        return tracingClient.withSpan("ShareFileClient-deleteIfExists", options, async (updatedOptions) => {
            try {
                const res = await this.delete(updatedOptions);
                return {
                    succeeded: true,
                    ...res,
                };
            }
            catch (e) {
                if (e.details?.errorCode === "ResourceNotFound" ||
                    e.details?.errorCode === "ParentNotFound") {
                    return {
                        succeeded: false,
                        ...e.response?.parsedHeaders,
                        _response: e.response,
                    };
                }
                throw e;
            }
        });
    }
    /**
     * Sets HTTP headers on the file.
     *
     * If no option provided, or no value provided for the file HTTP headers in the options,
     * these file HTTP headers without a value will be cleared.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param FileHttpHeaders - File HTTP headers like Content-Type.
     *                                             Provide undefined will remove existing HTTP headers.
     * @param options - Options to File Set HTTP Headers operation.
     * @returns Response data for the File Set HTTP Headers operation.
     */
    async setHttpHeaders(fileHttpHeaders = {}, options = {}) {
        // FileAttributes, filePermission, createTime, lastWriteTime will all be preserved
        options = validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(options);
        return tracingClient.withSpan("ShareFileClient-setHTTPHeaders", options, async (updatedOptions) => {
            const rawResponse = await this.context.setHttpHeaders({
                ...updatedOptions,
                fileHttpHeaders,
                fileCreatedOn: fileCreationTimeToString(updatedOptions.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(updatedOptions.lastWriteTime),
                fileChangeOn: fileChangeTimeToString(updatedOptions.changeTime),
                fileAttributes: updatedOptions.fileAttributes
                    ? fileAttributesToString(updatedOptions.fileAttributes)
                    : undefined,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
        });
    }
    /**
     * Resize file.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-properties
     *
     * @param length - Resizes a file to the specified size in bytes.
     *                        If the specified byte value is less than the current size of the file,
     *                        then all ranges above the specified byte value are cleared.
     * @param options - Options to File Resize operation.
     * @returns Response data for the File Set HTTP Headers operation.
     */
    async resize(length, options = {}) {
        if (length < 0) {
            throw new RangeError(`Size cannot less than 0 when resizing file.`);
        }
        // FileAttributes, filePermission, createTime, lastWriteTime will all be preserved.
        options = validateAndSetDefaultsForFileAndDirectorySetPropertiesCommonOptions(options);
        return tracingClient.withSpan("ShareFileClient-resize", options, async (updatedOptions) => {
            const rawResponse = await this.context.setHttpHeaders({
                ...updatedOptions,
                fileContentLength: length,
                fileChangeOn: fileChangeTimeToString(options.changeTime),
                fileCreatedOn: fileCreationTimeToString(options.creationTime),
                fileLastWriteOn: fileLastWriteTimeToString(options.lastWriteTime),
                fileAttributes: fileAttributesToString(updatedOptions.fileAttributes),
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(options.posixProperties?.fileMode),
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
        });
    }
    /**
     * Updates user-defined metadata for the specified file.
     *
     * If no metadata defined in the option parameter, the file
     * metadata will be removed.
     * @see https://learn.microsoft.com/rest/api/storageservices/set-file-metadata
     *
     * @param metadata - If no metadata provided, all existing directory metadata will be removed
     * @param options - Options to File Set Metadata operation.
     * @returns Response data for the File Set Metadata operation.
     */
    async setMetadata(metadata = {}, options = {}) {
        return tracingClient.withSpan("ShareFileClient-setMetadata", options, async (updatedOptions) => {
            return assertResponse(await this.context.setMetadata({
                ...updatedOptions,
                metadata,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Upload a range of bytes to a file. This operation can only be called on an existing file.
     * It won't change the size, properties or metadata of the file.
     * Both the start and count of the range must be specified. The range can be up to 4 MB in size.
     *
     * @param body - Blob, string, ArrayBuffer, ArrayBufferView or a function
     *                               which returns a new Readable stream whose offset is from data source beginning.
     * @param offset - Offset position of the destination Azure File to upload.
     * @param contentLength - Length of body in bytes. Use Buffer.byteLength() to calculate body length for a
     *                               string including non non-Base64/Hex-encoded characters.
     * @param options - Options to File Upload Range operation.
     * @returns Response data for the File Upload Range operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleCreateFileAndUpload
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const directoryClient = serviceClient.getShareClient(shareName).getDirectoryClient(directoryName);
     *
     * const content = "Hello World!";
     * const fileName = `newdirectory${+new Date()}`;
     * const fileClient = directoryClient.getFileClient(fileName);
     * await fileClient.create(content.length);
     * console.log(`Create file ${fileName} successfully`);
     *
     * // Upload file range
     * await fileClient.uploadRange(content, 0, content.length);
     * console.log(`Upload file range "${content}" to ${fileName} successfully`);
     * ```
     */
    async uploadRange(body, offset, contentLength, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadRange", options, async (updatedOptions) => {
            if (offset < 0) {
                throw new RangeError(`offset must be >= 0`);
            }
            if (contentLength <= 0 || contentLength > FILE_RANGE_MAX_SIZE_BYTES) {
                throw new RangeError(`contentLength must be > 0 and <= ${FILE_RANGE_MAX_SIZE_BYTES} bytes`);
            }
            if (contentLength > FILE_RANGE_MAX_SIZE_BYTES) {
                throw new RangeError(`offset must be < ${FILE_RANGE_MAX_SIZE_BYTES} bytes`);
            }
            return assertResponse(await this.context.uploadRange(rangeToString({ count: contentLength, offset }), "update", contentLength, {
                ...updatedOptions,
                requestOptions: {
                    onUploadProgress: updatedOptions.onProgress,
                },
                body,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Upload a range of bytes to a file where the contents are read from a another file's URL.
     * The range can be up to 4 MB in size.
     *
     * @param sourceURL - Specify a URL to the copy source, Shared Access Signature(SAS) maybe needed for authentication.
     * @param sourceOffset - The source offset to copy from. Pass 0 to copy from the beginning of source file.
     * @param destOffset - Offset of destination file.
     * @param count - Number of bytes to be uploaded from source file.
     * @param options - Options to configure File - Upload Range from URL operation.
     */
    async uploadRangeFromURL(sourceURL, sourceOffset, destOffset, count, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadRangeFromURL", options, async (updatedOptions) => {
            if (sourceOffset < 0 || destOffset < 0) {
                throw new RangeError(`sourceOffset and destOffset must be >= 0`);
            }
            if (count <= 0 || count > FILE_RANGE_MAX_SIZE_BYTES) {
                throw new RangeError(`count must be > 0 and <= ${FILE_RANGE_MAX_SIZE_BYTES} bytes`);
            }
            return assertResponse(await this.context.uploadRangeFromURL(rangeToString({ offset: destOffset, count }), sourceURL, 0, {
                ...updatedOptions,
                sourceRange: rangeToString({ offset: sourceOffset, count }),
                sourceModifiedAccessConditions: updatedOptions.sourceConditions,
                copySourceAuthorization: httpAuthorizationToString(updatedOptions.sourceAuthorization),
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Clears the specified range and
     * releases the space used in storage for that range.
     *
     * @param offset -
     * @param contentLength -
     * @param options - Options to File Clear Range operation.
     */
    async clearRange(offset, contentLength, options = {}) {
        return tracingClient.withSpan("ShareFileClient-clearRange", options, async (updatedOptions) => {
            if (offset < 0 || contentLength <= 0) {
                throw new RangeError(`offset must >= 0 and contentLength must be > 0`);
            }
            return assertResponse(await this.context.uploadRange(rangeToString({ count: contentLength, offset }), "clear", 0, { ...updatedOptions, ...this.shareClientConfig }));
        });
    }
    /**
     * Returns the list of valid ranges for a file.
     *
     * @param options - Options to File Get range List operation.
     */
    async getRangeList(options = {}) {
        return tracingClient.withSpan("ShareFileClient-getRangeList", options, async (updatedOptions) => {
            const originalResponse = assertResponse(await this.context.getRangeList({
                ...updatedOptions,
                range: updatedOptions.range ? rangeToString(updatedOptions.range) : undefined,
                ...this.shareClientConfig,
            }));
            // Only returns ranges, ignoring clearRanges.
            const parsedBody = originalResponse._response.parsedBody.ranges
                ? originalResponse._response.parsedBody.ranges
                : [];
            return {
                ...originalResponse,
                _response: { ...originalResponse._response, parsedBody },
                rangeList: originalResponse.ranges ? originalResponse.ranges : [],
            };
        });
    }
    /**
     * Returns the list of ranges that differ between a previous share snapshot and this file.
     *
     * @param prevShareSnapshot - The previous snapshot parameter is an opaque DateTime value that specifies the previous share snapshot to compare with.
     * @param options -
     */
    async getRangeListDiff(prevShareSnapshot, options = {}) {
        return tracingClient.withSpan("ShareFileClient-getRangeListDiff", options, async (updatedOptions) => {
            return assertResponse(await this.context.getRangeList({
                ...updatedOptions,
                prevsharesnapshot: prevShareSnapshot,
                supportRename: options.includeRenames,
                range: updatedOptions.range ? rangeToString(updatedOptions.range) : undefined,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Copies a blob or file to a destination file within the storage account.
     *
     * @param copySource - Specifies the URL of the source file or blob, up to 2 KB in length.
     * To copy a file to another file within the same storage account, you may use Shared Key to
     * authenticate the source file. If you are copying a file from another storage account, or if you
     * are copying a blob from the same storage account or another storage account, then you must
     * authenticate the source file or blob using a shared access signature. If the source is a public
     * blob, no authentication is required to perform the copy operation. A file in a share snapshot
     * can also be specified as a copy source.
     * @param options - Options to File Start Copy operation.
     */
    async startCopyFromURL(copySource, options = {}) {
        return tracingClient.withSpan("ShareFileClient-startCopyFromURL", options, async (updatedOptions) => {
            return assertResponse(await this.context.startCopy(copySource, {
                ...updatedOptions,
                ...this.shareClientConfig,
                owner: updatedOptions.posixProperties?.owner,
                group: updatedOptions.posixProperties?.group,
                fileMode: toOctalFileMode(updatedOptions.posixProperties?.fileMode),
                fileModeCopyMode: updatedOptions.fileModeCopyMode,
                fileOwnerCopyMode: updatedOptions.fileOwnerCopyMode,
            }));
        });
    }
    /**
     * Aborts a pending Copy File operation, and leaves a destination file with zero length and full
     * metadata.
     * @see https://learn.microsoft.com/rest/api/storageservices/abort-copy-file
     *
     * @param copyId - Id of the Copy File operation to abort.
     * @param options - Options to File Abort Copy From URL operation.
     */
    async abortCopyFromURL(copyId, options = {}) {
        return tracingClient.withSpan("ShareFileClient-abortCopyFromURL", options, async (updatedOptions) => {
            return assertResponse(await this.context.abortCopy(copyId, { ...updatedOptions, ...this.shareClientConfig }));
        });
    }
    // High Level functions
    /**
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a Buffer(Node)/Blob/ArrayBuffer/ArrayBufferView to it.
     *
     * @param data - Buffer(Node), Blob, ArrayBuffer or ArrayBufferView
     * @param options -
     */
    async uploadData(data, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadData", options, async (updatedOptions) => {
            if (isNodeLike) {
                let buffer;
                if (data instanceof Buffer) {
                    buffer = data;
                }
                else if (data instanceof ArrayBuffer) {
                    buffer = Buffer.from(data);
                }
                else {
                    data = data;
                    buffer = Buffer.from(data.buffer, data.byteOffset, data.byteLength);
                }
                return this.uploadSeekableInternal((offset, size) => buffer.slice(offset, offset + size), buffer.byteLength, updatedOptions);
            }
            else {
                const browserBlob = new Blob([data]);
                return this.uploadSeekableInternal((offset, size) => browserBlob.slice(offset, offset + size), browserBlob.size, updatedOptions);
            }
        });
    }
    /**
     * ONLY AVAILABLE IN BROWSERS.
     *
     * Uploads a browser Blob object to an Azure file. Requires a blobFactory as the data source,
     * which need to return a Blob object with the offset and size provided.
     *
     * @param blobFactory -
     * @param size -
     * @param options -
     */
    async uploadSeekableBlob(blobFactory, size, options = {}) {
        return tracingClient.withSpan("ShareFileClient-UploadSeekableBlob", options, async (updatedOptions) => {
            return this.uploadSeekableInternal(blobFactory, size, updatedOptions);
        });
    }
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a local file to it.
     *
     * @param filePath - Full path of local file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    async uploadFile(filePath, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadFile", options, async (updatedOptions) => {
            const size = (await fsStat(filePath)).size;
            return this.uploadSeekableInternal((offset, count) => {
                return () => fsCreateReadStream(filePath, {
                    autoClose: true,
                    end: count ? offset + count - 1 : Infinity,
                    start: offset,
                });
            }, size, updatedOptions);
        });
    }
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Accepts a Node.js Readable stream factory, and uploads in blocks to an Azure File.
     * The Readable stream factory must returns a Node.js Readable stream starting from the offset defined. The offset
     * is the offset in the Azure file to be uploaded.
     *
     * @param streamFactory - Returns a Node.js Readable stream starting
     *                                                                  from the offset defined
     * @param size - Size of the Azure file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    async uploadResetableStream(streamFactory, size, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadResetableStream", options, async (updatedOptions) => {
            return this.uploadSeekableInternal((offset, count) => {
                return () => streamFactory(offset, count);
            }, size, updatedOptions);
        });
    }
    /**
     *
     * @param bodyFactory -
     * @param size - Size of the Azure file
     * @param fileClient - ShareFileClient
     * @param options -
     */
    async uploadSeekableInternal(bodyFactory, size, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadSeekableInternal", options, async (updatedOptions) => {
            if (!options.rangeSize) {
                options.rangeSize = FILE_RANGE_MAX_SIZE_BYTES;
            }
            if (options.rangeSize < 0 || options.rangeSize > FILE_RANGE_MAX_SIZE_BYTES) {
                throw new RangeError(`options.rangeSize must be > 0 and <= ${FILE_RANGE_MAX_SIZE_BYTES}`);
            }
            if (!options.fileHttpHeaders) {
                options.fileHttpHeaders = {};
            }
            if (!options.concurrency) {
                options.concurrency = DEFAULT_HIGH_LEVEL_CONCURRENCY;
            }
            if (options.concurrency < 0) {
                throw new RangeError(`options.concurrency cannot less than 0.`);
            }
            // Create the file
            await this.create(size, {
                abortSignal: options.abortSignal,
                fileHttpHeaders: options.fileHttpHeaders,
                metadata: options.metadata,
                leaseAccessConditions: options.leaseAccessConditions,
                tracingOptions: updatedOptions.tracingOptions,
            });
            const numBlocks = Math.floor((size - 1) / options.rangeSize) + 1;
            let transferProgress = 0;
            const batch = new Batch(options.concurrency);
            for (let i = 0; i < numBlocks; i++) {
                batch.addOperation(async () => {
                    const start = options.rangeSize * i;
                    const end = i === numBlocks - 1 ? size : start + options.rangeSize;
                    const contentLength = end - start;
                    await this.uploadRange(bodyFactory(start, contentLength), start, contentLength, {
                        abortSignal: options.abortSignal,
                        leaseAccessConditions: options.leaseAccessConditions,
                        tracingOptions: updatedOptions.tracingOptions,
                    });
                    // Update progress after block is successfully uploaded to server, in case of block trying
                    transferProgress += contentLength;
                    if (options.onProgress) {
                        options.onProgress({ loadedBytes: transferProgress });
                    }
                });
            }
            return batch.do();
        });
    }
    async downloadToBuffer(bufferOrOffset, offsetOrCount, countOrOptions, optOptions = {}) {
        let buffer = undefined;
        let offset;
        let count;
        let options = optOptions;
        if (bufferOrOffset instanceof Buffer) {
            buffer = bufferOrOffset;
            offset = offsetOrCount || 0;
            count = typeof countOrOptions === "number" ? countOrOptions : 0;
        }
        else {
            offset = typeof bufferOrOffset === "number" ? bufferOrOffset : 0;
            count = typeof offsetOrCount === "number" ? offsetOrCount : 0;
            options = countOrOptions || {};
        }
        return tracingClient.withSpan("ShareFileClient-downloadToBuffer", options, async (updatedOptions) => {
            if (!options.rangeSize) {
                options.rangeSize = FILE_RANGE_MAX_SIZE_BYTES;
            }
            if (options.rangeSize < 0) {
                throw new RangeError("rangeSize option must be > 0");
            }
            if (offset < 0) {
                throw new RangeError("offset option must be >= 0");
            }
            if (count && count <= 0) {
                throw new RangeError("count option must be > 0");
            }
            if (!options.concurrency) {
                options.concurrency = DEFAULT_HIGH_LEVEL_CONCURRENCY;
            }
            if (options.concurrency < 0) {
                throw new RangeError(`options.concurrency cannot less than 0.`);
            }
            // Customer doesn't specify length, get it
            if (!count) {
                const response = await this.getProperties({
                    abortSignal: options.abortSignal,
                    leaseAccessConditions: options.leaseAccessConditions,
                    tracingOptions: updatedOptions.tracingOptions,
                });
                count = response.contentLength - offset;
                if (count < 0) {
                    throw new RangeError(`offset ${offset} shouldn't be larger than file size ${response.contentLength}`);
                }
            }
            if (!buffer) {
                try {
                    buffer = Buffer.alloc(count);
                }
                catch (error) {
                    throw new Error(`Unable to allocate a buffer of size: ${count} bytes. Please try passing your own Buffer to ` +
                        'the "downloadToBuffer method or try using other methods like "download" or "downloadToFile".' +
                        `\t ${error.message}`);
                }
            }
            if (buffer.length < count) {
                throw new RangeError(`The buffer's size should be equal to or larger than the request count of bytes: ${count}`);
            }
            let transferProgress = 0;
            const batch = new Batch(options.concurrency);
            for (let off = offset; off < offset + count; off = off + options.rangeSize) {
                batch.addOperation(async () => {
                    // Exclusive chunk end position
                    let chunkEnd = offset + count;
                    if (off + options.rangeSize < chunkEnd) {
                        chunkEnd = off + options.rangeSize;
                    }
                    const response = await this.download(off, chunkEnd - off, {
                        abortSignal: options.abortSignal,
                        maxRetryRequests: options.maxRetryRequestsPerRange,
                        leaseAccessConditions: options.leaseAccessConditions,
                        tracingOptions: updatedOptions.tracingOptions,
                    });
                    const stream = response.readableStreamBody;
                    await streamToBuffer(stream, buffer, off - offset, chunkEnd - offset);
                    // Update progress after block is downloaded, in case of block trying
                    // Could provide finer grained progress updating inside HTTP requests,
                    // only if convenience layer download try is enabled
                    transferProgress += chunkEnd - off;
                    if (options.onProgress) {
                        options.onProgress({ loadedBytes: transferProgress });
                    }
                });
            }
            await batch.do();
            return buffer;
        });
    }
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Creates a new Azure File or replaces an existing Azure File, and then uploads a Node.js Readable stream into it.
     * This method will try to create an Azure File, then starts uploading chunk by chunk.
     * Size of chunk is defined by `bufferSize` parameter.
     * Please make sure potential size of stream doesn't exceed file size.
     *
     * PERFORMANCE IMPROVEMENT TIPS:
     * * Input stream highWaterMark is better to set a same value with bufferSize
     *   parameter, which will avoid Buffer.concat() operations.
     *
     * @param stream - Node.js Readable stream. Must be less or equal than file size.
     * @param size - Size of file to be created. Maximum size allowed is 4 TB.
     *                      If this value is larger than stream size, there will be empty bytes in file tail.
     * @param bufferSize - Size of every buffer allocated in bytes, also the chunk/range size during
     *                            the uploaded file. Size must be greater than 0 and lesser than or equal to 4 * 1024 * 1024 (4MB)
     * @param maxBuffers - Max buffers will allocate during uploading, positive correlation
     *                            with max uploading concurrency
     * @param options -
     */
    async uploadStream(stream, size, bufferSize, maxBuffers, options = {}) {
        return tracingClient.withSpan("ShareFileClient-uploadStream", options, async (updatedOptions) => {
            if (!options.fileHttpHeaders) {
                options.fileHttpHeaders = {};
            }
            if (bufferSize <= 0 || bufferSize > FILE_RANGE_MAX_SIZE_BYTES) {
                throw new RangeError(`bufferSize must be > 0 and <= ${FILE_RANGE_MAX_SIZE_BYTES}`);
            }
            if (maxBuffers < 0) {
                throw new RangeError(`maxBuffers must be > 0.`);
            }
            // Create the file
            await this.create(size, {
                abortSignal: options.abortSignal,
                fileHttpHeaders: options.fileHttpHeaders,
                metadata: options.metadata,
                leaseAccessConditions: options.leaseAccessConditions,
                tracingOptions: updatedOptions.tracingOptions,
            });
            let transferProgress = 0;
            const scheduler = new BufferScheduler(stream, bufferSize, maxBuffers, async (buffer, offset) => {
                if (transferProgress + buffer.length > size) {
                    throw new RangeError(`Stream size is larger than file size ${size} bytes, uploading failed. ` +
                        `Please make sure stream length is less or equal than file size.`);
                }
                await this.uploadRange(buffer, offset, buffer.length, {
                    abortSignal: options.abortSignal,
                    leaseAccessConditions: options.leaseAccessConditions,
                    tracingOptions: updatedOptions.tracingOptions,
                });
                // Update progress after block is successfully uploaded to server, in case of block trying
                transferProgress += buffer.length;
                if (options.onProgress) {
                    options.onProgress({ loadedBytes: transferProgress });
                }
            }, 
            // Concurrency should set a smaller value than maxBuffers, which is helpful to
            // reduce the possibility when a outgoing handler waits for stream data, in
            // this situation, outgoing handlers are blocked.
            // Outgoing queue shouldn't be empty.
            Math.ceil((maxBuffers / 4) * 3));
            return scheduler.do();
        });
    }
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Downloads an Azure Blob to a local file.
     * Fails if the the given file path already exits.
     * Offset and count are optional, pass 0 and undefined respectively to download the entire blob.
     *
     * @param filePath -
     * @param offset - From which position of the block blob to download.
     * @param count - How much data to be downloaded. Will download to the end when passing undefined.
     * @param options - Options to Blob download options.
     * @returns The response data for blob download operation,
     *                                                 but with readableStreamBody set to undefined since its
     *                                                 content is already read and written into a local file
     *                                                 at the specified path.
     */
    async downloadToFile(filePath, offset = 0, count, options = {}) {
        return tracingClient.withSpan("ShareFileClient-downloadToFile", options, async (updatedOptions) => {
            const response = await this.download(offset, count, updatedOptions);
            if (response.readableStreamBody) {
                await readStreamToLocalFile(response.readableStreamBody, filePath);
            }
            // The stream is no longer accessible so setting it to undefined.
            response.fileDownloadStream = undefined;
            return response;
        });
    }
    /**
     * Lists handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/list-handles
     *
     * @param marker - Optional. A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options -
     */
    async listHandlesSegment(marker, options = {}) {
        return tracingClient.withSpan("ShareFileClient-listHandlesSegment", options, async (updatedOptions) => {
            marker = marker === "" ? undefined : marker;
            const response = assertResponse(await this.context.listHandles({
                ...updatedOptions,
                ...this.shareClientConfig,
                marker,
            }));
            // TODO: Protocol layer issue that when handle list is in returned XML
            // response.handleList is an empty string
            if (response.handleList === "") {
                response.handleList = undefined;
            }
            const wrappedResponse = {
                ...ConvertInternalResponseOfListHandles(response),
                _response: {
                    ...response._response,
                    parsedBody: ConvertInternalResponseOfListHandles(response._response.parsedBody),
                },
            };
            return wrappedResponse;
        });
    }
    /**
     * Returns an AsyncIterableIterator for FileListHandlesResponse
     *
     * @param marker - A string value that identifies the portion of the list to be
     *                          returned with the next list handles operation. The operation returns a
     *                          marker value within the response body if the list returned was not complete.
     *                          The marker value may then be used in a subsequent call to request the next
     *                          set of list items.
     * @param options - Options to list handles operation.
     */
    async *iterateHandleSegments(marker, options = {}) {
        let listHandlesResponse;
        if (!!marker || marker === undefined) {
            do {
                listHandlesResponse = await this.listHandlesSegment(marker, options);
                marker = listHandlesResponse.continuationToken;
                yield listHandlesResponse;
            } while (marker);
        }
    }
    /**
     * Returns an AsyncIterableIterator for handles
     *
     * @param options - Options to list handles operation.
     */
    async *listHandleItems(options = {}) {
        let marker;
        for await (const listHandlesResponse of this.iterateHandleSegments(marker, options)) {
            if (listHandlesResponse.handleList) {
                for (const handle of listHandlesResponse.handleList) {
                    yield handle;
                }
            }
        }
    }
    /**
     * Returns an async iterable iterator to list all the handles.
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the handles in pages.
     *
     * @param options - Options to list handles operation.
     *
     * An asyncIterableIterator that supports paging.
     */
    listHandles(options = {}) {
        // an AsyncIterableIterator to iterate over handles
        const iter = this.listHandleItems(options);
        return {
            /**
             * The next method, part of the iteration protocol
             */
            async next() {
                return iter.next();
            },
            /**
             * The connection to the async iterator, part of the iteration protocol
             */
            [Symbol.asyncIterator]() {
                return this;
            },
            /**
             * Return an AsyncIterableIterator that works a page at a time
             */
            byPage: (settings = {}) => {
                return this.iterateHandleSegments(removeEmptyString(settings.continuationToken), {
                    maxPageSize: settings.maxPageSize,
                    ...options,
                });
            },
        };
    }
    /**
     * Force close all handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param marker - Optional. A string value that identifies the position of handles that will
     *                          be closed with the next force close handles operation.
     *                          The operation returns a marker value within the response
     *                          body if there are more handles to close. The marker value
     *                          may then be used in a subsequent call to close the next set of handles.
     * @param options - Options to force close handles operation.
     */
    async forceCloseHandlesSegment(marker, options = {}) {
        return tracingClient.withSpan("ShareFileClient-forceCloseHandlesSegment", options, async (updatedOptions) => {
            marker = marker === "" ? undefined : marker;
            const rawResponse = await this.context.forceCloseHandles("*", {
                ...updatedOptions,
                ...this.shareClientConfig,
                marker,
            });
            const response = rawResponse;
            response.closedHandlesCount = rawResponse.numberOfHandlesClosed || 0;
            response.closeFailureCount = rawResponse.numberOfHandlesFailedToClose || 0;
            return response;
        });
    }
    /**
     * Force close all handles for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param options - Options to force close handles operation.
     */
    async forceCloseAllHandles(options = {}) {
        return tracingClient.withSpan("ShareFileClient-forceCloseAllHandles", options, async (updatedOptions) => {
            let handlesClosed = 0;
            let numberOfHandlesFailedToClose = 0;
            let marker = "";
            do {
                const response = await this.forceCloseHandlesSegment(marker, { tracingOptions: updatedOptions.tracingOptions });
                marker = response.marker;
                if (response.closedHandlesCount) {
                    handlesClosed += response.closedHandlesCount;
                }
                if (response.closeFailureCount) {
                    numberOfHandlesFailedToClose += response.closeFailureCount;
                }
            } while (marker);
            return {
                closedHandlesCount: handlesClosed,
                closeFailureCount: numberOfHandlesFailedToClose,
            };
        });
    }
    /**
     * Force close a specific handle for a file.
     * @see https://learn.microsoft.com/rest/api/storageservices/force-close-handles
     *
     * @param handleId - Specific handle ID, cannot be asterisk "*".
     *                          Use forceCloseAllHandles() to close all handles.
     * @param FileForceCloseHandlesOptions - Options to force close handles operation.
     */
    async forceCloseHandle(handleId, options = {}) {
        return tracingClient.withSpan("ShareFileClient-forceCloseHandle", options, async (updatedOptions) => {
            if (handleId === "*") {
                throw new RangeError(`Parameter handleID should be a specified handle ID. Use forceCloseHandlesSegment() to close all handles.`);
            }
            const rawResponse = await this.context.forceCloseHandles(handleId, {
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            const response = rawResponse;
            response.closedHandlesCount = rawResponse.numberOfHandlesClosed || 0;
            response.closeFailureCount = rawResponse.numberOfHandlesFailedToClose || 0;
            return response;
        });
    }
    /**
     * NFS only.  Creates a hard link to the file file specified by path.
     * @param targetFile - Path of the file to create the hard link to, not including the share.
     *  For example: "targetDirectory/targetSubDirectory/.../targetFile"
     * @param options - Options to create hard link operation.
     */
    async createHardLink(targetFile, options = {}) {
        return tracingClient.withSpan("ShareFileClient-createHardLink", options, async (updatedOptions) => {
            const rawResponse = await this.context.createHardLink(targetFile, {
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    fileType: rawResponse.nfsFileType,
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    linkCount: rawResponse.linkCount,
                },
            });
        });
    }
    /**
     * NFS only.  Creates a symbolic link.
     * @param linkText - The path to the original file, the symbolic link is pointing to.
     *                  The path is of type string which is not resolved and is stored as is. The path can be absolute path
     *                 or the relative path depending on the content stored in the symbolic link file.
     * @param options - Options to create hard link operation.
     */
    async createSymbolicLink(linkText, options = {}) {
        return tracingClient.withSpan("ShareFileClient-createSymbolicLink", options, async (updatedOptions) => {
            const rawResponse = await this.context.createSymbolicLink(linkText, {
                ...updatedOptions,
                ...this.shareClientConfig,
            });
            return assertResponse({
                ...rawResponse,
                _response: rawResponse._response, // _response is made non-enumerable,
                posixProperties: {
                    fileMode: parseOctalFileMode(rawResponse.fileMode),
                    owner: rawResponse.owner,
                    group: rawResponse.group,
                    fileType: rawResponse.nfsFileType,
                },
            });
        });
    }
    /**
     * NFS only.  Gets content of a symbolic link.
     * @param options - Options to get symbolic link operation.
     */
    async getSymbolicLink(options = {}) {
        return tracingClient.withSpan("ShareFileClient-getSymbolicLink", options, async (updatedOptions) => {
            return assertResponse(await this.context.getSymbolicLink({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * Get a {@link ShareLeaseClient} that manages leases on the file.
     *
     * @param proposeLeaseId - Initial proposed lease Id.
     * @returns A new ShareLeaseClient object for managing leases on the file.
     */
    getShareLeaseClient(proposeLeaseId) {
        return new ShareLeaseClient(this, proposeLeaseId);
    }
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the SAS when the client is initialized with a shared key credential");
        }
        const sas = generateFileSASQueryParameters({
            shareName: this.shareName,
            filePath: this.path,
            ...options,
        }, this.credential).toString();
        return appendToURLQuery(this.url, sas);
    }
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options*/
    generateSasStringToSign(options) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the SAS when the client is initialized with a shared key credential");
        }
        return generateFileSASQueryParametersInternal({
            shareName: this.shareName,
            filePath: this.path,
            ...options,
        }, this.credential).stringToSign;
    }
    /**
     * Renames a file.
     * This API only supports renaming a file in the same share.
     *
     * @param destinationPath - Specifies the destination path to rename to. The path will be encoded to put into a URL to specify the destination.
     * @param options - Options for the renaming operation.
     * @returns Response data for the file renaming operation.
     *
     * Example usage:
     *
     * ```ts snippet:ReadmeSampleRenameFile
     * import { StorageSharedKeyCredential, ShareServiceClient } from "@azure/storage-file-share";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     *
     * const credential = new StorageSharedKeyCredential(account, accountKey);
     * const serviceClient = new ShareServiceClient(
     *   `https://${account}.file.core.windows.net`,
     *   credential,
     * );
     *
     * const shareName = "<share name>";
     * const directoryName = "<directory name>";
     * const fileName = "<file name>";
     * const destinationPath = "<destination path>";
     * const fileClient = serviceClient
     *   .getShareClient(shareName)
     *   .getDirectoryClient(directoryName)
     *   .getFileClient(fileName);
     *
     * await fileClient.rename(destinationPath);
     * ```
     */
    async rename(destinationPath, options = {}) {
        const split = destinationPath.split("?");
        let destinationUrl;
        if (split.length === 2) {
            const pathOnly = EscapePath(split[0]);
            const renameDestination = `/${this.shareName}/${pathOnly}`;
            destinationUrl = setURLPath(this.url, renameDestination);
            destinationUrl = setURLQueries(destinationUrl, split[1]);
        }
        else if (split.length === 1) {
            const pathOnly = EscapePath(destinationPath);
            const renameDestination = `/${this.shareName}/${pathOnly}`;
            destinationUrl = setURLPath(this.url, renameDestination);
        }
        else {
            throw new RangeError("Destination path should not contain more than one query string");
        }
        const destFile = new ShareFileClient(destinationUrl, this.pipeline, this.shareClientConfig);
        return tracingClient.withSpan("ShareFileClient-rename", options, async (updatedOptions) => {
            const response = assertResponse(await destFile.context.rename(this.url, {
                ...updatedOptions,
                sourceLeaseAccessConditions: updatedOptions.sourceLeaseAccessConditions
                    ? {
                        sourceLeaseId: updatedOptions.sourceLeaseAccessConditions.leaseId,
                    }
                    : undefined,
                destinationLeaseAccessConditions: updatedOptions.destinationLeaseAccessConditions
                    ? {
                        destinationLeaseId: updatedOptions.destinationLeaseAccessConditions.leaseId,
                    }
                    : undefined,
                fileHttpHeaders: options.contentType
                    ? {
                        fileContentType: options.contentType,
                    }
                    : undefined,
                ...this.shareClientConfig,
            }));
            return {
                destinationFileClient: destFile,
                fileRenameResponse: response,
            };
        });
    }
}
/**
 * A client that manages leases for a {@link ShareFileClient} or {@link ShareClient}.
 * @see https://learn.microsoft.com/rest/api/storageservices/lease-file
 * and
 * @see https://learn.microsoft.com/rest/api/storageservices/lease-share
 */
export class ShareLeaseClient {
    _leaseId;
    _url;
    fileOrShare;
    shareClientConfig;
    /**
     * Gets the lease Id.
     *
     * @readonly
     */
    get leaseId() {
        return this._leaseId;
    }
    /**
     * Gets the url.
     *
     * @readonly
     */
    get url() {
        return this._url;
    }
    /**
     * Creates an instance of ShareLeaseClient.
     * @param client - The client to make the lease operation requests.
     * @param leaseId - Initial proposed lease id.
     */
    constructor(client, leaseId) {
        const clientContext = client["storageClientContext"];
        if (client instanceof ShareClient) {
            this.fileOrShare = clientContext.share;
            this.shareClientConfig = client["shareClientConfig"];
        }
        else {
            this.fileOrShare = clientContext.file;
            this.shareClientConfig = client["shareClientConfig"];
        }
        this._url = client.url;
        if (!leaseId) {
            leaseId = randomUUID();
        }
        this._leaseId = leaseId;
    }
    /**
     * Establishes and manages a lock on a file, share or share snapshot for write and delete operations.
     *
     * @param duration - Specifies the duration of lease in seconds. For file, the only allowed value is -1 for a lease that never expires. For share, must be -1 or between 15 to 60.
     * @param options - Options for the lease management operation.
     * @returns Response data for acquire lease operation.
     */
    async acquireLease(duration = -1, options = {}) {
        return tracingClient.withSpan("ShareLeaseClient-acquireLease", options, async (updatedOptions) => {
            return assertResponse(await this.fileOrShare.acquireLease({
                ...updatedOptions,
                ...this.shareClientConfig,
                duration,
                proposedLeaseId: this._leaseId,
            }));
        });
    }
    /**
     * To change the ID of an existing lease.
     *
     * @param proposedLeaseId - the proposed new lease Id.
     * @param options - Options for the lease management operation.
     * @returns Response data for change lease operation.
     */
    async changeLease(proposedLeaseId, options = {}) {
        return tracingClient.withSpan("ShareLeaseClient-changeLease", options, async (updatedOptions) => {
            const response = assertResponse(await this.fileOrShare.changeLease(this._leaseId, {
                ...updatedOptions,
                ...this.shareClientConfig,
                proposedLeaseId,
            }));
            this._leaseId = proposedLeaseId;
            return response;
        });
    }
    /**
     * To free the lease if it is no longer needed so that another client may
     * immediately acquire a lease.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for release lease operation.
     */
    async releaseLease(options = {}) {
        return tracingClient.withSpan("ShareLeaseClient-releaseLease", options, async (updatedOptions) => {
            return assertResponse(await this.fileOrShare.releaseLease(this._leaseId, {
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * To force end the lease.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for break lease operation.
     */
    async breakLease(options = {}) {
        return tracingClient.withSpan("ShareLeaseClient-breakLease", options, async (updatedOptions) => {
            return assertResponse(await this.fileOrShare.breakLease({
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
    /**
     * To renew the lease. Only available for lease on share or share snapshot.
     * Note that the lease may be renewed even if it has expired as long as the share has not been leased again since the expiration of that lease.
     * When you renew a lease, the lease duration clock resets.
     *
     * @param options - Options for the lease management operation.
     * @returns Response data for renew lease operation.
     */
    async renewLease(options = {}) {
        return tracingClient.withSpan("ShareLeaseClient-renewLease", options, async (updatedOptions) => {
            if (isFile(this.fileOrShare)) {
                throw new RangeError("The renewLease operation is not available for lease on file.");
            }
            return assertResponse(await this.fileOrShare.renewLease(this._leaseId, {
                ...updatedOptions,
                ...this.shareClientConfig,
            }));
        });
    }
}
/**
 * @internal
 */
function isFile(fileOrShare) {
    return "renewLease" in fileOrShare;
}
//# sourceMappingURL=Clients.js.map