from __future__ import annotations

from typing import Union, Dict, Any, List, Optional, Set, cast
import dpath
import re

from checkov.common.graph.graph_builder import CustomAttributes
from checkov.common.graph.graph_builder.utils import calculate_hash
from checkov.common.typing import TFDefinitionKeyType
from checkov.terraform.graph_builder.utils import INTERPOLATION_EXPR
from checkov.common.graph.graph_builder.graph_components.blocks import Block
from checkov.common.util.consts import RESOLVED_MODULE_ENTRY_NAME
from checkov.terraform.graph_builder.graph_components.block_types import BlockType
from checkov.terraform.modules.module_objects import TFModule


class TerraformBlock(Block):
    __slots__ = (
        "module_connections",
        "source_module",
        "has_dynamic_block",
        "dynamic_attributes",
        "foreach_attrs",
        "source_module_object",
        "for_each_index"
    )

    def __init__(
        self,
        name: str,
        config: Dict[str, Any],
        path: TFDefinitionKeyType,
        block_type: str,
        attributes: Dict[str, Any],
        id: str = "",
        source: str = "",
        has_dynamic_block: bool = False,
        dynamic_attributes: dict[str, Any] | None = None,
    ) -> None:
        """
            when adding a new field be sure to add it to the equality function below
            :param name: unique name given to the terraform block, for example: 'aws_vpc.example_name'
            :param config: the section in tf_definitions that belong to this block
            :param path: the file location of the block
            :param block_type: BlockType
            :param attributes: dictionary of the block's original attributes in the terraform file
        """
        super().__init__(
            name=name,
            config=config,
            path=path,  # type:ignore[arg-type]  # Block class would need to be a Generic type to make this pass
            block_type=str(block_type),
            attributes=attributes,
            id=id,
            source=source,
            has_dynamic_block=has_dynamic_block,
            dynamic_attributes=dynamic_attributes,
        )
        if path:
            self.path = path  # type:ignore[assignment]  # Block class would need to be a Generic type to make this pass
        if attributes.get(RESOLVED_MODULE_ENTRY_NAME):
            del attributes[RESOLVED_MODULE_ENTRY_NAME]
        self.attributes = attributes
        self.module_connections: Dict[str, List[int]] = {}
        self.source_module: Set[int] = set()
        self.has_dynamic_block = has_dynamic_block
        self.source_module_object: Optional[TFModule] = None
        self.for_each_index: Optional[Any] = None
        self.foreach_attrs: list[str] | None = None

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, TerraformBlock):
            return False

        return self.name == other.name and self.config == other.config and self.path == other.path and \
            self.block_type == other.block_type and self.attributes == other.attributes and \
            self.id == other.id and self.has_dynamic_block == other.has_dynamic_block and self.source == other.source

    def get_attribute_dict(self, add_hash: bool = True) -> dict[str, Any]:
        """
           :return: map of all the block's native attributes (from the source file),
           combined with the attributes generated by the module builder.
           If the attributes are not a primitive type, they are converted to strings.
           """
        base_attributes = self.get_base_attributes()
        self.get_origin_attributes(base_attributes)

        if hasattr(self, "module_dependency") and hasattr(self, "module_dependency_num"):
            base_attributes[CustomAttributes.MODULE_DEPENDENCY] = self.module_dependency
            base_attributes[CustomAttributes.MODULE_DEPENDENCY_NUM] = self.module_dependency_num

        if self.changed_attributes:
            # add changed attributes only for calculating the hash
            base_attributes["changed_attributes"] = sorted(self.changed_attributes.keys())

        if self.breadcrumbs:
            sorted_breadcrumbs = dict(sorted(self.breadcrumbs.items()))
            base_attributes[CustomAttributes.RENDERING_BREADCRUMBS] = sorted_breadcrumbs

        if hasattr(self, 'foreach_attrs'):
            base_attributes[CustomAttributes.FOREACH_ATTRS] = self.foreach_attrs

        if hasattr(self, 'source_module_object'):
            base_attributes[CustomAttributes.SOURCE_MODULE_OBJECT] = self.source_module_object

        if add_hash:
            base_attributes[CustomAttributes.HASH] = calculate_hash(base_attributes)

        if self.block_type == BlockType.DATA:
            base_attributes[CustomAttributes.RESOURCE_TYPE] = f'data.{self.id.split(".")[0]}'

        if self.block_type == BlockType.MODULE:
            # since module names are user defined we are just setting 'module' as resource type for easier searching
            base_attributes[CustomAttributes.RESOURCE_TYPE] = "module"

        if self.block_type == BlockType.PROVIDER:
            # provider_name is always a string, base_attributes needs better typing pipenv run mypy
            provider_name = cast(str, base_attributes[CustomAttributes.BLOCK_NAME])
            provider_type = provider_name.split(".")[0]
            # ex: provider.aws
            base_attributes[CustomAttributes.RESOURCE_TYPE] = f"provider.{provider_type}"

        if "changed_attributes" in base_attributes:
            # removed changed attributes if it was added previously for calculating hash.
            del base_attributes["changed_attributes"]

        return base_attributes

    def add_module_connection(self, attribute_key: str, vertex_id: int) -> None:
        self.module_connections.setdefault(attribute_key, []).append(vertex_id)

    def extract_additional_changed_attributes(self, attribute_key: str) -> List[str]:
        # if the `attribute_key` starts with a `for_each.` we know the attribute can't be a dynamic attribute as it
        # represents the for_each of the block, so we don't need extract dynamic changed attributes
        # Fix: https://github.com/bridgecrewio/checkov/issues/4324
        if self.has_dynamic_block and not attribute_key.startswith('for_each'):
            return self._extract_dynamic_changed_attributes(attribute_key)
        return super().extract_additional_changed_attributes(attribute_key)

    def _extract_dynamic_changed_attributes(self, dynamic_attribute_key: str, nesting_prefix: str = '') -> List[str]:
        dynamic_changed_attributes: list[str] = []
        dynamic_attribute_key_parts = dynamic_attribute_key.split('.')
        try:
            remainder_key_parts = ['start_extract_dynamic_changed_attributes']  # For 1st iteration
            while remainder_key_parts:
                dynamic_for_each_index = dynamic_attribute_key_parts.index('for_each')
                dynamic_content_key_parts, remainder_key_parts = \
                    dynamic_attribute_key_parts[:dynamic_for_each_index], dynamic_attribute_key_parts[dynamic_for_each_index + 1:]
                dynamic_block_name = dynamic_content_key_parts[-1]
                dynamic_content_path = dynamic_content_key_parts + ['content']
                if dpath.search(self.attributes, dynamic_content_path):
                    dynamic_block_content = dpath.get(self.attributes, dynamic_content_path)
                    for key, value in dynamic_block_content.items():
                        key_path = ".".join(filter(None, [nesting_prefix, dynamic_block_name, key]))
                        self._collect_dynamic_dependent_keys(dynamic_block_name, value, key_path, dynamic_content_path,
                                                             dynamic_changed_attributes)
                dynamic_attribute_key_parts = remainder_key_parts
            return dynamic_changed_attributes
        except ValueError:
            return dynamic_changed_attributes

    def _collect_dynamic_dependent_keys(self, dynamic_block_name: str, value: str | list[str] | dict[str, Any],
                                        key_path: str,
                                        dynamic_content_path: List[str], dynamic_changed_attributes: List[str]) -> None:
        if isinstance(value, str):
            dynamic_ref = f'{dynamic_block_name}.value'
            if "${" in value:
                interpolation_matches = re.findall(INTERPOLATION_EXPR, value)
                for match in interpolation_matches:
                    if dynamic_ref in match:
                        dynamic_changed_attributes.append(key_path)
        elif isinstance(value, list):
            for idx, sub_value in enumerate(value):
                self._collect_dynamic_dependent_keys(
                    dynamic_block_name, sub_value, f'{key_path}.{idx}', dynamic_content_path,
                    dynamic_changed_attributes)
        elif isinstance(value, dict):
            for sub_key, sub_value in value.items():
                if isinstance(sub_value, dict) and 'content' in sub_value.keys() and 'for_each' in sub_value.keys():
                    nested_dynamic_block_key_path = f'{".".join(dynamic_content_path)}.dynamic.{sub_key}.for_each'
                    dynamic_changed_attributes.extend(
                        self._extract_dynamic_changed_attributes(nested_dynamic_block_key_path,
                                                                 nesting_prefix=dynamic_block_name))
                else:
                    self._collect_dynamic_dependent_keys(
                        dynamic_block_name, sub_value, f'{key_path}.{sub_key}', dynamic_content_path,
                        dynamic_changed_attributes)

    def find_attribute(self, attribute: Optional[Union[str, List[str]]]) -> Optional[str]:
        """
        :param attribute: key to search in self.attributes
        The function searches for  attribute in self.attribute. It might not exist if the block is variable or output,
        or its search path might be different if its a resource.
        :return: the actual attribute key or None
        """
        if not attribute:
            return None

        if self.attributes.get(attribute[0]):
            return attribute[0]

        if self.block_type == BlockType.VARIABLE:
            return "default" if self.attributes.get("default") else None

        if self.block_type == BlockType.OUTPUT:
            return "value" if self.attributes.get("value") else None

        if self.block_type == BlockType.RESOURCE and len(attribute) > 1:
            # handle cases where attribute_at_dest == ['aws_s3_bucket.template_bucket', 'acl']
            if self.name == attribute[0] and self.attributes.get(attribute[1]):
                return attribute[1]

        return None

    def update_list_attribute(self, attribute_key: str, attribute_value: Any) -> None:
        """Updates list attributes with their index

        This needs to be overridden, because of our hcl parser adding a list around any value
        """

        if attribute_key not in self.attributes or isinstance(self.attributes[attribute_key][0], list):
            # sometimes the attribute_value is a list and replaces the whole value of the key, which makes it a normal value
            # ex. attribute_value = ["xyz"] and self.attributes[attribute_key][0] = "xyz"
            for idx, value in enumerate(attribute_value):
                self.attributes[f"{attribute_key}.{idx}"] = value

    @classmethod
    def get_inner_attributes(
            cls,
            attribute_key: str,
            attribute_value: Union[str, List[str], Dict[str, Any]],
            strip_list: bool = True
    ) -> Dict[str, Any]:
        if strip_list and isinstance(attribute_value, list) and len(attribute_value) == 1:
            attribute_value = attribute_value[0]

        return super().get_inner_attributes(
            attribute_key=attribute_key,
            attribute_value=attribute_value,
        )

    def to_dict(self) -> dict[str, Any]:
        return {
            'attributes': self.attributes,
            'block_type': self.block_type,
            'breadcrumbs': self.breadcrumbs,
            'config': self.config,
            'id': self.id,
            'module_connections': self.module_connections,
            'name': self.name,
            'path': self.path,
            'source': self.source,
            'source_module': list(self.source_module),
            'source_module_object': dict(self.source_module_object) if self.source_module_object else None
        }

    @staticmethod
    def from_dict(data: dict[str, Any]) -> TerraformBlock:
        tf_block = TerraformBlock(name=data.get('name', ''), block_type=data.get('block_type', ''),
                                  config=data.get('config', {}), id=data.get('id', ''),
                                  path=data.get('path', ''), source=data.get('source', ''),
                                  attributes=data.get('attributes', {})
                                  )

        tf_block.breadcrumbs = data.get('breadcrumbs', {})
        tf_block.module_connections = data.get('module_connections', {})
        tf_block.source_module = data.get('source_module', set())
        tf_block.source_module_object = TFModule.from_json(data.get('source_module_object'))
        return tf_block
