/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import Integer from './integer';
declare type StandardDate = Date;
declare type NumberOrInteger = number | Integer | bigint;
declare type Properties = {
    [key: string]: any;
};
/**
 * Class for Node Type.
 */
declare class Node<T extends NumberOrInteger = Integer, P extends Properties = Properties> {
    identity: T;
    labels: string[];
    properties: P;
    /**
     * @constructor
     * @protected
     * @param {Integer|number} identity - Unique identity
     * @param {Array<string>} labels - Array for all labels
     * @param {Properties} properties - Map with node properties
     */
    constructor(identity: T, labels: string[], properties: P);
    /**
     * @ignore
     */
    toString(): string;
}
/**
 * Test if given object is an instance of {@link Node} class.
 * @param {Object} obj the object to test.
 * @return {boolean} `true` if given object is a {@link Node}, `false` otherwise.
 */
declare function isNode(obj: object): obj is Node;
/**
 * Class for Relationship Type.
 */
declare class Relationship<T extends NumberOrInteger = Integer, P extends Properties = Properties> {
    identity: T;
    start: T;
    end: T;
    type: string;
    properties: P;
    /**
     * @constructor
     * @protected
     * @param {Integer|number} identity - Unique identity
     * @param {Integer|number} start - Identity of start Node
     * @param {Integer|number} end - Identity of end Node
     * @param {string} type - Relationship type
     * @param {Properties} properties - Map with relationship properties
     */
    constructor(identity: T, start: T, end: T, type: string, properties: P);
    /**
     * @ignore
     */
    toString(): string;
}
/**
 * Test if given object is an instance of {@link Relationship} class.
 * @param {Object} obj the object to test.
 * @return {boolean} `true` if given object is a {@link Relationship}, `false` otherwise.
 */
declare function isRelationship(obj: object): obj is Relationship;
/**
 * Class for UnboundRelationship Type.
 * @access private
 */
declare class UnboundRelationship<T extends NumberOrInteger = Integer, P extends Properties = Properties> {
    identity: T;
    type: string;
    properties: P;
    /**
     * @constructor
     * @protected
     * @param {Integer|number} identity - Unique identity
     * @param {string} type - Relationship type
     * @param {Properties} properties - Map with relationship properties
     */
    constructor(identity: T, type: string, properties: any);
    /**
     * Bind relationship
     *
     * @protected
     * @param {Integer} start - Identity of start node
     * @param {Integer} end - Identity of end node
     * @return {Relationship} - Created relationship
     */
    bind(start: T, end: T): Relationship<T>;
    /**
     * @ignore
     */
    toString(): string;
}
/**
 * Test if given object is an instance of {@link UnboundRelationship} class.
 * @param {Object} obj the object to test.
 * @return {boolean} `true` if given object is a {@link UnboundRelationship}, `false` otherwise.
 */
declare function isUnboundRelationship(obj: object): obj is UnboundRelationship;
/**
 * Class for PathSegment Type.
 */
declare class PathSegment<T extends NumberOrInteger = Integer> {
    start: Node<T>;
    relationship: Relationship<T>;
    end: Node<T>;
    /**
     * @constructor
     * @protected
     * @param {Node} start - start node
     * @param {Relationship} rel - relationship that connects start and end node
     * @param {Node} end - end node
     */
    constructor(start: Node<T>, rel: Relationship<T>, end: Node<T>);
}
/**
 * Test if given object is an instance of {@link PathSegment} class.
 * @param {Object} obj the object to test.
 * @return {boolean} `true` if given object is a {@link PathSegment}, `false` otherwise.
 */
declare function isPathSegment(obj: object): obj is PathSegment;
/**
 * Class for Path Type.
 */
declare class Path<T extends NumberOrInteger = Integer> {
    start: Node<T>;
    end: Node<T>;
    segments: PathSegment<T>[];
    length: number;
    /**
     * @constructor
     * @protected
     * @param {Node} start  - start node
     * @param {Node} end - end node
     * @param {Array<PathSegment>} segments - Array of Segments
     */
    constructor(start: Node<T>, end: Node<T>, segments: PathSegment<T>[]);
}
/**
 * Test if given object is an instance of {@link Path} class.
 * @param {Object} obj the object to test.
 * @return {boolean} `true` if given object is a {@link Path}, `false` otherwise.
 */
declare function isPath(obj: object): obj is Path;
export { Node, isNode, Relationship, isRelationship, UnboundRelationship, isUnboundRelationship, Path, isPath, PathSegment, isPathSegment };
export type { StandardDate, NumberOrInteger, };
