"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.canRetryOn = void 0;
var error_1 = require("../error");
/**
 * Verified error and returns if it could be retried or not
 *
 * @param _error The error
 * @returns If the transaction could be retried.
 */
function canRetryOn(_error) {
    return (_error &&
        _error instanceof error_1.Neo4jError &&
        _error.code &&
        (_error.code === error_1.SERVICE_UNAVAILABLE ||
            _error.code === error_1.SESSION_EXPIRED ||
            _isAuthorizationExpired(_error) ||
            _isTransientError(_error)));
}
exports.canRetryOn = canRetryOn;
function _isTransientError(error) {
    // Retries should not happen when transaction was explicitly terminated by the user.
    // Termination of transaction might result in two different error codes depending on where it was
    // terminated. These are really client errors but classification on the server is not entirely correct and
    // they are classified as transient.
    var code = error.code;
    if (code.indexOf('TransientError') >= 0) {
        if (code === 'Neo.TransientError.Transaction.Terminated' ||
            code === 'Neo.TransientError.Transaction.LockClientStopped') {
            return false;
        }
        return true;
    }
    return false;
}
function _isAuthorizationExpired(error) {
    return error.code === 'Neo.ClientError.Security.AuthorizationExpired';
}
