/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.IntMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.UnsignedLongs;
import java.math.RoundingMode;

@GwtCompatible(emulated=true)
public final class LongMath {
    @VisibleForTesting
    static final long MAX_SIGNED_POWER_OF_TWO = 0x4000000000000000L;
    @VisibleForTesting
    static final long MAX_POWER_OF_SQRT2_UNSIGNED = -5402926248376769404L;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{19, 18, 18, 18, 18, 17, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 13, 13, 13, 12, 12, 12, 12, 11, 11, 11, 10, 10, 10, 9, 9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] powersOf10 = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L};
    @GwtIncompatible
    @VisibleForTesting
    static final long[] halfPowersOf10 = new long[]{3L, 31L, 316L, 3162L, 31622L, 316227L, 3162277L, 31622776L, 316227766L, 3162277660L, 31622776601L, 316227766016L, 3162277660168L, 31622776601683L, 316227766016837L, 3162277660168379L, 31622776601683793L, 316227766016837933L, 3162277660168379331L};
    @VisibleForTesting
    static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    static final long[] factorials = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 3810779, 121977, 16175, 4337, 1733, 887, 534, 361, 265, 206, 169, 143, 125, 111, 101, 94, 88, 83, 79, 76, 74, 72, 70, 69, 68, 67, 67, 66, 66, 66, 66};
    @VisibleForTesting
    static final int[] biggestSimpleBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 2642246, 86251, 11724, 3218, 1313, 684, 419, 287, 214, 169, 139, 119, 105, 95, 87, 81, 76, 73, 70, 68, 66, 64, 63, 62, 62, 61, 61, 61};
    private static final int SIEVE_30 = -545925251;
    private static final long[][] millerRabinBaseSets = new long[][]{{291830L, 126401071349994536L}, {885594168L, 725270293939359937L, 3569819667048198375L}, {273919523040L, 15L, 7363882082L, 992620450144556L}, {47636622961200L, 2L, 2570940L, 211991001L, 3749873356L}, {7999252175582850L, 2L, 4130806001517L, 149795463772692060L, 186635894390467037L, 3967304179347715805L}, {585226005592931976L, 2L, 123635709730000L, 9233062284813009L, 43835965440333360L, 761179012939631437L, 1263739024124850375L}, {Long.MAX_VALUE, 2L, 325L, 9375L, 28178L, 450775L, 9780504L, 1795265022L}};

    @Beta
    public static long ceilingPowerOfTwo(long l2) {
        MathPreconditions.checkPositive("x", l2);
        if (l2 > 0x4000000000000000L) {
            throw new ArithmeticException("ceilingPowerOfTwo(" + l2 + ") is not representable as a long");
        }
        return 1L << -Long.numberOfLeadingZeros(l2 - 1L);
    }

    @Beta
    public static long floorPowerOfTwo(long l2) {
        MathPreconditions.checkPositive("x", l2);
        return 1L << 63 - Long.numberOfLeadingZeros(l2);
    }

    public static boolean isPowerOfTwo(long l2) {
        return l2 > 0L & (l2 & l2 - 1L) == 0L;
    }

    @VisibleForTesting
    static int lessThanBranchFree(long l2, long l3) {
        return (int)((l2 - l3 ^ 0xFFFFFFFFFFFFFFFFL ^ 0xFFFFFFFFFFFFFFFFL) >>> 63);
    }

    public static int log2(long l2, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", l2);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(LongMath.isPowerOfTwo(l2));
            }
            case DOWN: 
            case FLOOR: {
                return 63 - Long.numberOfLeadingZeros(l2);
            }
            case UP: 
            case CEILING: {
                return 64 - Long.numberOfLeadingZeros(l2 - 1L);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int n2 = Long.numberOfLeadingZeros(l2);
                long l3 = -5402926248376769404L >>> n2;
                int n3 = 63 - n2;
                return n3 + LongMath.lessThanBranchFree(l3, l2);
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    @GwtIncompatible
    public static int log10(long l2, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", l2);
        int n2 = LongMath.log10Floor(l2);
        long l3 = powersOf10[n2];
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l2 == l3);
            }
            case DOWN: 
            case FLOOR: {
                return n2;
            }
            case UP: 
            case CEILING: {
                return n2 + LongMath.lessThanBranchFree(l3, l2);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return n2 + LongMath.lessThanBranchFree(halfPowersOf10[n2], l2);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    static int log10Floor(long l2) {
        byte by2 = maxLog10ForLeadingZeros[Long.numberOfLeadingZeros(l2)];
        return by2 - LongMath.lessThanBranchFree(l2, powersOf10[by2]);
    }

    @GwtIncompatible
    public static long pow(long l2, int n2) {
        MathPreconditions.checkNonNegative("exponent", n2);
        if (-2L <= l2 && l2 <= 2L) {
            switch ((int)l2) {
                case 0: {
                    return n2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (n2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    return n2 < 64 ? 1L << n2 : 0L;
                }
                case -2: {
                    if (n2 < 64) {
                        return (n2 & 1) == 0 ? 1L << n2 : -(1L << n2);
                    }
                    return 0L;
                }
            }
            throw new AssertionError();
        }
        long l3 = 1L;
        while (true) {
            switch (n2) {
                case 0: {
                    return l3;
                }
                case 1: {
                    return l3 * l2;
                }
            }
            l3 *= (n2 & 1) == 0 ? 1L : l2;
            l2 *= l2;
            n2 >>= 1;
        }
    }

    @GwtIncompatible
    public static long sqrt(long l2, RoundingMode roundingMode) {
        MathPreconditions.checkNonNegative("x", l2);
        if (LongMath.fitsInInt(l2)) {
            return IntMath.sqrt((int)l2, roundingMode);
        }
        long l3 = (long)Math.sqrt(l2);
        long l4 = l3 * l3;
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l4 == l2);
                return l3;
            }
            case DOWN: 
            case FLOOR: {
                if (l2 < l4) {
                    return l3 - 1L;
                }
                return l3;
            }
            case UP: 
            case CEILING: {
                if (l2 > l4) {
                    return l3 + 1L;
                }
                return l3;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long l5 = l3 - (long)(l2 < l4 ? 1 : 0);
                long l6 = l5 * l5 + l5;
                return l5 + (long)LongMath.lessThanBranchFree(l6, l2);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static long divide(long l2, long l3, RoundingMode roundingMode) {
        boolean bl2;
        Preconditions.checkNotNull(roundingMode);
        long l4 = l2 / l3;
        long l5 = l2 - l3 * l4;
        if (l5 == 0L) {
            return l4;
        }
        int n2 = 1 | (int)((l2 ^ l3) >> 63);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(l5 == 0L);
            }
            case DOWN: {
                bl2 = false;
                break;
            }
            case UP: {
                bl2 = true;
                break;
            }
            case CEILING: {
                bl2 = n2 > 0;
                break;
            }
            case FLOOR: {
                bl2 = n2 < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long l6 = Math.abs(l5);
                long l7 = l6 - (Math.abs(l3) - l6);
                if (l7 == 0L) {
                    bl2 = roundingMode == RoundingMode.HALF_UP | roundingMode == RoundingMode.HALF_EVEN & (l4 & 1L) != 0L;
                    break;
                }
                bl2 = l7 > 0L;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return bl2 ? l4 + (long)n2 : l4;
    }

    @GwtIncompatible
    public static int mod(long l2, int n2) {
        return (int)LongMath.mod(l2, (long)n2);
    }

    @GwtIncompatible
    public static long mod(long l2, long l3) {
        if (l3 <= 0L) {
            throw new ArithmeticException("Modulus must be positive");
        }
        long l4 = l2 % l3;
        return l4 >= 0L ? l4 : l4 + l3;
    }

    public static long gcd(long l2, long l3) {
        MathPreconditions.checkNonNegative("a", l2);
        MathPreconditions.checkNonNegative("b", l3);
        if (l2 == 0L) {
            return l3;
        }
        if (l3 == 0L) {
            return l2;
        }
        int n2 = Long.numberOfTrailingZeros(l2);
        l2 >>= n2;
        int n3 = Long.numberOfTrailingZeros(l3);
        l3 >>= n3;
        while (l2 != l3) {
            long l4 = l2 - l3;
            long l5 = l4 & l4 >> 63;
            l2 = l4 - l5 - l5;
            l3 += l5;
            l2 >>= Long.numberOfTrailingZeros(l2);
        }
        return l2 << Math.min(n2, n3);
    }

    @GwtIncompatible
    public static long checkedAdd(long l2, long l3) {
        long l4 = l2 + l3;
        MathPreconditions.checkNoOverflow((l2 ^ l3) < 0L | (l2 ^ l4) >= 0L, "checkedAdd", l2, l3);
        return l4;
    }

    @GwtIncompatible
    public static long checkedSubtract(long l2, long l3) {
        long l4 = l2 - l3;
        MathPreconditions.checkNoOverflow((l2 ^ l3) >= 0L | (l2 ^ l4) >= 0L, "checkedSubtract", l2, l3);
        return l4;
    }

    public static long checkedMultiply(long l2, long l3) {
        int n2 = Long.numberOfLeadingZeros(l2) + Long.numberOfLeadingZeros(l2 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(l3) + Long.numberOfLeadingZeros(l3 ^ 0xFFFFFFFFFFFFFFFFL);
        if (n2 > 65) {
            return l2 * l3;
        }
        MathPreconditions.checkNoOverflow(n2 >= 64, "checkedMultiply", l2, l3);
        MathPreconditions.checkNoOverflow(l2 >= 0L | l3 != Long.MIN_VALUE, "checkedMultiply", l2, l3);
        long l4 = l2 * l3;
        MathPreconditions.checkNoOverflow(l2 == 0L || l4 / l2 == l3, "checkedMultiply", l2, l3);
        return l4;
    }

    @GwtIncompatible
    public static long checkedPow(long l2, int n2) {
        MathPreconditions.checkNonNegative("exponent", n2);
        if (l2 >= -2L & l2 <= 2L) {
            switch ((int)l2) {
                case 0: {
                    return n2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (n2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    MathPreconditions.checkNoOverflow(n2 < 63, "checkedPow", l2, (long)n2);
                    return 1L << n2;
                }
                case -2: {
                    MathPreconditions.checkNoOverflow(n2 < 64, "checkedPow", l2, (long)n2);
                    return (n2 & 1) == 0 ? 1L << n2 : -1L << n2;
                }
            }
            throw new AssertionError();
        }
        long l3 = 1L;
        while (true) {
            switch (n2) {
                case 0: {
                    return l3;
                }
                case 1: {
                    return LongMath.checkedMultiply(l3, l2);
                }
            }
            if ((n2 & 1) != 0) {
                l3 = LongMath.checkedMultiply(l3, l2);
            }
            if ((n2 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-3037000499L <= l2 && l2 <= 3037000499L, "checkedPow", l2, (long)n2);
            l2 *= l2;
        }
    }

    @Beta
    public static long saturatedAdd(long l2, long l3) {
        long l4;
        if ((l2 ^ l3) < 0L | (l2 ^ (l4 = l2 + l3)) >= 0L) {
            return l4;
        }
        return Long.MAX_VALUE + (l4 >>> 63 ^ 1L);
    }

    @Beta
    public static long saturatedSubtract(long l2, long l3) {
        long l4;
        if ((l2 ^ l3) >= 0L | (l2 ^ (l4 = l2 - l3)) >= 0L) {
            return l4;
        }
        return Long.MAX_VALUE + (l4 >>> 63 ^ 1L);
    }

    @Beta
    public static long saturatedMultiply(long l2, long l3) {
        int n2 = Long.numberOfLeadingZeros(l2) + Long.numberOfLeadingZeros(l2 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(l3) + Long.numberOfLeadingZeros(l3 ^ 0xFFFFFFFFFFFFFFFFL);
        if (n2 > 65) {
            return l2 * l3;
        }
        long l4 = Long.MAX_VALUE + ((l2 ^ l3) >>> 63);
        if (n2 < 64 | l2 < 0L & l3 == Long.MIN_VALUE) {
            return l4;
        }
        long l5 = l2 * l3;
        if (l2 == 0L || l5 / l2 == l3) {
            return l5;
        }
        return l4;
    }

    @Beta
    public static long saturatedPow(long l2, int n2) {
        MathPreconditions.checkNonNegative("exponent", n2);
        if (l2 >= -2L & l2 <= 2L) {
            switch ((int)l2) {
                case 0: {
                    return n2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (n2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    if (n2 >= 63) {
                        return Long.MAX_VALUE;
                    }
                    return 1L << n2;
                }
                case -2: {
                    if (n2 >= 64) {
                        return Long.MAX_VALUE + (long)(n2 & 1);
                    }
                    return (n2 & 1) == 0 ? 1L << n2 : -1L << n2;
                }
            }
            throw new AssertionError();
        }
        long l3 = 1L;
        long l4 = Long.MAX_VALUE + (l2 >>> 63 & (long)(n2 & 1));
        while (true) {
            switch (n2) {
                case 0: {
                    return l3;
                }
                case 1: {
                    return LongMath.saturatedMultiply(l3, l2);
                }
            }
            if ((n2 & 1) != 0) {
                l3 = LongMath.saturatedMultiply(l3, l2);
            }
            if ((n2 >>= 1) <= 0) continue;
            if (-3037000499L > l2 | l2 > 3037000499L) {
                return l4;
            }
            l2 *= l2;
        }
    }

    @GwtIncompatible
    public static long factorial(int n2) {
        MathPreconditions.checkNonNegative("n", n2);
        return n2 < factorials.length ? factorials[n2] : Long.MAX_VALUE;
    }

    public static long binomial(int n2, int n3) {
        MathPreconditions.checkNonNegative("n", n2);
        MathPreconditions.checkNonNegative("k", n3);
        Preconditions.checkArgument(n3 <= n2, "k (%s) > n (%s)", n3, n2);
        if (n3 > n2 >> 1) {
            n3 = n2 - n3;
        }
        switch (n3) {
            case 0: {
                return 1L;
            }
            case 1: {
                return n2;
            }
        }
        if (n2 < factorials.length) {
            return factorials[n2] / (factorials[n3] * factorials[n2 - n3]);
        }
        if (n3 >= biggestBinomials.length || n2 > biggestBinomials[n3]) {
            return Long.MAX_VALUE;
        }
        if (n3 < biggestSimpleBinomials.length && n2 <= biggestSimpleBinomials[n3]) {
            long l2 = n2--;
            for (int i2 = 2; i2 <= n3; ++i2) {
                l2 *= (long)n2;
                l2 /= (long)i2;
                --n2;
            }
            return l2;
        }
        int n4 = LongMath.log2(n2, RoundingMode.CEILING);
        long l3 = 1L;
        long l4 = n2--;
        long l5 = 1L;
        int n5 = n4;
        int n6 = 2;
        while (n6 <= n3) {
            if (n5 + n4 < 63) {
                l4 *= (long)n2;
                l5 *= (long)n6;
                n5 += n4;
            } else {
                l3 = LongMath.multiplyFraction(l3, l4, l5);
                l4 = n2;
                l5 = n6;
                n5 = n4;
            }
            ++n6;
            --n2;
        }
        return LongMath.multiplyFraction(l3, l4, l5);
    }

    static long multiplyFraction(long l2, long l3, long l4) {
        if (l2 == 1L) {
            return l3 / l4;
        }
        long l5 = LongMath.gcd(l2, l4);
        return (l2 /= l5) * (l3 / (l4 /= l5));
    }

    static boolean fitsInInt(long l2) {
        return (long)((int)l2) == l2;
    }

    public static long mean(long l2, long l3) {
        return (l2 & l3) + ((l2 ^ l3) >> 1);
    }

    @GwtIncompatible
    @Beta
    public static boolean isPrime(long l2) {
        if (l2 < 2L) {
            MathPreconditions.checkNonNegative("n", l2);
            return false;
        }
        if (l2 == 2L || l2 == 3L || l2 == 5L || l2 == 7L || l2 == 11L || l2 == 13L) {
            return true;
        }
        if ((0xDF75D77D & 1 << (int)(l2 % 30L)) != 0) {
            return false;
        }
        if (l2 % 7L == 0L || l2 % 11L == 0L || l2 % 13L == 0L) {
            return false;
        }
        if (l2 < 289L) {
            return true;
        }
        for (long[] lArray : millerRabinBaseSets) {
            if (l2 > lArray[0]) continue;
            for (int i2 = 1; i2 < lArray.length; ++i2) {
                if (MillerRabinTester.test(lArray[i2], l2)) continue;
                return false;
            }
            return true;
        }
        throw new AssertionError();
    }

    private LongMath() {
    }

    private static enum MillerRabinTester {
        SMALL{

            @Override
            long mulMod(long l2, long l3, long l4) {
                return l2 * l3 % l4;
            }

            @Override
            long squareMod(long l2, long l3) {
                return l2 * l2 % l3;
            }
        }
        ,
        LARGE{

            private long plusMod(long l2, long l3, long l4) {
                return l2 >= l4 - l3 ? l2 + l3 - l4 : l2 + l3;
            }

            private long times2ToThe32Mod(long l2, long l3) {
                int n2;
                int n3 = 32;
                do {
                    n2 = Math.min(n3, Long.numberOfLeadingZeros(l2));
                    l2 = UnsignedLongs.remainder(l2 << n2, l3);
                } while ((n3 -= n2) > 0);
                return l2;
            }

            @Override
            long mulMod(long l2, long l3, long l4) {
                long l5 = l2 >>> 32;
                long l6 = l3 >>> 32;
                long l7 = l2 & 0xFFFFFFFFL;
                long l8 = l3 & 0xFFFFFFFFL;
                long l9 = this.times2ToThe32Mod(l5 * l6, l4);
                if ((l9 += l5 * l8) < 0L) {
                    l9 = UnsignedLongs.remainder(l9, l4);
                }
                l9 += l7 * l6;
                l9 = this.times2ToThe32Mod(l9, l4);
                return this.plusMod(l9, UnsignedLongs.remainder(l7 * l8, l4), l4);
            }

            @Override
            long squareMod(long l2, long l3) {
                long l4 = l2 >>> 32;
                long l5 = l2 & 0xFFFFFFFFL;
                long l6 = this.times2ToThe32Mod(l4 * l4, l3);
                long l7 = l4 * l5 * 2L;
                if (l7 < 0L) {
                    l7 = UnsignedLongs.remainder(l7, l3);
                }
                l6 += l7;
                l6 = this.times2ToThe32Mod(l6, l3);
                return this.plusMod(l6, UnsignedLongs.remainder(l5 * l5, l3), l3);
            }
        };


        static boolean test(long l2, long l3) {
            return (l3 <= 3037000499L ? SMALL : LARGE).testWitness(l2, l3);
        }

        abstract long mulMod(long var1, long var3, long var5);

        abstract long squareMod(long var1, long var3);

        private long powMod(long l2, long l3, long l4) {
            long l5 = 1L;
            while (l3 != 0L) {
                if ((l3 & 1L) != 0L) {
                    l5 = this.mulMod(l5, l2, l4);
                }
                l2 = this.squareMod(l2, l4);
                l3 >>= 1;
            }
            return l5;
        }

        private boolean testWitness(long l2, long l3) {
            int n2 = Long.numberOfTrailingZeros(l3 - 1L);
            long l4 = l3 - 1L >> n2;
            if ((l2 %= l3) == 0L) {
                return true;
            }
            long l5 = this.powMod(l2, l4, l3);
            if (l5 == 1L) {
                return true;
            }
            int n3 = 0;
            while (l5 != l3 - 1L) {
                if (++n3 == n2) {
                    return false;
                }
                l5 = this.squareMod(l5, l3);
            }
            return true;
        }
    }
}

