/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.errorprone.annotations.concurrent.LazyInit;

@Beta
@GwtIncompatible
public abstract class LinearTransformation {
    public static LinearTransformationBuilder mapping(double d2, double d3) {
        Preconditions.checkArgument(DoubleUtils.isFinite(d2) && DoubleUtils.isFinite(d3));
        return new LinearTransformationBuilder(d2, d3);
    }

    public static LinearTransformation vertical(double d2) {
        Preconditions.checkArgument(DoubleUtils.isFinite(d2));
        return new VerticalLinearTransformation(d2);
    }

    public static LinearTransformation horizontal(double d2) {
        Preconditions.checkArgument(DoubleUtils.isFinite(d2));
        double d3 = 0.0;
        return new RegularLinearTransformation(d3, d2);
    }

    public static LinearTransformation forNaN() {
        return NaNLinearTransformation.INSTANCE;
    }

    public abstract boolean isVertical();

    public abstract boolean isHorizontal();

    public abstract double slope();

    public abstract double transform(double var1);

    public abstract LinearTransformation inverse();

    private static final class NaNLinearTransformation
    extends LinearTransformation {
        static final NaNLinearTransformation INSTANCE = new NaNLinearTransformation();

        private NaNLinearTransformation() {
        }

        @Override
        public boolean isVertical() {
            return false;
        }

        @Override
        public boolean isHorizontal() {
            return false;
        }

        @Override
        public double slope() {
            return Double.NaN;
        }

        @Override
        public double transform(double d2) {
            return Double.NaN;
        }

        @Override
        public LinearTransformation inverse() {
            return this;
        }

        public String toString() {
            return "NaN";
        }
    }

    private static final class VerticalLinearTransformation
    extends LinearTransformation {
        final double x;
        @LazyInit
        LinearTransformation inverse;

        VerticalLinearTransformation(double d2) {
            this.x = d2;
            this.inverse = null;
        }

        VerticalLinearTransformation(double d2, LinearTransformation linearTransformation) {
            this.x = d2;
            this.inverse = linearTransformation;
        }

        @Override
        public boolean isVertical() {
            return true;
        }

        @Override
        public boolean isHorizontal() {
            return false;
        }

        @Override
        public double slope() {
            throw new IllegalStateException();
        }

        @Override
        public double transform(double d2) {
            throw new IllegalStateException();
        }

        @Override
        public LinearTransformation inverse() {
            LinearTransformation linearTransformation = this.inverse;
            return linearTransformation == null ? (this.inverse = this.createInverse()) : linearTransformation;
        }

        public String toString() {
            return String.format("x = %g", this.x);
        }

        private LinearTransformation createInverse() {
            return new RegularLinearTransformation(0.0, this.x, this);
        }
    }

    private static final class RegularLinearTransformation
    extends LinearTransformation {
        final double slope;
        final double yIntercept;
        @LazyInit
        LinearTransformation inverse;

        RegularLinearTransformation(double d2, double d3) {
            this.slope = d2;
            this.yIntercept = d3;
            this.inverse = null;
        }

        RegularLinearTransformation(double d2, double d3, LinearTransformation linearTransformation) {
            this.slope = d2;
            this.yIntercept = d3;
            this.inverse = linearTransformation;
        }

        @Override
        public boolean isVertical() {
            return false;
        }

        @Override
        public boolean isHorizontal() {
            return this.slope == 0.0;
        }

        @Override
        public double slope() {
            return this.slope;
        }

        @Override
        public double transform(double d2) {
            return d2 * this.slope + this.yIntercept;
        }

        @Override
        public LinearTransformation inverse() {
            LinearTransformation linearTransformation = this.inverse;
            return linearTransformation == null ? (this.inverse = this.createInverse()) : linearTransformation;
        }

        public String toString() {
            return String.format("y = %g * x + %g", this.slope, this.yIntercept);
        }

        private LinearTransformation createInverse() {
            if (this.slope != 0.0) {
                return new RegularLinearTransformation(1.0 / this.slope, -1.0 * this.yIntercept / this.slope, this);
            }
            return new VerticalLinearTransformation(this.yIntercept, this);
        }
    }

    public static final class LinearTransformationBuilder {
        private final double x1;
        private final double y1;

        private LinearTransformationBuilder(double d2, double d3) {
            this.x1 = d2;
            this.y1 = d3;
        }

        public LinearTransformation and(double d2, double d3) {
            Preconditions.checkArgument(DoubleUtils.isFinite(d2) && DoubleUtils.isFinite(d3));
            if (d2 == this.x1) {
                Preconditions.checkArgument(d3 != this.y1);
                return new VerticalLinearTransformation(this.x1);
            }
            return this.withSlope((d3 - this.y1) / (d2 - this.x1));
        }

        public LinearTransformation withSlope(double d2) {
            Preconditions.checkArgument(!Double.isNaN(d2));
            if (DoubleUtils.isFinite(d2)) {
                double d3 = this.y1 - this.x1 * d2;
                return new RegularLinearTransformation(d2, d3);
            }
            return new VerticalLinearTransformation(this.x1);
        }
    }
}

