/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Ints;
import java.math.RoundingMode;

@GwtCompatible(emulated=true)
public final class IntMath {
    @VisibleForTesting
    static final int MAX_SIGNED_POWER_OF_TWO = 0x40000000;
    @VisibleForTesting
    static final int MAX_POWER_OF_SQRT2_UNSIGNED = -1257966797;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0};
    @VisibleForTesting
    static final int[] powersOf10 = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};
    @VisibleForTesting
    static final int[] halfPowersOf10 = new int[]{3, 31, 316, 3162, 31622, 316227, 3162277, 31622776, 316227766, Integer.MAX_VALUE};
    @VisibleForTesting
    static final int FLOOR_SQRT_MAX_INT = 46340;
    private static final int[] factorials = new int[]{1, 1, 2, 6, 24, 120, 720, 5040, 40320, 362880, 3628800, 39916800, 479001600};
    @VisibleForTesting
    static int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, 65536, 2345, 477, 193, 110, 75, 58, 49, 43, 39, 37, 35, 34, 34, 33};

    @Beta
    public static int ceilingPowerOfTwo(int n2) {
        MathPreconditions.checkPositive("x", n2);
        if (n2 > 0x40000000) {
            throw new ArithmeticException("ceilingPowerOfTwo(" + n2 + ") not representable as an int");
        }
        return 1 << -Integer.numberOfLeadingZeros(n2 - 1);
    }

    @Beta
    public static int floorPowerOfTwo(int n2) {
        MathPreconditions.checkPositive("x", n2);
        return Integer.highestOneBit(n2);
    }

    public static boolean isPowerOfTwo(int n2) {
        return n2 > 0 & (n2 & n2 - 1) == 0;
    }

    @VisibleForTesting
    static int lessThanBranchFree(int n2, int n3) {
        return ~(~(n2 - n3)) >>> 31;
    }

    public static int log2(int n2, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", n2);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(IntMath.isPowerOfTwo(n2));
            }
            case DOWN: 
            case FLOOR: {
                return 31 - Integer.numberOfLeadingZeros(n2);
            }
            case UP: 
            case CEILING: {
                return 32 - Integer.numberOfLeadingZeros(n2 - 1);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int n3 = Integer.numberOfLeadingZeros(n2);
                int n4 = -1257966797 >>> n3;
                int n5 = 31 - n3;
                return n5 + IntMath.lessThanBranchFree(n4, n2);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(int n2, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", n2);
        int n3 = IntMath.log10Floor(n2);
        int n4 = powersOf10[n3];
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(n2 == n4);
            }
            case DOWN: 
            case FLOOR: {
                return n3;
            }
            case UP: 
            case CEILING: {
                return n3 + IntMath.lessThanBranchFree(n4, n2);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return n3 + IntMath.lessThanBranchFree(halfPowersOf10[n3], n2);
            }
        }
        throw new AssertionError();
    }

    private static int log10Floor(int n2) {
        byte by2 = maxLog10ForLeadingZeros[Integer.numberOfLeadingZeros(n2)];
        return by2 - IntMath.lessThanBranchFree(n2, powersOf10[by2]);
    }

    @GwtIncompatible
    public static int pow(int n2, int n3) {
        MathPreconditions.checkNonNegative("exponent", n3);
        switch (n2) {
            case 0: {
                return n3 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (n3 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                return n3 < 32 ? 1 << n3 : 0;
            }
            case -2: {
                if (n3 < 32) {
                    return (n3 & 1) == 0 ? 1 << n3 : -(1 << n3);
                }
                return 0;
            }
        }
        int n4 = 1;
        while (true) {
            switch (n3) {
                case 0: {
                    return n4;
                }
                case 1: {
                    return n2 * n4;
                }
            }
            n4 *= (n3 & 1) == 0 ? 1 : n2;
            n2 *= n2;
            n3 >>= 1;
        }
    }

    @GwtIncompatible
    public static int sqrt(int n2, RoundingMode roundingMode) {
        MathPreconditions.checkNonNegative("x", n2);
        int n3 = IntMath.sqrtFloor(n2);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(n3 * n3 == n2);
            }
            case DOWN: 
            case FLOOR: {
                return n3;
            }
            case UP: 
            case CEILING: {
                return n3 + IntMath.lessThanBranchFree(n3 * n3, n2);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int n4 = n3 * n3 + n3;
                return n3 + IntMath.lessThanBranchFree(n4, n2);
            }
        }
        throw new AssertionError();
    }

    private static int sqrtFloor(int n2) {
        return (int)Math.sqrt(n2);
    }

    public static int divide(int n2, int n3, RoundingMode roundingMode) {
        boolean bl2;
        Preconditions.checkNotNull(roundingMode);
        if (n3 == 0) {
            throw new ArithmeticException("/ by zero");
        }
        int n4 = n2 / n3;
        int n5 = n2 - n3 * n4;
        if (n5 == 0) {
            return n4;
        }
        int n6 = 1 | (n2 ^ n3) >> 31;
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(n5 == 0);
            }
            case DOWN: {
                bl2 = false;
                break;
            }
            case UP: {
                bl2 = true;
                break;
            }
            case CEILING: {
                bl2 = n6 > 0;
                break;
            }
            case FLOOR: {
                bl2 = n6 < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int n7 = Math.abs(n5);
                int n8 = n7 - (Math.abs(n3) - n7);
                if (n8 == 0) {
                    bl2 = roundingMode == RoundingMode.HALF_UP || roundingMode == RoundingMode.HALF_EVEN & (n4 & 1) != 0;
                    break;
                }
                bl2 = n8 > 0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return bl2 ? n4 + n6 : n4;
    }

    public static int mod(int n2, int n3) {
        if (n3 <= 0) {
            throw new ArithmeticException("Modulus " + n3 + " must be > 0");
        }
        int n4 = n2 % n3;
        return n4 >= 0 ? n4 : n4 + n3;
    }

    public static int gcd(int n2, int n3) {
        MathPreconditions.checkNonNegative("a", n2);
        MathPreconditions.checkNonNegative("b", n3);
        if (n2 == 0) {
            return n3;
        }
        if (n3 == 0) {
            return n2;
        }
        int n4 = Integer.numberOfTrailingZeros(n2);
        n2 >>= n4;
        int n5 = Integer.numberOfTrailingZeros(n3);
        n3 >>= n5;
        while (n2 != n3) {
            int n6 = n2 - n3;
            int n7 = n6 & n6 >> 31;
            n2 = n6 - n7 - n7;
            n3 += n7;
            n2 >>= Integer.numberOfTrailingZeros(n2);
        }
        return n2 << Math.min(n4, n5);
    }

    public static int checkedAdd(int n2, int n3) {
        long l2 = (long)n2 + (long)n3;
        MathPreconditions.checkNoOverflow(l2 == (long)((int)l2), "checkedAdd", n2, n3);
        return (int)l2;
    }

    public static int checkedSubtract(int n2, int n3) {
        long l2 = (long)n2 - (long)n3;
        MathPreconditions.checkNoOverflow(l2 == (long)((int)l2), "checkedSubtract", n2, n3);
        return (int)l2;
    }

    public static int checkedMultiply(int n2, int n3) {
        long l2 = (long)n2 * (long)n3;
        MathPreconditions.checkNoOverflow(l2 == (long)((int)l2), "checkedMultiply", n2, n3);
        return (int)l2;
    }

    public static int checkedPow(int n2, int n3) {
        MathPreconditions.checkNonNegative("exponent", n3);
        switch (n2) {
            case 0: {
                return n3 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (n3 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                MathPreconditions.checkNoOverflow(n3 < 31, "checkedPow", n2, n3);
                return 1 << n3;
            }
            case -2: {
                MathPreconditions.checkNoOverflow(n3 < 32, "checkedPow", n2, n3);
                return (n3 & 1) == 0 ? 1 << n3 : -1 << n3;
            }
        }
        int n4 = 1;
        while (true) {
            switch (n3) {
                case 0: {
                    return n4;
                }
                case 1: {
                    return IntMath.checkedMultiply(n4, n2);
                }
            }
            if ((n3 & 1) != 0) {
                n4 = IntMath.checkedMultiply(n4, n2);
            }
            if ((n3 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-46340 <= n2 & n2 <= 46340, "checkedPow", n2, n3);
            n2 *= n2;
        }
    }

    @Beta
    public static int saturatedAdd(int n2, int n3) {
        return Ints.saturatedCast((long)n2 + (long)n3);
    }

    @Beta
    public static int saturatedSubtract(int n2, int n3) {
        return Ints.saturatedCast((long)n2 - (long)n3);
    }

    @Beta
    public static int saturatedMultiply(int n2, int n3) {
        return Ints.saturatedCast((long)n2 * (long)n3);
    }

    @Beta
    public static int saturatedPow(int n2, int n3) {
        MathPreconditions.checkNonNegative("exponent", n3);
        switch (n2) {
            case 0: {
                return n3 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (n3 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                if (n3 >= 31) {
                    return Integer.MAX_VALUE;
                }
                return 1 << n3;
            }
            case -2: {
                if (n3 >= 32) {
                    return Integer.MAX_VALUE + (n3 & 1);
                }
                return (n3 & 1) == 0 ? 1 << n3 : -1 << n3;
            }
        }
        int n4 = 1;
        int n5 = Integer.MAX_VALUE + (n2 >>> 31 & (n3 & 1));
        while (true) {
            switch (n3) {
                case 0: {
                    return n4;
                }
                case 1: {
                    return IntMath.saturatedMultiply(n4, n2);
                }
            }
            if ((n3 & 1) != 0) {
                n4 = IntMath.saturatedMultiply(n4, n2);
            }
            if ((n3 >>= 1) <= 0) continue;
            if (-46340 > n2 | n2 > 46340) {
                return n5;
            }
            n2 *= n2;
        }
    }

    public static int factorial(int n2) {
        MathPreconditions.checkNonNegative("n", n2);
        return n2 < factorials.length ? factorials[n2] : Integer.MAX_VALUE;
    }

    public static int binomial(int n2, int n3) {
        MathPreconditions.checkNonNegative("n", n2);
        MathPreconditions.checkNonNegative("k", n3);
        Preconditions.checkArgument(n3 <= n2, "k (%s) > n (%s)", n3, n2);
        if (n3 > n2 >> 1) {
            n3 = n2 - n3;
        }
        if (n3 >= biggestBinomials.length || n2 > biggestBinomials[n3]) {
            return Integer.MAX_VALUE;
        }
        switch (n3) {
            case 0: {
                return 1;
            }
            case 1: {
                return n2;
            }
        }
        long l2 = 1L;
        for (int i2 = 0; i2 < n3; ++i2) {
            l2 *= (long)(n2 - i2);
            l2 /= (long)(i2 + 1);
        }
        return (int)l2;
    }

    public static int mean(int n2, int n3) {
        return (n2 & n3) + ((n2 ^ n3) >> 1);
    }

    @GwtIncompatible
    @Beta
    public static boolean isPrime(int n2) {
        return LongMath.isPrime(n2);
    }

    private IntMath() {
    }
}

