/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Booleans;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.Iterator;

@GwtCompatible(emulated=true)
public final class DoubleMath {
    private static final double MIN_INT_AS_DOUBLE = -2.147483648E9;
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final double LN_2 = Math.log(2.0);
    @VisibleForTesting
    static final int MAX_FACTORIAL = 170;
    @VisibleForTesting
    static final double[] everySixteenthFactorial = new double[]{1.0, 2.0922789888E13, 2.631308369336935E35, 1.2413915592536073E61, 1.2688693218588417E89, 7.156945704626381E118, 9.916779348709496E149, 1.974506857221074E182, 3.856204823625804E215, 5.5502938327393044E249, 4.7147236359920616E284};

    @GwtIncompatible
    static double roundIntermediate(double d2, RoundingMode roundingMode) {
        if (!DoubleUtils.isFinite(d2)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isMathematicalInteger(d2));
                return d2;
            }
            case FLOOR: {
                if (d2 >= 0.0 || DoubleMath.isMathematicalInteger(d2)) {
                    return d2;
                }
                return (long)d2 - 1L;
            }
            case CEILING: {
                if (d2 <= 0.0 || DoubleMath.isMathematicalInteger(d2)) {
                    return d2;
                }
                return (long)d2 + 1L;
            }
            case DOWN: {
                return d2;
            }
            case UP: {
                if (DoubleMath.isMathematicalInteger(d2)) {
                    return d2;
                }
                return (long)d2 + (long)(d2 > 0.0 ? 1 : -1);
            }
            case HALF_EVEN: {
                return Math.rint(d2);
            }
            case HALF_UP: {
                double d3 = Math.rint(d2);
                if (Math.abs(d2 - d3) == 0.5) {
                    return d2 + Math.copySign(0.5, d2);
                }
                return d3;
            }
            case HALF_DOWN: {
                double d4 = Math.rint(d2);
                if (Math.abs(d2 - d4) == 0.5) {
                    return d2;
                }
                return d4;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int roundToInt(double d2, RoundingMode roundingMode) {
        double d3 = DoubleMath.roundIntermediate(d2, roundingMode);
        MathPreconditions.checkInRangeForRoundingInputs(d3 > -2.147483649E9 & d3 < 2.147483648E9, d2, roundingMode);
        return (int)d3;
    }

    @GwtIncompatible
    public static long roundToLong(double d2, RoundingMode roundingMode) {
        double d3 = DoubleMath.roundIntermediate(d2, roundingMode);
        MathPreconditions.checkInRangeForRoundingInputs(-9.223372036854776E18 - d3 < 1.0 & d3 < 9.223372036854776E18, d2, roundingMode);
        return (long)d3;
    }

    @GwtIncompatible
    public static BigInteger roundToBigInteger(double d2, RoundingMode roundingMode) {
        if (-9.223372036854776E18 - (d2 = DoubleMath.roundIntermediate(d2, roundingMode)) < 1.0 & d2 < 9.223372036854776E18) {
            return BigInteger.valueOf((long)d2);
        }
        int n2 = Math.getExponent(d2);
        long l2 = DoubleUtils.getSignificand(d2);
        BigInteger bigInteger = BigInteger.valueOf(l2).shiftLeft(n2 - 52);
        return d2 < 0.0 ? bigInteger.negate() : bigInteger;
    }

    @GwtIncompatible
    public static boolean isPowerOfTwo(double d2) {
        if (d2 > 0.0 && DoubleUtils.isFinite(d2)) {
            long l2 = DoubleUtils.getSignificand(d2);
            return (l2 & l2 - 1L) == 0L;
        }
        return false;
    }

    public static double log2(double d2) {
        return Math.log(d2) / LN_2;
    }

    @GwtIncompatible
    public static int log2(double d2, RoundingMode roundingMode) {
        boolean bl2;
        Preconditions.checkArgument(d2 > 0.0 && DoubleUtils.isFinite(d2), "x must be positive and finite");
        int n2 = Math.getExponent(d2);
        if (!DoubleUtils.isNormal(d2)) {
            return DoubleMath.log2(d2 * 4.503599627370496E15, roundingMode) - 52;
        }
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isPowerOfTwo(d2));
            }
            case FLOOR: {
                bl2 = false;
                break;
            }
            case CEILING: {
                bl2 = !DoubleMath.isPowerOfTwo(d2);
                break;
            }
            case DOWN: {
                bl2 = n2 < 0 & !DoubleMath.isPowerOfTwo(d2);
                break;
            }
            case UP: {
                bl2 = n2 >= 0 & !DoubleMath.isPowerOfTwo(d2);
                break;
            }
            case HALF_EVEN: 
            case HALF_UP: 
            case HALF_DOWN: {
                double d3 = DoubleUtils.scaleNormalize(d2);
                bl2 = d3 * d3 > 2.0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return bl2 ? n2 + 1 : n2;
    }

    @GwtIncompatible
    public static boolean isMathematicalInteger(double d2) {
        return DoubleUtils.isFinite(d2) && (d2 == 0.0 || 52 - Long.numberOfTrailingZeros(DoubleUtils.getSignificand(d2)) <= Math.getExponent(d2));
    }

    public static double factorial(int n2) {
        MathPreconditions.checkNonNegative("n", n2);
        if (n2 > 170) {
            return Double.POSITIVE_INFINITY;
        }
        double d2 = 1.0;
        for (int i2 = 1 + (n2 & 0xFFFFFFF0); i2 <= n2; ++i2) {
            d2 *= (double)i2;
        }
        return d2 * everySixteenthFactorial[n2 >> 4];
    }

    public static boolean fuzzyEquals(double d2, double d3, double d4) {
        MathPreconditions.checkNonNegative("tolerance", d4);
        return Math.copySign(d2 - d3, 1.0) <= d4 || d2 == d3 || Double.isNaN(d2) && Double.isNaN(d3);
    }

    public static int fuzzyCompare(double d2, double d3, double d4) {
        if (DoubleMath.fuzzyEquals(d2, d3, d4)) {
            return 0;
        }
        if (d2 < d3) {
            return -1;
        }
        if (d2 > d3) {
            return 1;
        }
        return Booleans.compare(Double.isNaN(d2), Double.isNaN(d3));
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(double ... dArray) {
        Preconditions.checkArgument(dArray.length > 0, "Cannot take mean of 0 values");
        long l2 = 1L;
        double d2 = DoubleMath.checkFinite(dArray[0]);
        for (int i2 = 1; i2 < dArray.length; ++i2) {
            DoubleMath.checkFinite(dArray[i2]);
            d2 += (dArray[i2] - d2) / (double)(++l2);
        }
        return d2;
    }

    @Deprecated
    public static double mean(int ... nArray) {
        Preconditions.checkArgument(nArray.length > 0, "Cannot take mean of 0 values");
        long l2 = 0L;
        for (int i2 = 0; i2 < nArray.length; ++i2) {
            l2 += (long)nArray[i2];
        }
        return (double)l2 / (double)nArray.length;
    }

    @Deprecated
    public static double mean(long ... lArray) {
        Preconditions.checkArgument(lArray.length > 0, "Cannot take mean of 0 values");
        long l2 = 1L;
        double d2 = lArray[0];
        for (int i2 = 1; i2 < lArray.length; ++i2) {
            d2 += ((double)lArray[i2] - d2) / (double)(++l2);
        }
        return d2;
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(Iterable<? extends Number> iterable) {
        return DoubleMath.mean(iterable.iterator());
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(Iterator<? extends Number> iterator) {
        Preconditions.checkArgument(iterator.hasNext(), "Cannot take mean of 0 values");
        long l2 = 1L;
        double d2 = DoubleMath.checkFinite(iterator.next().doubleValue());
        while (iterator.hasNext()) {
            double d3 = DoubleMath.checkFinite(iterator.next().doubleValue());
            d2 += (d3 - d2) / (double)(++l2);
        }
        return d2;
    }

    @GwtIncompatible
    @CanIgnoreReturnValue
    private static double checkFinite(double d2) {
        Preconditions.checkArgument(DoubleUtils.isFinite(d2));
        return d2;
    }

    private DoubleMath() {
    }
}

