/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.math.DoubleUtils;
import com.google.common.math.IntMath;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

@GwtCompatible(emulated=true)
public final class BigIntegerMath {
    @VisibleForTesting
    static final int SQRT2_PRECOMPUTE_THRESHOLD = 256;
    @VisibleForTesting
    static final BigInteger SQRT2_PRECOMPUTED_BITS = new BigInteger("16a09e667f3bcc908b2fb1366ea957d3e3adec17512775099da2f590b0667322a", 16);
    private static final double LN_10 = Math.log(10.0);
    private static final double LN_2 = Math.log(2.0);

    @Beta
    public static BigInteger ceilingPowerOfTwo(BigInteger bigInteger) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(bigInteger, RoundingMode.CEILING));
    }

    @Beta
    public static BigInteger floorPowerOfTwo(BigInteger bigInteger) {
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(bigInteger, RoundingMode.FLOOR));
    }

    public static boolean isPowerOfTwo(BigInteger bigInteger) {
        Preconditions.checkNotNull(bigInteger);
        return bigInteger.signum() > 0 && bigInteger.getLowestSetBit() == bigInteger.bitLength() - 1;
    }

    public static int log2(BigInteger bigInteger, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", Preconditions.checkNotNull(bigInteger));
        int n2 = bigInteger.bitLength() - 1;
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(BigIntegerMath.isPowerOfTwo(bigInteger));
            }
            case DOWN: 
            case FLOOR: {
                return n2;
            }
            case UP: 
            case CEILING: {
                return BigIntegerMath.isPowerOfTwo(bigInteger) ? n2 : n2 + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                if (n2 < 256) {
                    BigInteger bigInteger2 = SQRT2_PRECOMPUTED_BITS.shiftRight(256 - n2);
                    if (bigInteger.compareTo(bigInteger2) <= 0) {
                        return n2;
                    }
                    return n2 + 1;
                }
                BigInteger bigInteger3 = bigInteger.pow(2);
                int n3 = bigInteger3.bitLength() - 1;
                return n3 < 2 * n2 + 1 ? n2 : n2 + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(BigInteger bigInteger, RoundingMode roundingMode) {
        MathPreconditions.checkPositive("x", bigInteger);
        if (BigIntegerMath.fitsInLong(bigInteger)) {
            return LongMath.log10(bigInteger.longValue(), roundingMode);
        }
        int n2 = (int)((double)BigIntegerMath.log2(bigInteger, RoundingMode.FLOOR) * LN_2 / LN_10);
        BigInteger bigInteger2 = BigInteger.TEN.pow(n2);
        int n3 = bigInteger2.compareTo(bigInteger);
        if (n3 > 0) {
            do {
                --n2;
            } while ((n3 = (bigInteger2 = bigInteger2.divide(BigInteger.TEN)).compareTo(bigInteger)) > 0);
        } else {
            BigInteger bigInteger3 = BigInteger.TEN.multiply(bigInteger2);
            int n4 = bigInteger3.compareTo(bigInteger);
            while (n4 <= 0) {
                ++n2;
                bigInteger2 = bigInteger3;
                n3 = n4;
                bigInteger3 = BigInteger.TEN.multiply(bigInteger2);
                n4 = bigInteger3.compareTo(bigInteger);
            }
        }
        int n5 = n2;
        BigInteger bigInteger4 = bigInteger2;
        int n6 = n3;
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(n6 == 0);
            }
            case DOWN: 
            case FLOOR: {
                return n5;
            }
            case UP: 
            case CEILING: {
                return bigInteger4.equals(bigInteger) ? n5 : n5 + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger bigInteger5 = bigInteger.pow(2);
                BigInteger bigInteger6 = bigInteger4.pow(2).multiply(BigInteger.TEN);
                return bigInteger5.compareTo(bigInteger6) <= 0 ? n5 : n5 + 1;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static BigInteger sqrt(BigInteger bigInteger, RoundingMode roundingMode) {
        MathPreconditions.checkNonNegative("x", bigInteger);
        if (BigIntegerMath.fitsInLong(bigInteger)) {
            return BigInteger.valueOf(LongMath.sqrt(bigInteger.longValue(), roundingMode));
        }
        BigInteger bigInteger2 = BigIntegerMath.sqrtFloor(bigInteger);
        switch (roundingMode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(bigInteger2.pow(2).equals(bigInteger));
            }
            case DOWN: 
            case FLOOR: {
                return bigInteger2;
            }
            case UP: 
            case CEILING: {
                int n2 = bigInteger2.intValue();
                boolean bl2 = n2 * n2 == bigInteger.intValue() && bigInteger2.pow(2).equals(bigInteger);
                return bl2 ? bigInteger2 : bigInteger2.add(BigInteger.ONE);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                BigInteger bigInteger3 = bigInteger2.pow(2).add(bigInteger2);
                return bigInteger3.compareTo(bigInteger) >= 0 ? bigInteger2 : bigInteger2.add(BigInteger.ONE);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    private static BigInteger sqrtFloor(BigInteger bigInteger) {
        BigInteger bigInteger2;
        int n2 = BigIntegerMath.log2(bigInteger, RoundingMode.FLOOR);
        if (n2 < 1023) {
            bigInteger2 = BigIntegerMath.sqrtApproxWithDoubles(bigInteger);
        } else {
            int n3 = n2 - 52 & 0xFFFFFFFE;
            bigInteger2 = BigIntegerMath.sqrtApproxWithDoubles(bigInteger.shiftRight(n3)).shiftLeft(n3 >> 1);
        }
        BigInteger bigInteger3 = bigInteger2.add(bigInteger.divide(bigInteger2)).shiftRight(1);
        if (bigInteger2.equals(bigInteger3)) {
            return bigInteger2;
        }
        while ((bigInteger3 = (bigInteger2 = bigInteger3).add(bigInteger.divide(bigInteger2)).shiftRight(1)).compareTo(bigInteger2) < 0) {
        }
        return bigInteger2;
    }

    @GwtIncompatible
    private static BigInteger sqrtApproxWithDoubles(BigInteger bigInteger) {
        return DoubleMath.roundToBigInteger(Math.sqrt(DoubleUtils.bigToDouble(bigInteger)), RoundingMode.HALF_EVEN);
    }

    @GwtIncompatible
    public static BigInteger divide(BigInteger bigInteger, BigInteger bigInteger2, RoundingMode roundingMode) {
        BigDecimal bigDecimal = new BigDecimal(bigInteger);
        BigDecimal bigDecimal2 = new BigDecimal(bigInteger2);
        return bigDecimal.divide(bigDecimal2, 0, roundingMode).toBigIntegerExact();
    }

    public static BigInteger factorial(int n2) {
        MathPreconditions.checkNonNegative("n", n2);
        if (n2 < LongMath.factorials.length) {
            return BigInteger.valueOf(LongMath.factorials[n2]);
        }
        int n3 = IntMath.divide(n2 * IntMath.log2(n2, RoundingMode.CEILING), 64, RoundingMode.CEILING);
        ArrayList<BigInteger> arrayList = new ArrayList<BigInteger>(n3);
        int n4 = LongMath.factorials.length;
        long l2 = LongMath.factorials[n4 - 1];
        int n5 = Long.numberOfTrailingZeros(l2);
        int n6 = LongMath.log2(l2 >>= n5, RoundingMode.FLOOR) + 1;
        int n7 = LongMath.log2(n4, RoundingMode.FLOOR) + 1;
        int n8 = 1 << n7 - 1;
        for (long i2 = (long)n4; i2 <= (long)n2; ++i2) {
            if ((i2 & (long)n8) != 0L) {
                n8 <<= 1;
                ++n7;
            }
            int n9 = Long.numberOfTrailingZeros(i2);
            long l3 = i2 >> n9;
            n5 += n9;
            int n10 = n7 - n9;
            if (n10 + n6 >= 64) {
                arrayList.add(BigInteger.valueOf(l2));
                l2 = 1L;
                n6 = 0;
            }
            n6 = LongMath.log2(l2 *= l3, RoundingMode.FLOOR) + 1;
        }
        if (l2 > 1L) {
            arrayList.add(BigInteger.valueOf(l2));
        }
        return BigIntegerMath.listProduct(arrayList).shiftLeft(n5);
    }

    static BigInteger listProduct(List<BigInteger> list) {
        return BigIntegerMath.listProduct(list, 0, list.size());
    }

    static BigInteger listProduct(List<BigInteger> list, int n2, int n3) {
        switch (n3 - n2) {
            case 0: {
                return BigInteger.ONE;
            }
            case 1: {
                return list.get(n2);
            }
            case 2: {
                return list.get(n2).multiply(list.get(n2 + 1));
            }
            case 3: {
                return list.get(n2).multiply(list.get(n2 + 1)).multiply(list.get(n2 + 2));
            }
        }
        int n4 = n3 + n2 >>> 1;
        return BigIntegerMath.listProduct(list, n2, n4).multiply(BigIntegerMath.listProduct(list, n4, n3));
    }

    public static BigInteger binomial(int n2, int n3) {
        int n4;
        MathPreconditions.checkNonNegative("n", n2);
        MathPreconditions.checkNonNegative("k", n3);
        Preconditions.checkArgument(n3 <= n2, "k (%s) > n (%s)", n3, n2);
        if (n3 > n2 >> 1) {
            n3 = n2 - n3;
        }
        if (n3 < LongMath.biggestBinomials.length && n2 <= LongMath.biggestBinomials[n3]) {
            return BigInteger.valueOf(LongMath.binomial(n2, n3));
        }
        BigInteger bigInteger = BigInteger.ONE;
        long l2 = n2;
        long l3 = 1L;
        int n5 = n4 = LongMath.log2(n2, RoundingMode.CEILING);
        for (int i2 = 1; i2 < n3; ++i2) {
            int n6 = n2 - i2;
            int n7 = i2 + 1;
            if (n5 + n4 >= 63) {
                bigInteger = bigInteger.multiply(BigInteger.valueOf(l2)).divide(BigInteger.valueOf(l3));
                l2 = n6;
                l3 = n7;
                n5 = n4;
                continue;
            }
            l2 *= (long)n6;
            l3 *= (long)n7;
            n5 += n4;
        }
        return bigInteger.multiply(BigInteger.valueOf(l2)).divide(BigInteger.valueOf(l3));
    }

    @GwtIncompatible
    static boolean fitsInLong(BigInteger bigInteger) {
        return bigInteger.bitLength() <= 63;
    }

    private BigIntegerMath() {
    }
}

