/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.graph.SuccessorsFunction;
import com.google.common.graph.Traverser;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import com.google.common.io.CharSink;
import com.google.common.io.CharSource;
import com.google.common.io.InsecureRecursiveDeleteException;
import com.google.common.io.RecursiveDeleteOption;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.file.DirectoryIteratorException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SecureDirectoryStream;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Stream;
import org.checkerframework.checker.nullness.qual.Nullable;

@Beta
@GwtIncompatible
public final class MoreFiles {
    private static final SuccessorsFunction<Path> FILE_TREE = new SuccessorsFunction<Path>(){

        @Override
        public Iterable<Path> successors(Path path) {
            return MoreFiles.fileTreeChildren(path);
        }
    };

    private MoreFiles() {
    }

    public static ByteSource asByteSource(Path path, OpenOption ... openOptionArray) {
        return new PathByteSource(path, openOptionArray);
    }

    public static ByteSink asByteSink(Path path, OpenOption ... openOptionArray) {
        return new PathByteSink(path, openOptionArray);
    }

    public static CharSource asCharSource(Path path, Charset charset, OpenOption ... openOptionArray) {
        return MoreFiles.asByteSource(path, openOptionArray).asCharSource(charset);
    }

    public static CharSink asCharSink(Path path, Charset charset, OpenOption ... openOptionArray) {
        return MoreFiles.asByteSink(path, openOptionArray).asCharSink(charset);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ImmutableList<Path> listFiles(Path path) {
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(path);){
            ImmutableList<Path> immutableList = ImmutableList.copyOf(directoryStream);
            return immutableList;
        }
        catch (DirectoryIteratorException directoryIteratorException) {
            throw directoryIteratorException.getCause();
        }
    }

    public static Traverser<Path> fileTraverser() {
        return Traverser.forTree(FILE_TREE);
    }

    private static Iterable<Path> fileTreeChildren(Path path) {
        if (Files.isDirectory(path, LinkOption.NOFOLLOW_LINKS)) {
            try {
                return MoreFiles.listFiles(path);
            }
            catch (IOException iOException) {
                throw new DirectoryIteratorException(iOException);
            }
        }
        return ImmutableList.of();
    }

    public static Predicate<Path> isDirectory(LinkOption ... linkOptionArray) {
        final LinkOption[] linkOptionArray2 = (LinkOption[])linkOptionArray.clone();
        return new Predicate<Path>(){

            @Override
            public boolean apply(Path path) {
                return Files.isDirectory(path, linkOptionArray2);
            }

            public String toString() {
                return "MoreFiles.isDirectory(" + Arrays.toString(linkOptionArray2) + ")";
            }
        };
    }

    private static boolean isDirectory(SecureDirectoryStream<Path> secureDirectoryStream, Path path, LinkOption ... linkOptionArray) {
        return secureDirectoryStream.getFileAttributeView(path, BasicFileAttributeView.class, linkOptionArray).readAttributes().isDirectory();
    }

    public static Predicate<Path> isRegularFile(LinkOption ... linkOptionArray) {
        final LinkOption[] linkOptionArray2 = (LinkOption[])linkOptionArray.clone();
        return new Predicate<Path>(){

            @Override
            public boolean apply(Path path) {
                return Files.isRegularFile(path, linkOptionArray2);
            }

            public String toString() {
                return "MoreFiles.isRegularFile(" + Arrays.toString(linkOptionArray2) + ")";
            }
        };
    }

    public static boolean equal(Path path, Path path2) {
        Preconditions.checkNotNull(path);
        Preconditions.checkNotNull(path2);
        if (Files.isSameFile(path, path2)) {
            return true;
        }
        ByteSource byteSource = MoreFiles.asByteSource(path, new OpenOption[0]);
        ByteSource byteSource2 = MoreFiles.asByteSource(path2, new OpenOption[0]);
        long l2 = byteSource.sizeIfKnown().or(0L);
        long l3 = byteSource2.sizeIfKnown().or(0L);
        if (l2 != 0L && l3 != 0L && l2 != l3) {
            return false;
        }
        return byteSource.contentEquals(byteSource2);
    }

    public static void touch(Path path) {
        Preconditions.checkNotNull(path);
        try {
            Files.setLastModifiedTime(path, FileTime.fromMillis(System.currentTimeMillis()));
        }
        catch (NoSuchFileException noSuchFileException) {
            try {
                Files.createFile(path, new FileAttribute[0]);
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
                // empty catch block
            }
        }
    }

    public static void createParentDirectories(Path path, FileAttribute<?> ... fileAttributeArray) {
        Path path2 = path.toAbsolutePath().normalize();
        Path path3 = path2.getParent();
        if (path3 == null) {
            return;
        }
        if (!Files.isDirectory(path3, new LinkOption[0])) {
            Files.createDirectories(path3, fileAttributeArray);
            if (!Files.isDirectory(path3, new LinkOption[0])) {
                throw new IOException("Unable to create parent directories of " + path);
            }
        }
    }

    public static String getFileExtension(Path path) {
        Path path2 = path.getFileName();
        if (path2 == null) {
            return "";
        }
        String string = path2.toString();
        int n2 = string.lastIndexOf(46);
        return n2 == -1 ? "" : string.substring(n2 + 1);
    }

    public static String getNameWithoutExtension(Path path) {
        Path path2 = path.getFileName();
        if (path2 == null) {
            return "";
        }
        String string = path2.toString();
        int n2 = string.lastIndexOf(46);
        return n2 == -1 ? string : string.substring(0, n2);
    }

    public static void deleteRecursively(Path path, RecursiveDeleteOption ... recursiveDeleteOptionArray) {
        Path path2 = MoreFiles.getParentPath(path);
        if (path2 == null) {
            throw new FileSystemException(path.toString(), null, "can't delete recursively");
        }
        Collection<IOException> collection = null;
        try {
            boolean bl2 = false;
            try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(path2);){
                if (directoryStream instanceof SecureDirectoryStream) {
                    bl2 = true;
                    collection = MoreFiles.deleteRecursivelySecure((SecureDirectoryStream)directoryStream, path.getFileName());
                }
            }
            if (!bl2) {
                MoreFiles.checkAllowsInsecure(path, recursiveDeleteOptionArray);
                collection = MoreFiles.deleteRecursivelyInsecure(path);
            }
        }
        catch (IOException iOException) {
            if (collection == null) {
                throw iOException;
            }
            collection.add(iOException);
        }
        if (collection != null) {
            MoreFiles.throwDeleteFailed(path, collection);
        }
    }

    public static void deleteDirectoryContents(Path path, RecursiveDeleteOption ... recursiveDeleteOptionArray) {
        Collection<IOException> collection = null;
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(path);){
            if (directoryStream instanceof SecureDirectoryStream) {
                SecureDirectoryStream secureDirectoryStream = (SecureDirectoryStream)directoryStream;
                collection = MoreFiles.deleteDirectoryContentsSecure(secureDirectoryStream);
            } else {
                MoreFiles.checkAllowsInsecure(path, recursiveDeleteOptionArray);
                collection = MoreFiles.deleteDirectoryContentsInsecure(directoryStream);
            }
        }
        catch (IOException iOException) {
            if (collection == null) {
                throw iOException;
            }
            collection.add(iOException);
        }
        if (collection != null) {
            MoreFiles.throwDeleteFailed(path, collection);
        }
    }

    private static @Nullable Collection<IOException> deleteRecursivelySecure(SecureDirectoryStream<Path> secureDirectoryStream, Path path) {
        Collection<IOException> collection = null;
        try {
            if (MoreFiles.isDirectory(secureDirectoryStream, path, LinkOption.NOFOLLOW_LINKS)) {
                try (SecureDirectoryStream<Path> secureDirectoryStream2 = secureDirectoryStream.newDirectoryStream(path, LinkOption.NOFOLLOW_LINKS);){
                    collection = MoreFiles.deleteDirectoryContentsSecure(secureDirectoryStream2);
                }
                if (collection == null) {
                    secureDirectoryStream.deleteDirectory(path);
                }
            } else {
                secureDirectoryStream.deleteFile(path);
            }
            return collection;
        }
        catch (IOException iOException) {
            return MoreFiles.addException(collection, iOException);
        }
    }

    private static @Nullable Collection<IOException> deleteDirectoryContentsSecure(SecureDirectoryStream<Path> secureDirectoryStream) {
        Collection<IOException> collection = null;
        try {
            for (Path path : secureDirectoryStream) {
                collection = MoreFiles.concat(collection, MoreFiles.deleteRecursivelySecure(secureDirectoryStream, path.getFileName()));
            }
            return collection;
        }
        catch (DirectoryIteratorException directoryIteratorException) {
            return MoreFiles.addException(collection, directoryIteratorException.getCause());
        }
    }

    private static @Nullable Collection<IOException> deleteRecursivelyInsecure(Path path) {
        Collection<IOException> collection = null;
        try {
            if (Files.isDirectory(path, LinkOption.NOFOLLOW_LINKS)) {
                try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(path);){
                    collection = MoreFiles.deleteDirectoryContentsInsecure(directoryStream);
                }
            }
            if (collection == null) {
                Files.delete(path);
            }
            return collection;
        }
        catch (IOException iOException) {
            return MoreFiles.addException(collection, iOException);
        }
    }

    private static @Nullable Collection<IOException> deleteDirectoryContentsInsecure(DirectoryStream<Path> directoryStream) {
        Collection<IOException> collection = null;
        try {
            for (Path path : directoryStream) {
                collection = MoreFiles.concat(collection, MoreFiles.deleteRecursivelyInsecure(path));
            }
            return collection;
        }
        catch (DirectoryIteratorException directoryIteratorException) {
            return MoreFiles.addException(collection, directoryIteratorException.getCause());
        }
    }

    private static @Nullable Path getParentPath(Path path) {
        Path path2 = path.getParent();
        if (path2 != null) {
            return path2;
        }
        if (path.getNameCount() == 0) {
            return null;
        }
        return path.getFileSystem().getPath(".", new String[0]);
    }

    private static void checkAllowsInsecure(Path path, RecursiveDeleteOption[] recursiveDeleteOptionArray) {
        if (!Arrays.asList(recursiveDeleteOptionArray).contains((Object)RecursiveDeleteOption.ALLOW_INSECURE)) {
            throw new InsecureRecursiveDeleteException(path.toString());
        }
    }

    private static Collection<IOException> addException(@Nullable Collection<IOException> collection, IOException iOException) {
        if (collection == null) {
            collection = new ArrayList<IOException>();
        }
        collection.add(iOException);
        return collection;
    }

    private static @Nullable Collection<IOException> concat(@Nullable Collection<IOException> collection, @Nullable Collection<IOException> collection2) {
        if (collection == null) {
            return collection2;
        }
        if (collection2 != null) {
            collection.addAll(collection2);
        }
        return collection;
    }

    private static void throwDeleteFailed(Path path, Collection<IOException> collection) {
        FileSystemException fileSystemException = new FileSystemException(path.toString(), null, "failed to delete one or more files; see suppressed exceptions for details");
        for (IOException iOException : collection) {
            fileSystemException.addSuppressed(iOException);
        }
        throw fileSystemException;
    }

    private static final class PathByteSink
    extends ByteSink {
        private final Path path;
        private final OpenOption[] options;

        private PathByteSink(Path path, OpenOption ... openOptionArray) {
            this.path = Preconditions.checkNotNull(path);
            this.options = (OpenOption[])openOptionArray.clone();
        }

        @Override
        public OutputStream openStream() {
            return Files.newOutputStream(this.path, this.options);
        }

        public String toString() {
            return "MoreFiles.asByteSink(" + this.path + ", " + Arrays.toString(this.options) + ")";
        }
    }

    private static final class PathByteSource
    extends ByteSource {
        private static final LinkOption[] FOLLOW_LINKS = new LinkOption[0];
        private final Path path;
        private final OpenOption[] options;
        private final boolean followLinks;

        private PathByteSource(Path path, OpenOption ... openOptionArray) {
            this.path = Preconditions.checkNotNull(path);
            this.options = (OpenOption[])openOptionArray.clone();
            this.followLinks = PathByteSource.followLinks(this.options);
        }

        private static boolean followLinks(OpenOption[] openOptionArray) {
            for (OpenOption openOption : openOptionArray) {
                if (openOption != LinkOption.NOFOLLOW_LINKS) continue;
                return false;
            }
            return true;
        }

        @Override
        public InputStream openStream() {
            return Files.newInputStream(this.path, this.options);
        }

        private BasicFileAttributes readAttributes() {
            LinkOption[] linkOptionArray;
            if (this.followLinks) {
                linkOptionArray = FOLLOW_LINKS;
            } else {
                LinkOption[] linkOptionArray2 = new LinkOption[1];
                linkOptionArray = linkOptionArray2;
                linkOptionArray2[0] = LinkOption.NOFOLLOW_LINKS;
            }
            return Files.readAttributes(this.path, BasicFileAttributes.class, linkOptionArray);
        }

        @Override
        public Optional<Long> sizeIfKnown() {
            BasicFileAttributes basicFileAttributes;
            try {
                basicFileAttributes = this.readAttributes();
            }
            catch (IOException iOException) {
                return Optional.absent();
            }
            if (basicFileAttributes.isDirectory() || basicFileAttributes.isSymbolicLink()) {
                return Optional.absent();
            }
            return Optional.of(basicFileAttributes.size());
        }

        @Override
        public long size() {
            BasicFileAttributes basicFileAttributes = this.readAttributes();
            if (basicFileAttributes.isDirectory()) {
                throw new IOException("can't read: is a directory");
            }
            if (basicFileAttributes.isSymbolicLink()) {
                throw new IOException("can't read: is a symbolic link");
            }
            return basicFileAttributes.size();
        }

        @Override
        public byte[] read() {
            try (SeekableByteChannel seekableByteChannel = Files.newByteChannel(this.path, this.options);){
                byte[] byArray = ByteStreams.toByteArray(Channels.newInputStream(seekableByteChannel), seekableByteChannel.size());
                return byArray;
            }
        }

        @Override
        public CharSource asCharSource(Charset charset) {
            if (this.options.length == 0) {
                return new ByteSource.AsCharSource(charset){

                    @Override
                    public Stream<String> lines() {
                        return Files.lines(path, this.charset);
                    }
                };
            }
            return super.asCharSource(charset);
        }

        public String toString() {
            return "MoreFiles.asByteSource(" + this.path + ", " + Arrays.toString(this.options) + ")";
        }
    }
}

