/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.Ascii;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.Funnels;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteProcessor;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteStreams;
import com.google.common.io.CharSource;
import com.google.common.io.Closer;
import com.google.common.io.MultiInputStream;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;

@GwtIncompatible
public abstract class ByteSource {
    protected ByteSource() {
    }

    public CharSource asCharSource(Charset charset) {
        return new AsCharSource(charset);
    }

    public abstract InputStream openStream();

    public InputStream openBufferedStream() {
        InputStream inputStream = this.openStream();
        return inputStream instanceof BufferedInputStream ? (BufferedInputStream)inputStream : new BufferedInputStream(inputStream);
    }

    public ByteSource slice(long l2, long l3) {
        return new SlicedByteSource(l2, l3);
    }

    public boolean isEmpty() {
        Optional<Long> optional = this.sizeIfKnown();
        if (optional.isPresent()) {
            return optional.get() == 0L;
        }
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            boolean bl2 = inputStream.read() == -1;
            return bl2;
        }
    }

    @Beta
    public Optional<Long> sizeIfKnown() {
        return Optional.absent();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long size() {
        Optional<Long> optional = this.sizeIfKnown();
        if (optional.isPresent()) {
            return optional.get();
        }
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            long l2 = this.countBySkipping(inputStream);
            return l2;
        }
        closer = Closer.create();
        try {
            InputStream inputStream = closer.register(this.openStream());
            long l3 = ByteStreams.exhaust(inputStream);
            return l3;
        }
        catch (Throwable throwable) {
            throw closer.rethrow(throwable);
        }
        finally {
            closer.close();
        }
    }

    private long countBySkipping(InputStream inputStream) {
        long l2;
        long l3 = 0L;
        while ((l2 = ByteStreams.skipUpTo(inputStream, Integer.MAX_VALUE)) > 0L) {
            l3 += l2;
        }
        return l3;
    }

    @CanIgnoreReturnValue
    public long copyTo(OutputStream outputStream) {
        Preconditions.checkNotNull(outputStream);
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            long l2 = ByteStreams.copy(inputStream, outputStream);
            return l2;
        }
    }

    @CanIgnoreReturnValue
    public long copyTo(ByteSink byteSink) {
        Preconditions.checkNotNull(byteSink);
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            OutputStream outputStream = closer.register(byteSink.openStream());
            long l2 = ByteStreams.copy(inputStream, outputStream);
            return l2;
        }
    }

    public byte[] read() {
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            Optional<Long> optional = this.sizeIfKnown();
            byte[] byArray = optional.isPresent() ? ByteStreams.toByteArray(inputStream, optional.get()) : ByteStreams.toByteArray(inputStream);
            return byArray;
        }
    }

    @Beta
    @CanIgnoreReturnValue
    public <T> T read(ByteProcessor<T> byteProcessor) {
        Preconditions.checkNotNull(byteProcessor);
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            T t2 = ByteStreams.readBytes(inputStream, byteProcessor);
            return t2;
        }
    }

    public HashCode hash(HashFunction hashFunction) {
        Hasher hasher = hashFunction.newHasher();
        this.copyTo(Funnels.asOutputStream(hasher));
        return hasher.hash();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean contentEquals(ByteSource byteSource) {
        Preconditions.checkNotNull(byteSource);
        byte[] byArray = ByteStreams.createBuffer();
        byte[] byArray2 = ByteStreams.createBuffer();
        try (Closer closer = Closer.create();){
            InputStream inputStream = closer.register(this.openStream());
            InputStream inputStream2 = closer.register(byteSource.openStream());
            while (true) {
                int n2;
                int n3;
                if ((n3 = ByteStreams.read(inputStream, byArray, 0, byArray.length)) != (n2 = ByteStreams.read(inputStream2, byArray2, 0, byArray2.length)) || !Arrays.equals(byArray, byArray2)) {
                    boolean bl2 = false;
                    return bl2;
                }
                if (n3 != byArray.length) {
                    boolean bl3 = true;
                    return bl3;
                }
                continue;
                break;
            }
        }
    }

    public static ByteSource concat(Iterable<? extends ByteSource> iterable) {
        return new ConcatenatedByteSource(iterable);
    }

    public static ByteSource concat(Iterator<? extends ByteSource> iterator) {
        return ByteSource.concat(ImmutableList.copyOf(iterator));
    }

    public static ByteSource concat(ByteSource ... byteSourceArray) {
        return ByteSource.concat(ImmutableList.copyOf(byteSourceArray));
    }

    public static ByteSource wrap(byte[] byArray) {
        return new ByteArrayByteSource(byArray);
    }

    public static ByteSource empty() {
        return EmptyByteSource.INSTANCE;
    }

    private static final class ConcatenatedByteSource
    extends ByteSource {
        final Iterable<? extends ByteSource> sources;

        ConcatenatedByteSource(Iterable<? extends ByteSource> iterable) {
            this.sources = Preconditions.checkNotNull(iterable);
        }

        @Override
        public InputStream openStream() {
            return new MultiInputStream(this.sources.iterator());
        }

        @Override
        public boolean isEmpty() {
            for (ByteSource byteSource : this.sources) {
                if (byteSource.isEmpty()) continue;
                return false;
            }
            return true;
        }

        @Override
        public Optional<Long> sizeIfKnown() {
            if (!(this.sources instanceof Collection)) {
                return Optional.absent();
            }
            long l2 = 0L;
            for (ByteSource byteSource : this.sources) {
                Optional<Long> optional = byteSource.sizeIfKnown();
                if (!optional.isPresent()) {
                    return Optional.absent();
                }
                if ((l2 += optional.get().longValue()) >= 0L) continue;
                return Optional.of(Long.MAX_VALUE);
            }
            return Optional.of(l2);
        }

        @Override
        public long size() {
            long l2 = 0L;
            for (ByteSource byteSource : this.sources) {
                if ((l2 += byteSource.size()) >= 0L) continue;
                return Long.MAX_VALUE;
            }
            return l2;
        }

        public String toString() {
            return "ByteSource.concat(" + this.sources + ")";
        }
    }

    private static final class EmptyByteSource
    extends ByteArrayByteSource {
        static final EmptyByteSource INSTANCE = new EmptyByteSource();

        EmptyByteSource() {
            super(new byte[0]);
        }

        @Override
        public CharSource asCharSource(Charset charset) {
            Preconditions.checkNotNull(charset);
            return CharSource.empty();
        }

        @Override
        public byte[] read() {
            return this.bytes;
        }

        @Override
        public String toString() {
            return "ByteSource.empty()";
        }
    }

    private static class ByteArrayByteSource
    extends ByteSource {
        final byte[] bytes;
        final int offset;
        final int length;

        ByteArrayByteSource(byte[] byArray) {
            this(byArray, 0, byArray.length);
        }

        ByteArrayByteSource(byte[] byArray, int n2, int n3) {
            this.bytes = byArray;
            this.offset = n2;
            this.length = n3;
        }

        @Override
        public InputStream openStream() {
            return new ByteArrayInputStream(this.bytes, this.offset, this.length);
        }

        @Override
        public InputStream openBufferedStream() {
            return this.openStream();
        }

        @Override
        public boolean isEmpty() {
            return this.length == 0;
        }

        @Override
        public long size() {
            return this.length;
        }

        @Override
        public Optional<Long> sizeIfKnown() {
            return Optional.of(Long.valueOf(this.length));
        }

        @Override
        public byte[] read() {
            return Arrays.copyOfRange(this.bytes, this.offset, this.offset + this.length);
        }

        @Override
        public <T> T read(ByteProcessor<T> byteProcessor) {
            byteProcessor.processBytes(this.bytes, this.offset, this.length);
            return byteProcessor.getResult();
        }

        @Override
        public long copyTo(OutputStream outputStream) {
            outputStream.write(this.bytes, this.offset, this.length);
            return this.length;
        }

        @Override
        public HashCode hash(HashFunction hashFunction) {
            return hashFunction.hashBytes(this.bytes, this.offset, this.length);
        }

        @Override
        public ByteSource slice(long l2, long l3) {
            Preconditions.checkArgument(l2 >= 0L, "offset (%s) may not be negative", l2);
            Preconditions.checkArgument(l3 >= 0L, "length (%s) may not be negative", l3);
            l2 = Math.min(l2, (long)this.length);
            l3 = Math.min(l3, (long)this.length - l2);
            int n2 = this.offset + (int)l2;
            return new ByteArrayByteSource(this.bytes, n2, (int)l3);
        }

        public String toString() {
            return "ByteSource.wrap(" + Ascii.truncate(BaseEncoding.base16().encode(this.bytes, this.offset, this.length), 30, "...") + ")";
        }
    }

    private final class SlicedByteSource
    extends ByteSource {
        final long offset;
        final long length;

        SlicedByteSource(long l2, long l3) {
            Preconditions.checkArgument(l2 >= 0L, "offset (%s) may not be negative", l2);
            Preconditions.checkArgument(l3 >= 0L, "length (%s) may not be negative", l3);
            this.offset = l2;
            this.length = l3;
        }

        @Override
        public InputStream openStream() {
            return this.sliceStream(ByteSource.this.openStream());
        }

        @Override
        public InputStream openBufferedStream() {
            return this.sliceStream(ByteSource.this.openBufferedStream());
        }

        private InputStream sliceStream(InputStream inputStream) {
            if (this.offset > 0L) {
                long l2;
                try {
                    l2 = ByteStreams.skipUpTo(inputStream, this.offset);
                }
                catch (Throwable throwable) {
                    Closer closer = Closer.create();
                    closer.register(inputStream);
                    try {
                        throw closer.rethrow(throwable);
                    }
                    catch (Throwable throwable2) {
                        closer.close();
                        throw throwable2;
                    }
                }
                if (l2 < this.offset) {
                    inputStream.close();
                    return new ByteArrayInputStream(new byte[0]);
                }
            }
            return ByteStreams.limit(inputStream, this.length);
        }

        @Override
        public ByteSource slice(long l2, long l3) {
            Preconditions.checkArgument(l2 >= 0L, "offset (%s) may not be negative", l2);
            Preconditions.checkArgument(l3 >= 0L, "length (%s) may not be negative", l3);
            long l4 = this.length - l2;
            return ByteSource.this.slice(this.offset + l2, Math.min(l3, l4));
        }

        @Override
        public boolean isEmpty() {
            return this.length == 0L || super.isEmpty();
        }

        @Override
        public Optional<Long> sizeIfKnown() {
            Optional<Long> optional = ByteSource.this.sizeIfKnown();
            if (optional.isPresent()) {
                long l2 = optional.get();
                long l3 = Math.min(this.offset, l2);
                return Optional.of(Math.min(this.length, l2 - l3));
            }
            return Optional.absent();
        }

        public String toString() {
            return ByteSource.this.toString() + ".slice(" + this.offset + ", " + this.length + ")";
        }
    }

    class AsCharSource
    extends CharSource {
        final Charset charset;

        AsCharSource(Charset charset) {
            this.charset = Preconditions.checkNotNull(charset);
        }

        @Override
        public ByteSource asByteSource(Charset charset) {
            if (charset.equals(this.charset)) {
                return ByteSource.this;
            }
            return super.asByteSource(charset);
        }

        @Override
        public Reader openStream() {
            return new InputStreamReader(ByteSource.this.openStream(), this.charset);
        }

        @Override
        public String read() {
            return new String(ByteSource.this.read(), this.charset);
        }

        public String toString() {
            return ByteSource.this.toString() + ".asCharSource(" + this.charset + ")";
        }
    }
}

