/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.annotations.Beta;
import com.google.common.base.Preconditions;
import com.google.common.hash.AbstractCompositeHashFunction;
import com.google.common.hash.ChecksumHashFunction;
import com.google.common.hash.Crc32cHashFunction;
import com.google.common.hash.FarmHashFingerprint64;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.ImmutableSupplier;
import com.google.common.hash.MacHashFunction;
import com.google.common.hash.MessageDigestHashFunction;
import com.google.common.hash.Murmur3_128HashFunction;
import com.google.common.hash.Murmur3_32HashFunction;
import com.google.common.hash.SipHashFunction;
import com.google.errorprone.annotations.Immutable;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.zip.Adler32;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import javax.crypto.spec.SecretKeySpec;
import org.checkerframework.checker.nullness.qual.Nullable;

@Beta
public final class Hashing {
    static final int GOOD_FAST_HASH_SEED = (int)System.currentTimeMillis();

    public static HashFunction goodFastHash(int n2) {
        int n3 = Hashing.checkPositiveAndMakeMultipleOf32(n2);
        if (n3 == 32) {
            return Murmur3_32HashFunction.GOOD_FAST_HASH_32;
        }
        if (n3 <= 128) {
            return Murmur3_128HashFunction.GOOD_FAST_HASH_128;
        }
        int n4 = (n3 + 127) / 128;
        HashFunction[] hashFunctionArray = new HashFunction[n4];
        hashFunctionArray[0] = Murmur3_128HashFunction.GOOD_FAST_HASH_128;
        int n5 = GOOD_FAST_HASH_SEED;
        for (int i2 = 1; i2 < n4; ++i2) {
            hashFunctionArray[i2] = Hashing.murmur3_128(n5 += 1500450271);
        }
        return new ConcatenatedHashFunction(hashFunctionArray);
    }

    public static HashFunction murmur3_32(int n2) {
        return new Murmur3_32HashFunction(n2);
    }

    public static HashFunction murmur3_32() {
        return Murmur3_32HashFunction.MURMUR3_32;
    }

    public static HashFunction murmur3_128(int n2) {
        return new Murmur3_128HashFunction(n2);
    }

    public static HashFunction murmur3_128() {
        return Murmur3_128HashFunction.MURMUR3_128;
    }

    public static HashFunction sipHash24() {
        return SipHashFunction.SIP_HASH_24;
    }

    public static HashFunction sipHash24(long l2, long l3) {
        return new SipHashFunction(2, 4, l2, l3);
    }

    @Deprecated
    public static HashFunction md5() {
        return Md5Holder.MD5;
    }

    @Deprecated
    public static HashFunction sha1() {
        return Sha1Holder.SHA_1;
    }

    public static HashFunction sha256() {
        return Sha256Holder.SHA_256;
    }

    public static HashFunction sha384() {
        return Sha384Holder.SHA_384;
    }

    public static HashFunction sha512() {
        return Sha512Holder.SHA_512;
    }

    public static HashFunction hmacMd5(Key key) {
        return new MacHashFunction("HmacMD5", key, Hashing.hmacToString("hmacMd5", key));
    }

    public static HashFunction hmacMd5(byte[] byArray) {
        return Hashing.hmacMd5(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacMD5"));
    }

    public static HashFunction hmacSha1(Key key) {
        return new MacHashFunction("HmacSHA1", key, Hashing.hmacToString("hmacSha1", key));
    }

    public static HashFunction hmacSha1(byte[] byArray) {
        return Hashing.hmacSha1(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA1"));
    }

    public static HashFunction hmacSha256(Key key) {
        return new MacHashFunction("HmacSHA256", key, Hashing.hmacToString("hmacSha256", key));
    }

    public static HashFunction hmacSha256(byte[] byArray) {
        return Hashing.hmacSha256(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA256"));
    }

    public static HashFunction hmacSha512(Key key) {
        return new MacHashFunction("HmacSHA512", key, Hashing.hmacToString("hmacSha512", key));
    }

    public static HashFunction hmacSha512(byte[] byArray) {
        return Hashing.hmacSha512(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA512"));
    }

    private static String hmacToString(String string, Key key) {
        return String.format("Hashing.%s(Key[algorithm=%s, format=%s])", string, key.getAlgorithm(), key.getFormat());
    }

    public static HashFunction crc32c() {
        return Crc32cHashFunction.CRC_32_C;
    }

    public static HashFunction crc32() {
        return ChecksumType.CRC_32.hashFunction;
    }

    public static HashFunction adler32() {
        return ChecksumType.ADLER_32.hashFunction;
    }

    public static HashFunction farmHashFingerprint64() {
        return FarmHashFingerprint64.FARMHASH_FINGERPRINT_64;
    }

    public static int consistentHash(HashCode hashCode, int n2) {
        return Hashing.consistentHash(hashCode.padToLong(), n2);
    }

    public static int consistentHash(long l2, int n2) {
        int n3;
        Preconditions.checkArgument(n2 > 0, "buckets must be positive: %s", n2);
        LinearCongruentialGenerator linearCongruentialGenerator = new LinearCongruentialGenerator(l2);
        int n4 = 0;
        while ((n3 = (int)((double)(n4 + 1) / linearCongruentialGenerator.nextDouble())) >= 0 && n3 < n2) {
            n4 = n3;
        }
        return n4;
    }

    public static HashCode combineOrdered(Iterable<HashCode> iterable) {
        Iterator<HashCode> iterator = iterable.iterator();
        Preconditions.checkArgument(iterator.hasNext(), "Must be at least 1 hash code to combine.");
        int n2 = iterator.next().bits();
        byte[] byArray = new byte[n2 / 8];
        for (HashCode hashCode : iterable) {
            byte[] byArray2 = hashCode.asBytes();
            Preconditions.checkArgument(byArray2.length == byArray.length, "All hashcodes must have the same bit length.");
            for (int i2 = 0; i2 < byArray2.length; ++i2) {
                byArray[i2] = (byte)(byArray[i2] * 37 ^ byArray2[i2]);
            }
        }
        return HashCode.fromBytesNoCopy(byArray);
    }

    public static HashCode combineUnordered(Iterable<HashCode> iterable) {
        Iterator<HashCode> iterator = iterable.iterator();
        Preconditions.checkArgument(iterator.hasNext(), "Must be at least 1 hash code to combine.");
        byte[] byArray = new byte[iterator.next().bits() / 8];
        for (HashCode hashCode : iterable) {
            byte[] byArray2 = hashCode.asBytes();
            Preconditions.checkArgument(byArray2.length == byArray.length, "All hashcodes must have the same bit length.");
            for (int i2 = 0; i2 < byArray2.length; ++i2) {
                int n2 = i2;
                byArray[n2] = (byte)(byArray[n2] + byArray2[i2]);
            }
        }
        return HashCode.fromBytesNoCopy(byArray);
    }

    static int checkPositiveAndMakeMultipleOf32(int n2) {
        Preconditions.checkArgument(n2 > 0, "Number of bits must be positive");
        return n2 + 31 & 0xFFFFFFE0;
    }

    public static HashFunction concatenating(HashFunction hashFunction, HashFunction hashFunction2, HashFunction ... hashFunctionArray) {
        ArrayList<HashFunction> arrayList = new ArrayList<HashFunction>();
        arrayList.add(hashFunction);
        arrayList.add(hashFunction2);
        arrayList.addAll(Arrays.asList(hashFunctionArray));
        return new ConcatenatedHashFunction(arrayList.toArray(new HashFunction[0]));
    }

    public static HashFunction concatenating(Iterable<HashFunction> iterable) {
        Preconditions.checkNotNull(iterable);
        ArrayList<HashFunction> arrayList = new ArrayList<HashFunction>();
        for (HashFunction hashFunction : iterable) {
            arrayList.add(hashFunction);
        }
        Preconditions.checkArgument(arrayList.size() > 0, "number of hash functions (%s) must be > 0", arrayList.size());
        return new ConcatenatedHashFunction(arrayList.toArray(new HashFunction[0]));
    }

    private Hashing() {
    }

    private static final class LinearCongruentialGenerator {
        private long state;

        public LinearCongruentialGenerator(long l2) {
            this.state = l2;
        }

        public double nextDouble() {
            this.state = 2862933555777941757L * this.state + 1L;
            return (double)((int)(this.state >>> 33) + 1) / 2.147483648E9;
        }
    }

    private static final class ConcatenatedHashFunction
    extends AbstractCompositeHashFunction {
        private ConcatenatedHashFunction(HashFunction ... hashFunctionArray) {
            super(hashFunctionArray);
            for (HashFunction hashFunction : hashFunctionArray) {
                Preconditions.checkArgument(hashFunction.bits() % 8 == 0, "the number of bits (%s) in hashFunction (%s) must be divisible by 8", hashFunction.bits(), (Object)hashFunction);
            }
        }

        @Override
        HashCode makeHash(Hasher[] hasherArray) {
            byte[] byArray = new byte[this.bits() / 8];
            int n2 = 0;
            for (Hasher hasher : hasherArray) {
                HashCode hashCode = hasher.hash();
                n2 += hashCode.writeBytesTo(byArray, n2, hashCode.bits() / 8);
            }
            return HashCode.fromBytesNoCopy(byArray);
        }

        @Override
        public int bits() {
            int n2 = 0;
            for (HashFunction hashFunction : this.functions) {
                n2 += hashFunction.bits();
            }
            return n2;
        }

        public boolean equals(@Nullable Object object) {
            if (object instanceof ConcatenatedHashFunction) {
                ConcatenatedHashFunction concatenatedHashFunction = (ConcatenatedHashFunction)object;
                return Arrays.equals(this.functions, concatenatedHashFunction.functions);
            }
            return false;
        }

        public int hashCode() {
            return Arrays.hashCode(this.functions);
        }
    }

    @Immutable
    static enum ChecksumType implements ImmutableSupplier<Checksum>
    {
        CRC_32("Hashing.crc32()"){

            @Override
            public Checksum get() {
                return new CRC32();
            }
        }
        ,
        ADLER_32("Hashing.adler32()"){

            @Override
            public Checksum get() {
                return new Adler32();
            }
        };

        public final HashFunction hashFunction;

        private ChecksumType(String string2) {
            this.hashFunction = new ChecksumHashFunction(this, 32, string2);
        }
    }

    private static class Sha512Holder {
        static final HashFunction SHA_512 = new MessageDigestHashFunction("SHA-512", "Hashing.sha512()");

        private Sha512Holder() {
        }
    }

    private static class Sha384Holder {
        static final HashFunction SHA_384 = new MessageDigestHashFunction("SHA-384", "Hashing.sha384()");

        private Sha384Holder() {
        }
    }

    private static class Sha256Holder {
        static final HashFunction SHA_256 = new MessageDigestHashFunction("SHA-256", "Hashing.sha256()");

        private Sha256Holder() {
        }
    }

    private static class Sha1Holder {
        static final HashFunction SHA_1 = new MessageDigestHashFunction("SHA-1", "Hashing.sha1()");

        private Sha1Holder() {
        }
    }

    private static class Md5Holder {
        static final HashFunction MD5 = new MessageDigestHashFunction("MD5", "Hashing.md5()");

        private Md5Holder() {
        }
    }
}

