/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.hash.BloomFilterStrategies;
import com.google.common.hash.Funnel;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.SignedBytes;
import com.google.common.primitives.UnsignedBytes;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.RoundingMode;
import java.util.stream.Collector;
import org.checkerframework.checker.nullness.qual.Nullable;

@Beta
public final class BloomFilter<T>
implements Predicate<T>,
Serializable {
    private final BloomFilterStrategies.LockFreeBitArray bits;
    private final int numHashFunctions;
    private final Funnel<? super T> funnel;
    private final Strategy strategy;

    private BloomFilter(BloomFilterStrategies.LockFreeBitArray lockFreeBitArray, int n2, Funnel<? super T> funnel, Strategy strategy) {
        Preconditions.checkArgument(n2 > 0, "numHashFunctions (%s) must be > 0", n2);
        Preconditions.checkArgument(n2 <= 255, "numHashFunctions (%s) must be <= 255", n2);
        this.bits = Preconditions.checkNotNull(lockFreeBitArray);
        this.numHashFunctions = n2;
        this.funnel = Preconditions.checkNotNull(funnel);
        this.strategy = Preconditions.checkNotNull(strategy);
    }

    public BloomFilter<T> copy() {
        return new BloomFilter<T>(this.bits.copy(), this.numHashFunctions, this.funnel, this.strategy);
    }

    public boolean mightContain(T t2) {
        return this.strategy.mightContain(t2, this.funnel, this.numHashFunctions, this.bits);
    }

    @Override
    @Deprecated
    public boolean apply(T t2) {
        return this.mightContain(t2);
    }

    @CanIgnoreReturnValue
    public boolean put(T t2) {
        return this.strategy.put(t2, this.funnel, this.numHashFunctions, this.bits);
    }

    public double expectedFpp() {
        return Math.pow((double)this.bits.bitCount() / (double)this.bitSize(), this.numHashFunctions);
    }

    public long approximateElementCount() {
        long l2 = this.bits.bitSize();
        long l3 = this.bits.bitCount();
        double d2 = (double)l3 / (double)l2;
        return DoubleMath.roundToLong(-Math.log1p(-d2) * (double)l2 / (double)this.numHashFunctions, RoundingMode.HALF_UP);
    }

    @VisibleForTesting
    long bitSize() {
        return this.bits.bitSize();
    }

    public boolean isCompatible(BloomFilter<T> bloomFilter) {
        Preconditions.checkNotNull(bloomFilter);
        return this != bloomFilter && this.numHashFunctions == bloomFilter.numHashFunctions && this.bitSize() == bloomFilter.bitSize() && this.strategy.equals(bloomFilter.strategy) && this.funnel.equals(bloomFilter.funnel);
    }

    public void putAll(BloomFilter<T> bloomFilter) {
        Preconditions.checkNotNull(bloomFilter);
        Preconditions.checkArgument(this != bloomFilter, "Cannot combine a BloomFilter with itself.");
        Preconditions.checkArgument(this.numHashFunctions == bloomFilter.numHashFunctions, "BloomFilters must have the same number of hash functions (%s != %s)", this.numHashFunctions, bloomFilter.numHashFunctions);
        Preconditions.checkArgument(this.bitSize() == bloomFilter.bitSize(), "BloomFilters must have the same size underlying bit arrays (%s != %s)", this.bitSize(), bloomFilter.bitSize());
        Preconditions.checkArgument(this.strategy.equals(bloomFilter.strategy), "BloomFilters must have equal strategies (%s != %s)", (Object)this.strategy, (Object)bloomFilter.strategy);
        Preconditions.checkArgument(this.funnel.equals(bloomFilter.funnel), "BloomFilters must have equal funnels (%s != %s)", this.funnel, bloomFilter.funnel);
        this.bits.putAll(bloomFilter.bits);
    }

    @Override
    public boolean equals(@Nullable Object object) {
        if (object == this) {
            return true;
        }
        if (object instanceof BloomFilter) {
            BloomFilter bloomFilter = (BloomFilter)object;
            return this.numHashFunctions == bloomFilter.numHashFunctions && this.funnel.equals(bloomFilter.funnel) && this.bits.equals(bloomFilter.bits) && this.strategy.equals(bloomFilter.strategy);
        }
        return false;
    }

    public int hashCode() {
        return Objects.hashCode(this.numHashFunctions, this.funnel, this.strategy, this.bits);
    }

    public static <T> Collector<T, ?, BloomFilter<T>> toBloomFilter(Funnel<? super T> funnel, long l2) {
        return BloomFilter.toBloomFilter(funnel, l2, 0.03);
    }

    public static <T> Collector<T, ?, BloomFilter<T>> toBloomFilter(Funnel<? super T> funnel, long l2, double d2) {
        Preconditions.checkNotNull(funnel);
        Preconditions.checkArgument(l2 >= 0L, "Expected insertions (%s) must be >= 0", l2);
        Preconditions.checkArgument(d2 > 0.0, "False positive probability (%s) must be > 0.0", (Object)d2);
        Preconditions.checkArgument(d2 < 1.0, "False positive probability (%s) must be < 1.0", (Object)d2);
        return Collector.of(() -> BloomFilter.create(funnel, l2, d2), BloomFilter::put, (bloomFilter, bloomFilter2) -> {
            bloomFilter.putAll((BloomFilter)bloomFilter2);
            return bloomFilter;
        }, Collector.Characteristics.UNORDERED, Collector.Characteristics.CONCURRENT);
    }

    public static <T> BloomFilter<T> create(Funnel<? super T> funnel, int n2, double d2) {
        return BloomFilter.create(funnel, (long)n2, d2);
    }

    public static <T> BloomFilter<T> create(Funnel<? super T> funnel, long l2, double d2) {
        return BloomFilter.create(funnel, l2, d2, BloomFilterStrategies.MURMUR128_MITZ_64);
    }

    @VisibleForTesting
    static <T> BloomFilter<T> create(Funnel<? super T> funnel, long l2, double d2, Strategy strategy) {
        Preconditions.checkNotNull(funnel);
        Preconditions.checkArgument(l2 >= 0L, "Expected insertions (%s) must be >= 0", l2);
        Preconditions.checkArgument(d2 > 0.0, "False positive probability (%s) must be > 0.0", (Object)d2);
        Preconditions.checkArgument(d2 < 1.0, "False positive probability (%s) must be < 1.0", (Object)d2);
        Preconditions.checkNotNull(strategy);
        if (l2 == 0L) {
            l2 = 1L;
        }
        long l3 = BloomFilter.optimalNumOfBits(l2, d2);
        int n2 = BloomFilter.optimalNumOfHashFunctions(l2, l3);
        try {
            return new BloomFilter<T>(new BloomFilterStrategies.LockFreeBitArray(l3), n2, funnel, strategy);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            throw new IllegalArgumentException("Could not create BloomFilter of " + l3 + " bits", illegalArgumentException);
        }
    }

    public static <T> BloomFilter<T> create(Funnel<? super T> funnel, int n2) {
        return BloomFilter.create(funnel, (long)n2);
    }

    public static <T> BloomFilter<T> create(Funnel<? super T> funnel, long l2) {
        return BloomFilter.create(funnel, l2, 0.03);
    }

    @VisibleForTesting
    static int optimalNumOfHashFunctions(long l2, long l3) {
        return Math.max(1, (int)Math.round((double)l3 / (double)l2 * Math.log(2.0)));
    }

    @VisibleForTesting
    static long optimalNumOfBits(long l2, double d2) {
        if (d2 == 0.0) {
            d2 = Double.MIN_VALUE;
        }
        return (long)((double)(-l2) * Math.log(d2) / (Math.log(2.0) * Math.log(2.0)));
    }

    private Object writeReplace() {
        return new SerialForm(this);
    }

    public void writeTo(OutputStream outputStream) {
        DataOutputStream dataOutputStream = new DataOutputStream(outputStream);
        dataOutputStream.writeByte(SignedBytes.checkedCast(this.strategy.ordinal()));
        dataOutputStream.writeByte(UnsignedBytes.checkedCast(this.numHashFunctions));
        dataOutputStream.writeInt(this.bits.data.length());
        for (int i2 = 0; i2 < this.bits.data.length(); ++i2) {
            dataOutputStream.writeLong(this.bits.data.get(i2));
        }
    }

    public static <T> BloomFilter<T> readFrom(InputStream inputStream, Funnel<? super T> funnel) {
        Preconditions.checkNotNull(inputStream, "InputStream");
        Preconditions.checkNotNull(funnel, "Funnel");
        int n2 = -1;
        int n3 = -1;
        int n4 = -1;
        try {
            DataInputStream dataInputStream = new DataInputStream(inputStream);
            n2 = dataInputStream.readByte();
            n3 = UnsignedBytes.toInt(dataInputStream.readByte());
            n4 = dataInputStream.readInt();
            BloomFilterStrategies bloomFilterStrategies = BloomFilterStrategies.values()[n2];
            long[] lArray = new long[n4];
            for (int i2 = 0; i2 < lArray.length; ++i2) {
                lArray[i2] = dataInputStream.readLong();
            }
            return new BloomFilter<T>(new BloomFilterStrategies.LockFreeBitArray(lArray), n3, funnel, bloomFilterStrategies);
        }
        catch (RuntimeException runtimeException) {
            String string = "Unable to deserialize BloomFilter from InputStream. strategyOrdinal: " + n2 + " numHashFunctions: " + n3 + " dataLength: " + n4;
            throw new IOException(string, runtimeException);
        }
    }

    private static class SerialForm<T>
    implements Serializable {
        final long[] data;
        final int numHashFunctions;
        final Funnel<? super T> funnel;
        final Strategy strategy;
        private static final long serialVersionUID = 1L;

        SerialForm(BloomFilter<T> bloomFilter) {
            this.data = BloomFilterStrategies.LockFreeBitArray.toPlainArray(((BloomFilter)bloomFilter).bits.data);
            this.numHashFunctions = ((BloomFilter)bloomFilter).numHashFunctions;
            this.funnel = ((BloomFilter)bloomFilter).funnel;
            this.strategy = ((BloomFilter)bloomFilter).strategy;
        }

        Object readResolve() {
            return new BloomFilter(new BloomFilterStrategies.LockFreeBitArray(this.data), this.numHashFunctions, this.funnel, this.strategy);
        }
    }

    static interface Strategy
    extends Serializable {
        public <T> boolean put(T var1, Funnel<? super T> var2, int var3, BloomFilterStrategies.LockFreeBitArray var4);

        public <T> boolean mightContain(T var1, Funnel<? super T> var2, int var3, BloomFilterStrategies.LockFreeBitArray var4);

        public int ordinal();
    }
}

