# frozen_string_literal: true

module Wpxf
  # Provides functionality for specifying module metadata.
  module ModuleInfo
    # Initialize a new instance of {ModuleInfo}.
    def initialize
      super
      @_module_info = {}
    end

    # Update the module info.
    # @param info [Hash] a hash containing the module info.
    def update_info(info)
      required_keys = %i[name desc author date]
      unless required_keys.all? { |key| info.key?(key) || @_module_info.key?(key) }
        raise 'Missing one or more required module info keys'
      end

      _update_info_without_validation(info)
    end

    # @return [String] the name of the module.
    def module_name
      @_module_info[:name]
    end

    # @return [String] the description of the module.
    def module_desc
      @_module_info[:desc]
    end

    # @return [Array] an aray of references relating to the module.
    def module_references
      @_module_info[:references]
    end

    # @return [Array] the name of the module author(s).
    def module_author
      @_module_info[:author]
    end

    # @return [Date] the disclosure date of the vulnerability.
    def module_date
      @_module_info[:date]
    end

    # @return [Boolean] true if the description is preformatted.
    def module_description_preformatted
      @_module_info[:desc_preformatted]
    end

    # Emits any information that the user should be aware of before using the module.
    def emit_usage_info
      nil
    end

    private

    def _update_info_without_validation(info)
      @_module_info.merge!(info)

      if @_module_info[:date]
        @_module_info[:date] = Date.parse(@_module_info[:date].to_s)
      end

      if @_module_info[:desc]
        @_module_info[:desc] = @_module_info[:desc].gsub(/  +/, ' ')
      end

      @_module_info
    end
  end
end
