/*
 * Decompiled with CFR 0.152.
 */
package io.github.dsheirer.dsp.window;

import io.github.dsheirer.dsp.window.WindowType;
import java.util.Arrays;
import org.apache.commons.math3.util.FastMath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WindowFactory {
    private static final Logger mLog = LoggerFactory.getLogger(WindowFactory.class);
    private static final float PI = (float)Math.PI;
    private static final float TWO_PI = (float)Math.PI * 2;
    private static final float FOUR_PI = (float)Math.PI * 4;
    private static final float SIX_PI = (float)Math.PI * 6;
    private static final float EIGHT_PI = (float)Math.PI * 8;

    public static float[] getWindow(WindowType type, int length) {
        switch (type) {
            case BLACKMAN: {
                return WindowFactory.getBlackman(length);
            }
            case BLACKMAN_HARRIS_4: {
                return WindowFactory.getBlackmanHarris4(length);
            }
            case BLACKMAN_HARRIS_7: {
                return WindowFactory.getBlackmanHarris7(length);
            }
            case BLACKMAN_NUTALL: {
                return WindowFactory.getBlackmanNutall(length);
            }
            case COSINE: {
                return WindowFactory.getCosine(length);
            }
            case FLAT_TOP: {
                return WindowFactory.getFlatTop(length);
            }
            case HAMMING: {
                return WindowFactory.getHamming(length);
            }
            case HANN: {
                return WindowFactory.getHann(length);
            }
            case KAISER: {
                throw new IllegalArgumentException("Kaiser Window cannot be created via this method.  Use the getKaiser() method instead.");
            }
            case NUTALL: {
                return WindowFactory.getNutall(length);
            }
        }
        return WindowFactory.getRectangular(length);
    }

    public static float[] getRectangular(int length) {
        float[] coefficients = new float[length];
        Arrays.fill(coefficients, 1.0f);
        return coefficients;
    }

    public static float[] getCosine(int length) {
        float[] coefficients = new float[length];
        if (length % 2 == 0) {
            int half = (length - 1) / 2;
            for (int x = -half; x < length / 2 + 1; ++x) {
                coefficients[x + half] = (float)FastMath.cos((double)((float)x * (float)Math.PI / ((float)length + 1.0f)));
            }
        } else {
            int half = length / 2;
            for (int x = -half; x < half + 1; ++x) {
                coefficients[x + half] = (float)FastMath.cos((double)((float)x * (float)Math.PI / ((float)length + 1.0f)));
            }
        }
        return coefficients;
    }

    public static float[] getBlackman(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.4265907f;
        float a1 = 0.49656063f;
        float a2 = 0.07684867f;
        for (int x = 0; x < length; ++x) {
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / denominator)) + a2 * (float)FastMath.cos((double)((float)Math.PI * 4 * (float)x / denominator));
        }
        return coefficients;
    }

    public static float[] getNutall(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.355768f;
        float a1 = 0.487396f;
        float a2 = 0.144232f;
        float a3 = 0.012604f;
        for (int x = 0; x < length; ++x) {
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / denominator)) + a2 * (float)FastMath.cos((double)((float)Math.PI * 4 * (float)x / denominator)) - a3 * (float)FastMath.cos((double)((float)Math.PI * 6 * (float)x / denominator));
        }
        return coefficients;
    }

    public static float[] getBlackmanNutall(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.3635819f;
        float a1 = 0.4891775f;
        float a2 = 0.1365995f;
        float a3 = 0.0106411f;
        for (int x = 0; x < length; ++x) {
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / denominator)) + a2 * (float)FastMath.cos((double)((float)Math.PI * 4 * (float)x / denominator)) - a3 * (float)FastMath.cos((double)((float)Math.PI * 6 * (float)x / denominator));
        }
        return coefficients;
    }

    public static float[] getBlackmanHarris4(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.35875f;
        float a1 = 0.48829f;
        float a2 = 0.14128f;
        float a3 = 0.01168f;
        for (int x = 0; x < length; ++x) {
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / denominator)) + a2 * (float)FastMath.cos((double)((float)Math.PI * 4 * (float)x / denominator)) - a3 * (float)FastMath.cos((double)((float)Math.PI * 6 * (float)x / denominator));
        }
        return coefficients;
    }

    public static float[] getBlackmanHarris7(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.2710514f;
        float a1 = 0.43329793f;
        float a2 = 0.218123f;
        float a3 = 0.06592545f;
        float a4 = 0.010811742f;
        float a5 = 7.7658484E-4f;
        float a6 = 1.3887217E-5f;
        for (int x = 0; x < length; ++x) {
            float w = (float)Math.PI * 2 * (float)x / denominator;
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)w) + a2 * (float)FastMath.cos((double)(2.0 * (double)w)) - a3 * (float)FastMath.cos((double)(3.0 * (double)w)) + a4 * (float)FastMath.cos((double)(4.0 * (double)w)) - a5 * (float)FastMath.cos((double)(5.0 * (double)w)) + a6 * (float)FastMath.cos((double)(6.0 * (double)w));
        }
        return coefficients;
    }

    public static float[] getHamming(int length) {
        float[] coefficients = new float[length];
        for (int x = 0; x < length; ++x) {
            coefficients[x] = 0.54f - 0.46f * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / (float)(length - 1)));
        }
        return coefficients;
    }

    public static float[] getHann(int length) {
        float[] coefficients = new float[length];
        for (int x = 0; x < length; ++x) {
            coefficients[x] = 0.5f - 0.5f * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / (float)(length - 1)));
        }
        return coefficients;
    }

    public static float[] getFlatTop(int length) {
        float[] coefficients = new float[length];
        float denominator = length - 1;
        float a0 = 0.21557894f;
        float a1 = 0.41663158f;
        float a2 = 0.27726316f;
        float a3 = 0.083578944f;
        float a4 = 0.006947368f;
        for (int x = 0; x < length; ++x) {
            coefficients[x] = a0 - a1 * (float)FastMath.cos((double)((float)Math.PI * 2 * (float)x / denominator)) + a2 * (float)FastMath.cos((double)((float)Math.PI * 4 * (float)x / denominator)) - a3 * (float)FastMath.cos((double)((float)Math.PI * 6 * (float)x / denominator)) + a4 * (float)FastMath.cos((double)((float)Math.PI * 8 * (float)x / denominator));
        }
        return coefficients;
    }

    public static float getKaiserBeta(float attenuation) {
        if ((double)attenuation > 50.0) {
            return 0.1102f * (attenuation - 8.7f);
        }
        if ((double)attenuation >= 21.0) {
            return 0.5842f * (float)FastMath.pow((double)((double)attenuation - 21.0), (double)0.4) + 0.07886f * (attenuation - 21.0f);
        }
        return 0.0f;
    }

    public static float[] getKaiser(int length, float attenuation) {
        float[] coefficients = new float[length];
        float beta = WindowFactory.getKaiserBeta(attenuation);
        float betaBesselZerothOrder = WindowFactory.getBesselZerothOrder(beta);
        for (int x = 0; x < coefficients.length; ++x) {
            float temp = beta * (float)FastMath.sqrt((double)(1.0 - FastMath.pow((double)(2.0 * (double)x / (double)(length - 1) - 1.0), (int)2)));
            coefficients[x] = WindowFactory.getBesselZerothOrder(temp) / betaBesselZerothOrder;
        }
        return coefficients;
    }

    public static final float getBesselZerothOrder(float x) {
        float x2;
        float f = 1.0f;
        float xc = x2 = x * x * 0.25f;
        float v = 1.0f + x2;
        for (int i = 2; i < 100; ++i) {
            float a = (xc *= x2) / ((f *= (float)i) * f);
            v += a;
            if ((double)a < 1.0E-20) break;
        }
        return v;
    }

    public static float[] apply(float[] coefficients, float[] samples) {
        for (int x = 0; x < coefficients.length; ++x) {
            samples[x] = samples[x] * coefficients[x];
        }
        return samples;
    }

    public static float[] apply(WindowType type, float[] samples) {
        float[] coefficients = WindowFactory.getWindow(type, samples.length);
        return WindowFactory.apply(coefficients, samples);
    }

    public static void main(String[] args) {
        float[] kaiser = WindowFactory.getKaiser(16, 80.0f);
        mLog.debug("Window:" + Arrays.toString(kaiser));
    }
}

