"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.8.18"
__checksum__ = "38841b1e08f1362f26523a6140f85254ddcd9a579bdd5d6a2d925bc9d43bbc78"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 22), (152, 7), (159, 20), (179, 18), None, (197, 22), (219, 45), (264, 7), (271, 9), (280, 36), (316, 10), (326, 10), (336, 21), None, (357, 50), (407, 8), (415, 9), (424, 19), (443, 13), (456, 14), (470, 14), None, None, (484, 29), (513, 16), (529, 35), (564, 14), (578, 24), (602, 9), None, (611, 25), (636, 20), (656, 8), (664, 13), (677, 10), None, (687, 17), (704, 6), (710, 26), (736, 5), (741, 5), (746, 10), (756, 10), (766, 11), (777, 12), (789, 27), None, (816, 11), (827, 11), (838, 7), (845, 29), (874, 18), (892, 27), (919, 46), (965, 25), (990, 16), (1006, 8), (1014, 5), (1019, 22), (1041, 18), None, (1059, 36), (1095, 15), (1110, 8), (1118, 5), None, (1123, 5), (1128, 16), (1144, 14), (1158, 18), None, (1176, 14), (1190, 18), (1208, 48), (1256, 19), (1275, 5), (1280, 46), (1326, 14), (1340, 14), (1354, 20), None, (1374, 10), (1384, 13), (1397, 10), (1407, 19), None, (1426, 13), (1439, 19), (1458, 5), (1463, 4), (1467, 22), (1489, 10), (1499, 7), (1506, 14), (1520, 21), (1541, 11), (1552, 10), (1562, 12), (1574, 32), None, (1606, 10), (1616, 14), (1630, 12), (1642, 45), (1687, 15), None, (1702, 11), (1713, 23), (1736, 21), (1757, 26), (1783, 6), (1789, 6), (1795, 7), (1802, 5), (1807, 20), (1827, 23), (1850, 24), (1874, 13), (1887, 15), (1902, 19), (1921, 6), (1927, 61), (1988, 44), (2032, 12), (2044, 23), (2067, 16), (2083, 38), (2121, 6), (2127, 12), (2139, 44), (2183, 6), (2189, 41), (2230, 13), (2243, 23), (2266, 30), (2296, 16), (2312, 8), (2320, 15), (2335, 12), (2347, 19), (2366, 21), (2387, 15), None, (2402, 35), (2437, 21), (2458, 17), (2475, 19), (2494, 26), (2520, 5), (2525, 37), (2562, 30), (2592, 16), (2608, 10), (2618, 17), (2635, 23), (2658, 14), (2672, 17), (2689, 8), (2697, 4), (2701, 7), (2708, 29), (2737, 6), (2743, 18), (2761, 27), (2788, 20), (2808, 17), (2825, 19), (2844, 12), (2856, 40), (2896, 40), (2936, 12), (2948, 48), (2996, 25), (3021, 12), None, (3033, 8), (3041, 20), (3061, 19), (3080, 6), (3086, 23), None, (3109, 30), (3139, 33), (3172, 14), (3186, 12), (3198, 27), None, (3225, 26), (3251, 31), (3282, 50), (3332, 15), (3347, 20), (3367, 15), (3382, 21), (3403, 32), (3435, 24), (3459, 20), (3479, 13), (3492, 60), (3552, 19), (3571, 9), (3580, 12), (3592, 12), (3604, 11), (3615, 10), (3625, 48), (3673, 32), None, (3705, 25), (3730, 12), None, (3742, 8), (3750, 8), (3758, 7), None, (3765, 25), (3790, 17), None, (3807, 21), (3828, 35), (3863, 12), (3875, 10), (3885, 36), (3921, 20), (3941, 22), (3963, 23), (3986, 19), (4005, 12), (4017, 5), (4022, 30), (4052, 24), (4076, 14), (4090, 14), (4104, 47), (4151, 46), None, None, (4197, 51), (4248, 42), None, (4290, 14), None, (4304, 15), (4319, 8), (4327, 21), (4348, 6), (4354, 16), (4370, 17)], [(4387, 6160), (10547, 6631), (17178, 7039), (24217, 5961), (30178, 6340), (36518, 6012), (42530, 6895), (49425, 6196), (55621, 6780), (62401, 6050), (68451, 7069), (75520, 6404), (81924, 6743), (88667, 7092), (95759, 6496), (102255, 6441), (108696, 6993), (115689, 5955), (121644, 6278), (127922, 6501), (134423, 6902), (141325, 6519), (147844, 6874), (154718, 6105), (160823, 6265), (167088, 6588), (173676, 6553), (180229, 6866), (187095, 6286), (193381, 6649), (200030, 6836), (206866, 6509), (213375, 6416), (219791, 7030), (226821, 6159), (232980, 6818), (239798, 6208), (246006, 6967), (252973, 6766), (259739, 6959), (266698, 7488), (274186, 6402), (280588, 6256), (286844, 6165), (293009, 6345), (299354, 6101), (305455, 6275), (311730, 7010), (318740, 6412), (325152, 5760), (330912, 6373), (337285, 6570), (343855, 6505), (350360, 6719), (357079, 6806), (363885, 6772), (370657, 6813), (377470, 5930), (383400, 6854), (390254, 5811), (396065, 6609), (402674, 6346), (409020, 6337), (415357, 6820), (422177, 6643), (428820, 6538), (435358, 6220), (441578, 7052), (448630, 6708), (455338, 6771), (462109, 6518), (468627, 6481), (475108, 5697), (480805, 6964), (487769, 7059), (494828, 7045), (501873, 6126), (507999, 7163), (515162, 7085), (522247, 6033), (528280, 6738), (535018, 5710), (540728, 6373), (547101, 6548), (553649, 6474), (560123, 6417), (566540, 6573), (573113, 6674), (579787, 6783), (586570, 6499), (593069, 7205), (600274, 6022), (606296, 6251), (612547, 6580), (619127, 6521), (625648, 7132), (632780, 6874), (639654, 6461), (646115, 6252), (652367, 6094), (658461, 6273), (664734, 6717), (671451, 6112), (677563, 6463), (684026, 6155), (690181, 6848), (697029, 6648), (703677, 7020), (710697, 8002), (718699, 7119), (725818, 6879), (732697, 6439), (739136, 6283), (745419, 6693), (752112, 6897), (759009, 6628), (765637, 6281), (771918, 6355), (778273, 6358), (784631, 6984), (791615, 6765), (798380, 6885), (805265, 7033), (812298, 6798), (819096, 7729), (826825, 6399), (833224, 5782), (839006, 6914), (845920, 6610), (852530, 7985), (860515, 7006), (867521, 6138), (873659, 6798), (880457, 6814), (887271, 6310), (893581, 6706), (900287, 6147), (906434, 6778), (913212, 6463), (919675, 6526), (926201, 6530), (932731, 7225), (939956, 6228), (946184, 6271), (952455, 6600), (959055, 6577), (965632, 6461), (972093, 6831), (978924, 6143), (985067, 7125), (992192, 6626), (998818, 6690), (1005508, 6849), (1012357, 6443), (1018800, 6525), (1025325, 6453), (1031778, 6295), (1038073, 6416), (1044489, 6247), (1050736, 5931), (1056667, 6164), (1062831, 6551), (1069382, 7225), (1076607, 6129), (1082736, 6611), (1089347, 6884), (1096231, 6394), (1102625, 6072), (1108697, 6905), (1115602, 6518), (1122120, 5921), (1128041, 6507), (1134548, 7699), (1142247, 6030), (1148277, 6207), (1154484, 6692), (1161176, 6240), (1167416, 6668), (1174084, 6369), (1180453, 5953), (1186406, 7379), (1193785, 6718), (1200503, 6421), (1206924, 6972), (1213896, 7358), (1221254, 7297), (1228551, 6124), (1234675, 6937), (1241612, 6329), (1247941, 6597), (1254538, 6745), (1261283, 6139), (1267422, 6861), (1274283, 6982), (1281265, 6603), (1287868, 6638), (1294506, 6426), (1300932, 6526), (1307458, 6759), (1314217, 6326), (1320543, 6680), (1327223, 5962), (1333185, 7121), (1340306, 6862), (1347168, 6639), (1353807, 6913), (1360720, 5713), (1366433, 6734), (1373167, 6460), (1379627, 6745), (1386372, 6735), (1393107, 7129), (1400236, 6620), (1406856, 6738), (1413594, 6821), (1420415, 6338), (1426753, 6438), (1433191, 6490), (1439681, 6530), (1446211, 6349), (1452560, 6489), (1459049, 5939), (1464988, 7516), (1472504, 6683), (1479187, 6284), (1485471, 6653), (1492124, 6625), (1498749, 5823), (1504572, 6671), (1511243, 6553), (1517796, 7565), (1525361, 6484), (1531845, 5971), (1537816, 7060), (1544876, 6364), (1551240, 7136), (1558376, 6140), (1564516, 6221), (1570737, 5829), (1576566, 6595), (1583161, 6480), (1589641, 6854), (1596495, 6228), (1602723, 6575), (1609298, 6516), (1615814, 7008), (1622822, 6347), (1629169, 5843), (1635012, 6568), (1641580, 6176), (1647756, 6750), (1654506, 6892), (1661398, 7069), (1668467, 6308), (1674775, 6278), (1681053, 6701)], [(1687754, 703), (1688457, 625), (1689082, 649), (1689731, 679), (1690410, 523), (1690933, 633), (1691566, 644), (1692210, 836), (1693046, 640), (1693686, 627), (1694313, 536), (1694849, 574), (1695423, 758), (1696181, 866), (1697047, 987), (1698034, 714), (1698748, 1224), (1699972, 606), (1700578, 875), (1701453, 673), (1702126, 745), (1702871, 746), (1703617, 813), (1704430, 731), (1705161, 703), (1705864, 646), (1706510, 955), (1707465, 1131), (1708596, 791), (1709387, 682), (1710069, 922), (1710991, 787), (1711778, 557), (1712335, 691), (1713026, 732), (1713758, 743), (1714501, 619), (1715120, 688), (1715808, 692), (1716500, 1005), (1717505, 683), (1718188, 812), (1719000, 713), (1719713, 689), (1720402, 728), (1721130, 405), (1721535, 879), (1722414, 857), (1723271, 721), (1723992, 542), (1724534, 805), (1725339, 656), (1725995, 763), (1726758, 988), (1727746, 932), (1728678, 531), (1729209, 661), (1729870, 527), (1730397, 578), (1730975, 724), (1731699, 772), (1732471, 776), (1733247, 1025), (1734272, 861), (1735133, 669), (1735802, 692), (1736494, 767), (1737261, 444), (1737705, 561), (1738266, 539), (1738805, 692), (1739497, 850), (1740347, 523), (1740870, 725), (1741595, 634), (1742229, 671), (1742900, 552), (1743452, 696), (1744148, 785), (1744933, 428), (1745361, 711), (1746072, 629), (1746701, 828), (1747529, 623), (1748152, 607), (1748759, 410), (1749169, 597), (1749766, 725), (1750491, 767), (1751258, 713), (1751971, 832), (1752803, 1074), (1753877, 826), (1754703, 773), (1755476, 687), (1756163, 436), (1756599, 971), (1757570, 878), (1758448, 580), (1759028, 647), (1759675, 683), (1760358, 843), (1761201, 855), (1762056, 552), (1762608, 632), (1763240, 740), (1763980, 395), (1764375, 479), (1764854, 954), (1765808, 880), (1766688, 792), (1767480, 774), (1768254, 632), (1768886, 771), (1769657, 659), (1770316, 699), (1771015, 709), (1771724, 433), (1772157, 667), (1772824, 649), (1773473, 914), (1774387, 653), (1775040, 804), (1775844, 404), (1776248, 703), (1776951, 739), (1777690, 835), (1778525, 908), (1779433, 781), (1780214, 904), (1781118, 791), (1781909, 524), (1782433, 757), (1783190, 583), (1783773, 758), (1784531, 732), (1785263, 676), (1785939, 700), (1786639, 622), (1787261, 673), (1787934, 594), (1788528, 674), (1789202, 715), (1789917, 616), (1790533, 455), (1790988, 587), (1791575, 656), (1792231, 559), (1792790, 717), (1793507, 594), (1794101, 761), (1794862, 532), (1795394, 510), (1795904, 656), (1796560, 596), (1797156, 658), (1797814, 639), (1798453, 796), (1799249, 594), (1799843, 609), (1800452, 857), (1801309, 841), (1802150, 543), (1802693, 695), (1803388, 854), (1804242, 632), (1804874, 659), (1805533, 455), (1805988, 609), (1806597, 633), (1807230, 754), (1807984, 598), (1808582, 932), (1809514, 695), (1810209, 807), (1811016, 721), (1811737, 668), (1812405, 586), (1812991, 664), (1813655, 699), (1814354, 1277), (1815631, 514), (1816145, 624), (1816769, 608), (1817377, 986), (1818363, 754), (1819117, 745), (1819862, 546), (1820408, 587), (1820995, 808), (1821803, 553), (1822356, 569), (1822925, 847), (1823772, 650), (1824422, 883), (1825305, 791), (1826096, 668), (1826764, 710), (1827474, 822), (1828296, 637), (1828933, 889), (1829822, 631), (1830453, 762), (1831215, 570), (1831785, 742), (1832527, 468), (1832995, 775), (1833770, 779), (1834549, 665), (1835214, 916), (1836130, 775), (1836905, 790), (1837695, 903), (1838598, 1077), (1839675, 848), (1840523, 618), (1841141, 854), (1841995, 702), (1842697, 483), (1843180, 443), (1843623, 692), (1844315, 762), (1845077, 406), (1845483, 988), (1846471, 488), (1846959, 758), (1847717, 863), (1848580, 737), (1849317, 769), (1850086, 647), (1850733, 788), (1851521, 715), (1852236, 784), (1853020, 628), (1853648, 566), (1854214, 408), (1854622, 666), (1855288, 456), (1855744, 767), (1856511, 855), (1857366, 764), (1858130, 718), (1858848, 621), (1859469, 589), (1860058, 848), (1860906, 474), (1861380, 590), (1861970, 780), (1862750, 494), (1863244, 847), (1864091, 2104), (1866195, 548), (1866743, 661), (1867404, 890), (1868294, 885), (1869179, 510)], [(1869689, 48), None, (1869737, 35), (1869772, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1869814, 42), None, (1869856, 25), (1869881, 16), None, (1869897, 18), None, None, None, None, (1869915, 26), None, None, None, None, (1869941, 21), (1869962, 25), None, None, (1869987, 26), None, None, None, None, (1870013, 44), (1870057, 21), (1870078, 23), None, None, None, None, (1870101, 48), None, None, None, None, None, (1870149, 31), None, None, None, None, (1870180, 42), None, (1870222, 22), None, (1870244, 21), None, (1870265, 26), (1870291, 42), None, None, (1870333, 77), None, None, None, None, None, (1870410, 21), (1870431, 21), None, None, (1870452, 34), (1870486, 42), None, None, None, (1870528, 25), None, None, (1870553, 21), None, None, None, None, None, (1870574, 24), (1870598, 21), None, None, (1870619, 26), None, (1870645, 18), None, (1870663, 54), None, None, None, None, None, None, (1870717, 26), None, (1870743, 19), None, (1870762, 20), None, None, (1870782, 42), (1870824, 42), (1870866, 17), None, (1870883, 26), None, (1870909, 26), None, None, None, (1870935, 26), (1870961, 20), (1870981, 26), None, (1871007, 42), (1871049, 63), None, None, None, (1871112, 40), (1871152, 48), None, None, None, (1871200, 47), None, None, None, None, None, None, None, (1871247, 42), None, (1871289, 55), None, (1871344, 9), None, (1871353, 21), (1871374, 42), None, None, (1871416, 42), (1871458, 82), None, None, (1871540, 42), None, None, None, None, None, None, None, None, None, (1871582, 42), (1871624, 21), (1871645, 21), None, (1871666, 42), (1871708, 25), None, None, (1871733, 21), (1871754, 42), None, None, (1871796, 21), (1871817, 19), (1871836, 26), None, None, None, (1871862, 21), None, None, (1871883, 38), None, (1871921, 22), (1871943, 21), (1871964, 21), None, None, (1871985, 63), None, (1872048, 21), (1872069, 42), None, (1872111, 17), None, None, None, None, (1872128, 21), (1872149, 21), None, None, (1872170, 21), None, None, (1872191, 21), None, (1872212, 26), None, (1872238, 50), None, None, None, (1872288, 50), (1872338, 26), (1872364, 21), (1872385, 21), (1872406, 19), None, (1872425, 35), (1872460, 26), (1872486, 23), (1872509, 21), (1872530, 42), None, None, None, None, None, None, (1872572, 21), None, None, None, (1872593, 21), None, None, (1872614, 90), None, (1872704, 239), (1872943, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path), "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
