#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# python standard library
import re


class codebook():

    '''
    ┌───────────────────────────────────────────────────┐
    │                  PJL Error Codes                  │
    ├───────────────────────────────────────────────────┤
    │ 10xxx: Informational Messages                     │
    │ 11xxx: Background Paper Mount                     │
    │ 12xxx: Background Paper Tray Status               │
    │ 15xxx: Output Bin Status                          │
    │ 20xxx: PJL Parser Errors                          │
    │ 25xxx: PJL Parser Warnings                        │
    │ 27xxx: PJL Semantic Errors                        │
    │ 30xxx: Auto-Continuable Conditions                │
    │ 32xxx: PJL File System Errors                     │
    │ 35xxx: Potential Operator Intervention Conditions │
    │ 40xxx: Operator Intervention Required             │
    │ 41xxx: Foreground Paper Mount                     │
    │ 42xxx: Paper Jam Errors                           │
    │ 43xxx: Optional Paper Handling Device Errors      │
    │ 44xxx: Paper Jam Information                      │
    │ 50xxx: Hardware Errors                            │
    └───────────────────────────────────────────────────┘
    '''

    def get_errors(self, code):
        return (self.codelist[key] for key in self.codelist if re.match(key, code))

    codelist = {
        # 10xxx: Informational Messages
        '10001': "READY (online)",
        '10002': "READY (offline)",
        '10003': "WARMING UP",
        '10004': "SELF TEST",
        '10005': "RESET, CLEARING MEMORY",
        '10006': "TONER LOW",
        '10007': "CANCELING JOB",
        '10010': "STATUS BUFFER OVERFLOW",
        '10013': "SELF TEST",
        '10011': "INITIALIZING",
        '10014': "PRINTING TEST",
        '10015': "TYPEFACE LIST",
        '10016': "ENGINE TEST",
        '10017': "DEMO PAGE",
        '10018': "MENU RESET",
        '10019': "RESET ACTIVE I/O",
        '10020': "RESET ALL I/O",
        '10021': "COLD RESET",
        '10022': "CONFIG PAGE",
        '10023': "PROCESSING JOB",
        '10024': "DATA RECEIVED",
        '10025': "ACCESS DENIED",
        '10026': "NO JOB TO CANCEL",
        '10027': "CLEARING PAPER PATH",
        '10028': "PRINTING ERROR LOG",
        '10029': "FORMFEEDING",
        '10030': "DISPLAY JOB MESSACE RECEIVED",
        '10031': "ENGINE CLEANING",

        # 11xxx: Background Paper Mount
        '11.00': "Unknown Paper",
        '11.01': "Unknown Envelope",
        '11.02': "Letter Paper",
        '11.03': "Legal Paper",
        '11.04': "A4 Paper",
        '11.05': "Executive Paper",
        '11.06': "Ledger Paper",
        '11.07': "A3 Paper",
        '11.08': "COM10 Envelope",
        '11.09': "Monarch Envelope",
        '11.10': "C5 Envelope",
        '11.11': "DL Envelope",
        '11.12': "Japan B4 Paper",
        '11.13': "Japan B5 Paper",
        '11.14': "B5 Envelope",
        '11.15': "Custom Media Paper",
        '11.16': "J-Postcard",
        '11.17': "J-RoundTrip Postcard",
        '11.18': "A5 Paper",
        '11.19': "Letter-R Paper",
        '11.20': "A4-R Paper",
        '11.21': "A5-R Paper",
        '11.22': "Japan B5-R Paper",
        '11.23': "Exec-R Paper",
        '11.24': "A6 Paper",
        '11.25': "Foolscap Paper",
        '11.26': "Japan B6 Paper",

        # 12xxx: Background Paper Tray Status
        '12.01': "Tray Open",
        '12.02': "Tray Lifting",

        # 15xxx: Output Bin Status
        '15..1': "Output Bin Full",
        '15..2': "Output Bin Open",
        '15..3': "Output Bin Broken",

        # 20xxx: PJL Parser Errors
        '20001': "Generic syntax error (entire PJL command ignored)",
        '20002': "Unsupported command",
        '20004': "Unsupported personality, system, or I/O port",
        '20005': "PJL command buffer overflow",
        '20006': "Illegal character or line terminated by the UEL",
        '20007': "<WS> or [<CR>]<LF> missing after closing quotes",
        '20008': "Invalid character in an alphanumeric value",
        '20009': "Invalid character in a numeric value",
        '20010': "Invalid character at the start of a string, alphanumeric value, or numeric value",
        '20011': "String missing closing double-quote character",
        '20012': "Numeric value starts with a decimal point",
        '20013': "Numeric value does not contain any digits",
        '20014': "No alphanumeric value after command modifier",
        '20015': "Option name and equal sign encountered, but the value field is missing",
        '20016': "More than one command modifier",
        '20017': "Command modifier encountered after an option (command modifier must precede option)",
        '20018': "Command not an alphanumeric value",
        '20019': "Numeric value encountered when an alphanumeric value expected",
        '20020': "String encountered when an alphanumeric value expected",
        '20021': "Unsupported command modifier",
        '20022': "Command modifier missing",
        '20023': "Option missing",
        '20024': "Extra data received after option name (used for commands like SET that limit the number of options supported)",
        '20025': "Two decimal points in a numeric value",
        '20026': "Invalid binary value",

        # 25xxx: PJL Parser Warnings
        '25001': "Generic warning error (part of the PJL command ignored)",
        '25002': "PJL prefix missing",
        '25003': "Alphanumeric value too long",
        '25004': "String too long",
        '25005': "Numeric value too long",
        '25006': "Unsupported option name",
        '25007': "Option name requires a value which is missing",
        '25008': "Option name requires a value of a different type",
        '25009': "Option name received with a value, but this option does not support values",
        '25010': "Same option name received more than once",
        '25011': "Ignored option name due to value underflow or overflow",
        '25012': "Value for option experienced data loss due to data conversion (value truncated or rounded)",
        '25013': "Value for option experienced data loss due to value being out of range; the value used was the closest supported limit",
        '25014': "Value is of the correct type, but is out of range (value was ignored)",
        '25016': "Option name received with an alphanumeric value, but this value is not supported",
        '25017': "String empty, option ignored",
        '25018': "A Universal Exit Language command was expected but not found",

        # 27xxx: PJL Semantic Errors
        '27001': "Generic semantic error",
        '27002': "EOJ command encountered without a previously matching JOB command. An EOJ command does not have a matching JOB command if the number of valid EOJ commands received is greater than the number of valid JOB commands received.",
        '27003': "Password protected—attempted to change NVRAM value when password is set and the job is not a secure PJL job.",
        '27004': "Cannot modify the value of a read-only variable.",
        '27005': "Can only use DEFAULT with this variable; cannot use SET.",
        '27006': "Attempted to pass a NULL string to a command or command option that requires the string to contain at least one character.",
        '27007': "Attempted to DEFAULT a variable which can only be SET.",

        # 30xxx: Auto-Continuable Conditions
        '30010': "STATUS BUFFER OVERFLOW",
        '30016': "20 MEM OVERFLOW or 20 MEMORY OVERFLOW DATA LOST or 20 INSUFFICIENT MEMORY, PRESS GO KEY",
        '30017': "21 PRINT OVERRUN or 21 PRINT OVERRUN DATA LOST or 21 PAGE TOO COMPLEX",
        '30018': "40 ERROR or 40.1 HP MIO 1 ERROR or 40.2 HP MIO 2 ERROR or 40 SER IO ERROR",
        '30027': "22 I/O CONFIG ERROR or 22 SER IO ERROR 22 HP MIO 1 IO ERROR CHECK SETUP or 22 HP MIO 2 IO ERROR CHECK SETUP or 22 PARALLEL IO ERROR CHECK SETUP",
        '30034': "41.x ERROR or 41.3 PAPER ERROR",
        '30035': "68 ERROR CHECK CONFIGURATION or 68 NVRAM ERROR, SETTINGS NOT SAVED or 68 ERROR",
        '30036': "68 SERVICE or 68 NVRAM FULL CHECK CONFIGURATION or 68 NVRAM ERROR, SETTINGS NOT SAVED",
        '30072': "49 REMOVE PAGE",
        '30076': "PERSONALITY MEMORY OVERFLOW/ OUT OF MEMORY",
        '30094': "41.1 GENERAL MISPRINT ERROR",
        '30095': "41.2 BEAM DETECT MALFUNCTION",
        '30096': "41.3 IMPROPER PAPER SIZE or 41.3 UNEXPECTED PAPER SIZE",
        '30097': "41.4 NO VIDEO SYNC",
        '30098': "41.5 NOISY VIDEO SYNC REQ SIGNAL",
        '30099': "INVALID INPUT SOURCE",
        '30100': "INVALID OUTPUT DESTINATION",
        '30101': "BINDING AGENT OUT OF SUPPLIES",
        '30102': "BINDING AGENT MEDIA MISALIGNMENT",
        '30103': "BINDING AGENT MEDIA CAPACITY EXCEEDED",
        '30104': "EXTERNAL INPUT DEVICE PAPER PATH ERROR",
        '30105': "EXTERNAL OUTPUT DEVICE PAPER PATH ERROR",
        '30106': "EXTERNAL INPUT DEVICE OPERATIONAL ERROR",
        '30107': "EXTERNAL OUTPUT DEVICE OPERATIONAL ERROR",
        '30108': "40 HP MIO ERROR, PRESS GO KEY",
        '30109': "40 BAD SERIAL DATA FORMAT, PRESS GO KEY",
        '30110': "22 HP MIO ERROR, PRESS GO KEY",
        '30111': "22 PARALLEL IO ERROR, PRESS GO KEY",
        '30112': "22 SERIAL IO ERROR, PRESS GO KEY",
        '30113': "STOPPING CHECK LAST PAGE",
        '30114': "PRESS SELECT TO CANCEL JOB",
        '30119': "MEDIA FEED ERROR",

        # 32xxx: PJL File System Errors
        '30000': "General error",
        '30001': "Volume not available",
        '30002': "Disk full",
        '30003': "File not found",
        '30004': "No free file descriptors",
        '30005': "Invalid number of bytes",
        '30006': "File already exists",
        '30007': "Illegal name",
        '30008': "Can't delete root",
        '30009': "File operation attempted on a directory",
        '30010': "Directory operation attempted on a file",
        '30011': "Not same volume",
        '30012': "Read only",
        '30013': "Directory full",
        '30014': "Directory not empty",
        '30015': "Bad disk",
        '30016': "No label",
        '30017': "Invalid parameter",
        '30018': "No contiguous space",
        '30019': "Can't change root",
        '30020': "File Descriptor obsolete",
        '30021': "Deleted",
        '30022': "No block device",
        '30023': "Bad seek",
        '30024': "Internal error",
        '30025': "Write only",
        '30026': "Write protected",
        '30027': "No filename",
        '30051': "End of directory",
        '30052': "No file system",
        '30053': "No memory",
        '30054': "Vol name out of range",
        '30055': "Bad FS",
        '30056': "Hardware failure",

        # 35xxx: Potential Operator Intervention Conditions
        '35028': "BAD MIO",
        '35029': "W1 IMAGE ADAPT or IMAGE ASSIST or MEMORY LOW, PAGE SIMPLIFIED",
        '35030': "WA JOB ALTERED",
        '35031': "W2 INVALID PERS or REQUESTED LANGUAGE NOT AVAILABLE",
        '35032': "WL WARNINGS LOST",
        '35037': "W3 JOB ABORTED or CANCELING JOB or INSUFFICIENT MEMORY, JOB CLEARED",
        '35039': "W9 JOB 600/LTR",
        '35040': "W0 JOB 600/A4",
        '35041': "W8 JOB 600/OFF",
        '35042': "W7 JOB 300/LGL",
        '35043': "W5 JOB 300/LTR",
        '35044': "W6 JOB 300/A4",
        '35045': "W4 JOB 300/OFF",
        '35073': "WM CHK MEM CNFIG or MEMORY SETTINGS CHANGED",
        '35074': "WM MEM CNFIG N/A",
        '35075': "USER MAINTENANCE REQUESTED",
        '35076': "WM personality MEM FULL or XXXX MEMORY FULL, STORED DATA LOST",
        '35078': "Entered powersave mode (00 POWERSAVE) or POWERSAVE ON",
        '35081': "WM JOB 300",
        '35082': "WD DISK NOT INIT or Volume 0 NOT INIT (Volume 0 will be indicated as either DISK, FLASH, or RAMDISK as appropriate)",
        '35084': "WM JOB 600/LGL",
        '35085': "WM JOB 600/A3",
        '35086': "WM JOB 600/11x17",
        '35087': "WM JOB 300/A3",
        '35088': "WM JOB 300/11x17",
        '35115': "Volume 1 NOT INIT (Volume 1 is not initialized. Volume 1 will be indicated as either DISK, FLASH, or RAMDISK as appropriate)",
        '35117': "Volume 2 NOT INIT (Volume 2 is not initialized. Volume 2 will be indicated as either DISK, FLASH, or RAMDISK as appropriate.)",

        # 40xxx: Operator Intervention Required
        '40000': "SLEEP MODE (STANDBY)",
        '40005': "Cartridge error during align cartridges (or reading setup sheet—DesignJet)",
        '40010': "14 NO EP CART or INSTALL TONER CARTRIDGE or no electrical contact with one or more ink cartridges (DesignJet)",
        '40011': "Accessing ink cartridges (DesignJet)",
        '40019': "REMOVE PAPER FROM [bin name]",
        '40020': "NO MICR TONER or INSTALL MICR TONER CARTRIDGE",
        '40021': "12 PRINTER OPEN OR NO EP CART or CLOSE PRINTER COVER (or lower lever or lower window—DesignJet)",
        '40022': "13 PAPER JAM or REMOVE PAPER JAM",
        '40024': "FE CARTRIDGE",
        '40026': "PC INSTALL or INSTALL TRAY 2",
        '40038': "16 LOW TONER or TONER LOW or TONER LOW, PRESS GO KEY",
        '40046': "FI INSERT CART",
        '40047': "FR REMOVE CART",
        '40048': "[PJL OPMSG]",
        '40049': "[PJL STMSG]",
        '40050': "50 SERVICE or 50 FUSER ERROR, CYCLE POWER",
        '40051': "51 ERROR or 51 PRINTER ERROR, CYCLE POWER",
        '40052': "52 ERROR or 52 PRINTER ERROR, CYCLE POWER",
        '40053': "53-xy-zz ERROR",
        '40054': "54 ERROR",
        '40055': "55 ERROR",
        '40056': "56 ERROR",
        '40057': "57 SERVICE or 57 MOTOR FAILURE, CALL SERVICE",
        '40058': "58 SERVICE or FAN MOTOR FAILURE, CALL SERVICE",
        '40059': "59 ERROR",
        '40061': "61.x SERVICE",
        '40062': "62.x SERVICE",
        '40063': "63 SERVICE",
        '40064': "64 SERVICE or PRINTER ERROR, CYCLE POWER",
        '40065': "65 SERVICE",
        '40066': "External paper handling device failure",
        '40067': "67 SERVICE",
        '40068': "69 SERVICE",
        '40069': "70 ERROR",
        '40070': "71 ERROR",
        '40071': "72 SERVICE",
        '40079': "PRINTER MANUALLY TAKEN OFFLINE or OFFLINE or FORMFEEDING or SERVICE MODE or 00 OFFLINE",
        '40080': "EE INCOMPATIBLE or LC INCOMPATIBLE",
        '40083': "FS DISK FAILURE or 311.1.1 DISK FAILURE or Volume 0 FAILURE (Volume 0 will be indicated as either DISK, FLASH, or RAMDISK as appropriate)",
        '40089': "INCOMPLETE TRAY 3 INSTALLED",
        '40090': "INCOMPATIBLE ENVELOPE FEEDER INSTALLED",
        '40092': "81 SERVICE (XXX)",
        '40093': "REMOVE DUPLEX JAM",
        '40096': "41.3 UNEXPECTED PAPER SIZE CHECK PAPER IN TRAY",
        '40099': "56.1 ERROR PRESS SELECT KEY",
        '40100': "56.2 ERROR PRESS SELECT KEY",
        '40102': "FINISHER ALIGN ERROR [BIN NAME]",
        '40103': "FINISH LIMIT REACHED [BIN NAME]",
        '40104': "INPUT DEVICE FEED PATH OPEN",
        '40105': "OUTPUT DEVICE DELIVERY PATH OPEN",
        '40106': "INPUT OPERATION ERROR X.YY",
        '40107': "OUTPUT OPERATION ERROR X.YY",
        '40116': "Volume 1 FAILURE (Failure on Volume 1. Volume 1 will be indicated as either DISK, FLASH, or RAMDISK as appropriate)",
        '40118': "Volume 2 FAILURE (Failure on Volume 2. Volume 2 is indicated as either DISK, FLASH, or RAMDISK as appropriate.)",
        '40119': "PAPER MISFEED",
        '40120': "Open face-up output bin",
        '40121': "Close face-up output bin",
        '40122': "Duplexer must be installed",
        '40123': "Duplexer error, remove duplexer",
        '40124': "Bad duplexer connection",
        '40128': "DRUM ERROR REPLACE DRUM KIT",
        '40129': "DRUM LIFE OUT REPLACE DRUM KIT",
        '40130': "DRUM LIFE LOW REPLACE DRUM KIT",
        '40131': "TRANSFER KIT OUT REPLACE KIT",
        '40132': "TRANSFER KIT LOW REPLACE KIT",
        '40141': "WASTE TONER FULL REPLACE DRUM KIT",
        '40142': "INSTALL DRUM KIT",
        '40143': "REINSTALL TRANSFER BELT",
        '40144': "PRESS GO TO PRINT PRESS SELECT TO CHANGE TONER",
        '40146': "41.5 UNEXPECTED PAPER TYPE CHECK PAPER IN TRAY",

        # 41xxx: Foreground Paper Mount
        '41.00': "Unknown Paper",
        '41.01': "Unknown Envelope",
        '41.02': "Letter Paper",
        '41.03': "Legal Paper",
        '41.04': "A4 Paper",
        '41.05': "Executive Paper",
        '41.06': "Ledger Paper",
        '41.07': "A3 Paper",
        '41.08': "COM10 Envelope",
        '41.09': "Monarch Envelope",
        '41.10': "C5 Envelope",
        '41.11': "DL Envelope",
        '41.12': "Japan B4 Paper",
        '41.13': "Japan B5 Paper",
        '41.14': "B5 Envelope",
        '41.15': "Custom Media Paper",
        '41.16': "J-Postcard",
        '41.17': "J-RoundTrip Postcard",
        '41.18': "A5 Paper",
        '41.19': "Letter-R Paper",
        '41.20': "A4-R Paper",
        '41.21': "A5-R Paper",
        '41.22': "Japan B5-R Paper",
        '41.23': "Exec-R Paper",
        '41.24': "A6 Paper",
        '41.25': "Foolscap Paper",
        '41.26': "Japan B6 Paper",

        # 42xxx: Paper Jam Errors
        '42.00': "Non-specific jam",
        '42.01': "Paper feed 1, paper late jam",
        '42.02': "Paper feed 2, paper late jam",
        '42.03': "Fuser output, paper late jam",
        '42.04': "Fuser output, paper stopped jam",
        '42.05': "Face down output, paper late jam",
        '42.06': "Face down output, paper stopped jam",
        '42.07': "2-Sided turn around, paper late jam",
        '42.08': "2-Sided turn around, paper stopped jam",
        '42.09': "2-Sided path, paper late jam",
        '42.10': "2-Sided path, paper stopped jam",
        '42.11': "External input device paper jam",
        '42.12': "External output device paper jam",
        '42.13': "Fuser accordion jam",
        '42.14': "Fuser accordion jam",
        '42.15': "Printer could not auto eject paper",

        # 43xxx: Optional Paper Handling Device Errors
        '430.1': "Unable to communicate with the PHC",
        '430.2': "Out of sequence packet",
        '430.3': "Bad transaction",
        '430.4': "Invalid status bits",
        '430.5': "Invalid status code",
        '430.6': "Invalid device specified",
        '430.7': "Invalid tray specified",
        '430.8': "Invalid bin specified",
        '430.9': "Invalid number of special purpose bytes",
        '43.10': "Invalid number of NVEE bytes",
        '43.11': "General PHC malfunction",
        '43.12': "Bad PHC RAM",
        '43.13': "PHC ROM checksum error",
        '43.14': "Faulty PHC FIFO",
        '43.15': "PHC cannot communicate with device(s)",
        '43.16': "PHC never ready during initialization",
        '43.17': "FIFO empty when expecting data",
        '43.18': "FIFO full when tray to send data",
        '43.19': "Invalid page ACK (formatter not expecting a page ACK)",
        '43.20': "Page ACK timeout",
        '43.21': "Inconsistent paper size",
        '43.22': "Wrong page ACK",
        '43.23': "Device(s) never ready during installation",
        '43.24': "Negative ACK of ready tray",

        # 44xxx: Paper Jam Information
        '44.00': "Paper jam in unknown location",
        '44.01': "Paper jam in the paper input area",
        '44.02': "Paper jam in the paper output area (may need to open rear door of printer)",
        '44.03': "Paper jam underneath the top cover. Remove toner cartridge to clear.",
        '44.04': "Paper jam in the duplexer",
        '44.05': "Paper jam in the duplex front cover area",
        '44.06': "Paper jam in the face down output area",
        '44.07': "Paper jam in the front door area",

        # 50xxx: Hardware Errors
        '50000': "General Hardware Failure",
        '50001': "ROM or RAM Error, ROM Checksum Failed (or interface error [DesignJet])",
        '50002': "RAM Error, RAM Test Failed (or part malfunction [DesignJet])",
        '50003': "Engine Fuser Error",
        '50004': "Engine Beam Detect Error",
        '50005': "Engine Scanner Error",
        '50006': "Engine Fan Error",
        '50007': "Engine Communications Error",
        '50008': "50.1 FUSER ERROR CYCLE POWER or LOW FUSER TEMPERATURE",
        '50009': "50.2 FUSER ERROR CYCLE POWER or FUSER TOOK TOO LONG TO WARM UP",
        '50010': "50.3 FUSER ERROR CYCLE POWER or FUSER TOO HOT",
        '50011': "50.4 FUSER ERROR CYCLE POWER or BAD FUSER",
        '50012': "51.1 ERROR PRESS SELECT KEY or BEAM DETECT MALFUNCTION",
        '50013': "51.2 ERROR PRESS SELECT KEY or LASER MALFUNCTION",
        '50014': "52.1 ERROR PRESS SELECT KEY or SCANNER STARTUP FAILURE",
        '50015': "52.2 ERROR PRESS SELECT KEY or SCANNER ROTATION FAILURE",
        '50016': "57.1 FAN FAILURE CALL SERVICE or FAN MOTOR 1 FAILURE",
        '50017': "57.2 FAN FAILURE CALL SERVICE or FAN MOTOR 2 FAILURE",
        '50018': "57.2 FAN FAILURE CALL SERVICE or FAN MOTOR 3 FAILURE",
        '50019': "57.4 FAN FAILURE CALL SERVICE or FAN MOTOR 4 FAILURE",
        '50020': "UPPER INPUT TRAY LIFTER MALFUNCTION",
        '50021': "LOWER INPUT TRAY LIFTER MALFUNCTION",
        '50022': "58.3 PRINTER ERROR CYCLE POWER or MULTIPURPOSE TRAY LIFTER MALFUNCTION",
        '50023': "59.1 PRINTER ERROR CYCLE POWER or MAIN MOTOR STARTUP FAILURE",
        '50024': "59.2 PRINTER ERROR CYCLE POWER or MAIN MOTOR ROTATION FAILURE",
        '50025': "FINISHER MALFUNCTION [BIN NAME] or EXTERNAL BINDING DEVICE HAS A MALFUNCTION",
        '50026': "DEVICE ERROR X.YY CYCLE POWER or AN EXTERNAL DEVICE HAS REPORTED A MALFUNCTION",
        '50027': "DUPLEX ERROR CHECK DUPLEX UNIT or DUPLEX UNIT GUIDE FAILED AND REQUIRES SERVICE",
        '50028': "55.XX ERROR PRESS SELECT KEY or COMMUNICATION FAILURE BETWEEN FORMATTER AND ENGINE",
        '50029': "64 PRINTER ERROR CYCLE POWER or VIDEO DMA TRANSFER TIMED OUT",
        '50030': "FAN MOTOR 5 FAILURE",
        '50031': "FAN MOTOR 6 FAILURE",
        '50032': "FAN MOTOR 7 FAILURE",
        '50033': "FAN MOTOR 8 FAILURE",
        '505..': "Firmware Error, Power Cycle",
        '50599': "Processor Error, Power Cycle"}
