#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

__all__ = ["SuspiciousHeader"]

from .. import Vulnerability
from ... import identity
from ...information.http import HTTP_Response


#------------------------------------------------------------------------------
class SuspiciousHeader(Vulnerability):
    """
    Suspicious HTTP Header.

    An HTTP header was found that may contain sensitive information.
    User attention could be required.
    """

    TARGET_CLASS = HTTP_Response

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"]     = "informational"
    DEFAULTS["cvss_base"] = "2.9"

    #--------------------------------------------------------------------------
    def __init__(self, target, name, value, **kwargs):
        """
        :param target: HTTP response where the suspicious header was found.
        :type target: HTTP_Response

        :param name: Name of the suspicious header.
        :type name: str

        :param value: Header value.
        :type value: str
        """

        # Validate the data types.
        if type(name) is not str:
            raise TypeError("Expected str, got %r instead" % type(name))
        if type(value) is not str:
            raise TypeError("Expected str, got %r instead" % type(value))

        # Save the properties.
        self.__name  = name
        self.__value = value

        # Parent constructor.
        super(SuspiciousHeader, self).__init__(target, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @property
    def name(self):
        """
        :returns: Header name.
        :rtype: str
        """
        return self.__name


    #--------------------------------------------------------------------------
    @property
    def value(self):
        """
        :returns: Header value.
        :rtype: str
        """
        return self.__value
