#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__= """
GoLismero 2.0 - The web knife - Copyright (C) 2011-2014

Golismero project site: https://github.com/golismero
Golismero project mail: contact@golismero-project.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
"""

from .. import Vulnerability
from ... import identity
from ...resource import Resource


#------------------------------------------------------------------------------
class OutdatedSoftware(Vulnerability):
    "" # this is intentional, do not remove or change!

    DEFAULTS = Vulnerability.DEFAULTS.copy()
    DEFAULTS["level"]      = "high"
    DEFAULTS["cvss_base"]  = "8.5"
    DEFAULTS["description"] = \
        "Outdated, potentially vulnerable software was found."
    DEFAULTS["solution"] = \
"""If possible, apply all missing patches or upgrade to a newer version.
If not, consider adding firewall rules to restrict access to these hosts."""
    DEFAULTS["references"] = (
        "https://www.owasp.org/index.php/"
                                "Top_10_2013-A5-Security_Misconfiguration",
    )


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe, **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource

        :param cpe: CPE name of the outdated software.
        :type cpe: str
        """

        # Validate the parameters.
        if type(cpe) is not str:
            raise TypeError("Expected str, got %r instead" % type(cpe))
        if not cpe.startswith("cpe:"):
            raise ValueError("Not a CPE name: %r" % cpe)

        # Convert CPE <2.3 (URI binding) to CPE 2.3 (formatted string binding).
        if cpe.startswith("cpe:/"):
            cpe_parts = cpe[5:].split(":")
            if len(cpe_parts) < 11:
                cpe_parts.extend( "*" * (11 - len(cpe_parts)) )
            cpe = "cpe:2.3:" + ":".join(cpe_parts)

        # CPE tag of the outdated software.
        self.__cpe = cpe

        # Parent constructor.
        super(OutdatedSoftware, self).__init__(target, **kwargs)

    __init__.__doc__ += Vulnerability.__init__.__doc__[
            Vulnerability.__init__.__doc__.find("\n        :keyword"):]


    #--------------------------------------------------------------------------
    @identity
    def cpe(self):
        """
        :returns: CPE 2.3 name of the outdated software.
        :rtype: str
        """
        return self.__cpe


#------------------------------------------------------------------------------
#
# Generic programing languages and frameworks
#
#------------------------------------------------------------------------------
class OutdatedSoftwarePHP(OutdatedSoftware):
    data_subtype = "php"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:php:php", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePHP, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: PHP"


#------------------------------------------------------------------------------
class OutdatedSoftwareJava(OutdatedSoftware):
    data_subtype = "java"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:sun:java_web_start", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareJava, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Java"


#------------------------------------------------------------------------------
class OutdatedSoftwarePython(OutdatedSoftware):
    data_subtype = "python"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:python:python", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePython, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Python"


#------------------------------------------------------------------------------
class OutdatedSoftwareColdFusion(OutdatedSoftware):
    data_subtype = "coldfusion"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:adobe:coldfusion", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareColdFusion, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Cold Fusion"


#------------------------------------------------------------------------------
class OutdatedSoftwareASPNet(OutdatedSoftware):
    data_subtype = "aspnet"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:microsoft:.net_framework:", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareASPNet, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Microsoft ASP .Net"


#------------------------------------------------------------------------------
class OutdatedSoftwareRubyRails(OutdatedSoftware):
    data_subtype = "rubyrails"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:rubyonrails:rails", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareRubyRails, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Ruby on Rails"


#------------------------------------------------------------------------------
class OutdatedSoftwareStruts(OutdatedSoftware):
    data_subtype = "javastruts"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:apache:struts", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareStruts, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Struts framework"


#------------------------------------------------------------------------------
class OutdatedSoftwareDjango(OutdatedSoftware):
    data_subtype = "pythondjango"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:djangoproject:django", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareDjango, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Django framework"


#------------------------------------------------------------------------------
#
# CMS
#
#------------------------------------------------------------------------------
class OutdatedSoftwareDrupal(OutdatedSoftware):
    data_subtype = "drupal"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:drupal:drupal", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareDrupal, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Drupal"


#------------------------------------------------------------------------------
class OutdatedSoftwarePHPbb(OutdatedSoftware):
    data_subtype = "phpbb"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePHPbb, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: phpBB"


#------------------------------------------------------------------------------
class OutdatedSoftwarePHPMyAdmin(OutdatedSoftware):
    data_subtype = "phpmyadmin"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:phpmyadmin:phpmyadmin", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePHPMyAdmin, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: phpMYAdmin"


#------------------------------------------------------------------------------
class OutdatedSoftwareJoomla(OutdatedSoftware):
    data_subtype = "joomla"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:joomla:joomla%21", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareJoomla, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Joomla"


#------------------------------------------------------------------------------
class OutdatedSoftwarePHPGroupWare(OutdatedSoftware):
    data_subtype = "phpgroupware"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:horde:groupware", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePHPGroupWare, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Horde Groupware"


#------------------------------------------------------------------------------
class OutdatedSoftwareVBulletin(OutdatedSoftware):
    data_subtype = "vbulletin"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:vbulletin:vbulletin", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareVBulletin, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: vBulletin"


#------------------------------------------------------------------------------
class OutdatedSoftwareDotNetNuke(OutdatedSoftware):
    data_subtype = "dotnetnuke"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:dotnetnuke:dotnetnuke", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareDotNetNuke, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: DotNetNuke"


#------------------------------------------------------------------------------
class OutdatedSoftwarePHPNuke(OutdatedSoftware):
    data_subtype = "phpnuke"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:phpnuke:php-nuke", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePHPNuke, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: PHP-Nuke"


#------------------------------------------------------------------------------
class OutdatedSoftwareZope(OutdatedSoftware):
    data_subtype = "zope"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:cpe:/a:zope:zope", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareZope, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Zope"


#------------------------------------------------------------------------------
#
# Web servers
#
#------------------------------------------------------------------------------
class OutdatedSoftwareApache(OutdatedSoftware):
    data_subtype = "apache"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:apache:http_server", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareApache, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Apache"


#------------------------------------------------------------------------------
class OutdatedSoftwareNginx(OutdatedSoftware):
    data_subtype = "nginx"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:igor_sysoev:nginx", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareNginx, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Nginx"


#------------------------------------------------------------------------------
class OutdatedSoftwareIIS(OutdatedSoftware):
    data_subtype = "iis"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:microsoft:iis", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareIIS, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Internet Information Server (IIS)"


#------------------------------------------------------------------------------
class OutdatedSoftwareTomcat(OutdatedSoftware):
    data_subtype = "tomcat"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:apache:tomcat", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareTomcat, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Apache Tomcat"


#------------------------------------------------------------------------------
class OutdatedSoftwareOracleWebLogic(OutdatedSoftware):
    data_subtype = "oracleweblogic"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:oracle:weblogic_portal", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareOracleWebLogic, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Oracle Web Logic"


#------------------------------------------------------------------------------
class OutdatedSoftwareBEAWebLogic(OutdatedSoftware):
    data_subtype = "beaweblogic"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:bea:weblogic_platform", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareBEAWebLogic, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: BEA Web Logic"


#------------------------------------------------------------------------------
class OutdatedSoftwareFrontPage(OutdatedSoftware):
    data_subtype = "msfrontpageoutdated"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:microsoft:frontpage", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareFrontPage, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Microsoft Front Page"


#------------------------------------------------------------------------------
class OutdatedSoftwareLighttpd(OutdatedSoftware):
    data_subtype = "lighttpd"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:cpe:/a:lighttpd:lighttpd", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareLighttpd, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: lighttpd"


#------------------------------------------------------------------------------
#
# Database servers
#
#------------------------------------------------------------------------------
class OutdatedSoftwareMSSQLServer(OutdatedSoftware):
    data_subtype = "mssqlserver"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:microsoft:sql_server", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareMSSQLServer, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Microsoft SQL Server"


#------------------------------------------------------------------------------
class OutdatedSoftwareMySQL(OutdatedSoftware):
    data_subtype = "mysql"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:oracle:mysql", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareMySQL, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: MySQL Database"


#------------------------------------------------------------------------------
class OutdatedSoftwareOracleDB(OutdatedSoftware):
    data_subtype = "oracledb"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:/a:oracle:database_server", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwareOracleDB, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: Oracle Database"


#------------------------------------------------------------------------------
class OutdatedSoftwarePostgreSQL(OutdatedSoftware):
    data_subtype = "postgresql"


    #--------------------------------------------------------------------------
    def __init__(self, target, cpe="cpe:cpe:/a:postgresql:postgresql", **kwargs):
        """
        :param target: Vulnerable resource.
        :type target: Resource
        """
        super(OutdatedSoftwarePostgreSQL, self).__init__(target, cpe, **kwargs)


    #--------------------------------------------------------------------------
    @property
    def display_name(self):
        return "Outdated Software: PostgreSQL Database"


#------------------------------------------------------------------------------

__all__ = [ x for x in dir() if x.startswith("OutdatedSoftware") ]
