﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using System.Text.RegularExpressions;
using System.ComponentModel;
using System.Collections.ObjectModel;
using FuzzTalk.Actions;

namespace FuzzTalk
{
    [SerializableAttribute]
    [XmlRoot]
    public class FuzzTalkScript
    {
        public static FuzzTalkScript Current;

        private FuzzTalkScriptConfiguration _configuration;

        public FuzzTalkScriptConfiguration Configuration
        {
            get { return _configuration; }
            set { _configuration = value; }
        }

        private ObservableCollection<FuzzTalkAction> _actions;

        [XmlArrayItem(typeof(TcpConnect)),
            XmlArrayItem(typeof(Disconnect)),
            XmlArrayItem(typeof(Receive)),
            XmlArrayItem(typeof(Send)),
            XmlArrayItem(typeof(SendFuzz)),
            XmlArrayItem(typeof(AttachDebugger)),
            XmlArrayItem(typeof(DetachDebugger)),
            XmlArrayItem(typeof(RegexMonitor)),
            XmlArrayItem(typeof(FuzzTalkSleep))]
        public ObservableCollection<FuzzTalkAction> Actions
        {
            get { return _actions; }
            set { _actions = value; }
        }

        private ObservableCollection<FuzzTalkAction> _initActions;

        [XmlArrayItem(typeof(TcpConnect)),
            XmlArrayItem(typeof(Disconnect)),
            XmlArrayItem(typeof(Receive)),
            XmlArrayItem(typeof(Send)),
            XmlArrayItem(typeof(SendFuzz)),
            XmlArrayItem(typeof(AttachDebugger)),
            XmlArrayItem(typeof(DetachDebugger)),
            XmlArrayItem(typeof(RegexMonitor)),
            XmlArrayItem(typeof(FuzzTalkSleep))]
        public ObservableCollection<FuzzTalkAction> InitActions
        {
            get { return _initActions; }
            set { _initActions = value; }
        }

        private ObservableCollection<FuzzTalkAction> _cleanupActions;

        [XmlArrayItem(typeof(TcpConnect)),
            XmlArrayItem(typeof(Disconnect)),
            XmlArrayItem(typeof(Receive)),
            XmlArrayItem(typeof(Send)),
            XmlArrayItem(typeof(SendFuzz)),
            XmlArrayItem(typeof(AttachDebugger)),
            XmlArrayItem(typeof(DetachDebugger)),
            XmlArrayItem(typeof(RegexMonitor)),
            XmlArrayItem(typeof(FuzzTalkSleep))]
        public ObservableCollection<FuzzTalkAction> CleanupActions
        {
            get { return _cleanupActions; }
            set { _cleanupActions = value; }
        }

        public FuzzTalkAction[] GetAllActions()
        {
            var actions = _initActions.ToArray() ?? new FuzzTalkAction[0];

            if (_actions != null)
                actions = actions.Concat(_actions).ToArray();

            if (_cleanupActions != null)
                actions = actions.Concat(_cleanupActions).ToArray();

            return actions;
        }

        private FuzzTalkScriptInputTemplate _inputTemplate;

        public FuzzTalkScriptInputTemplate InputTemplate
        {
            get { return _inputTemplate; }
            set { _inputTemplate = value; }
        }

        private FuzzTalkScriptTemplateComponents _templateComponents;

        public FuzzTalkScriptTemplateComponents TemplateComponents
        {
            get { return _templateComponents; }
            set { _templateComponents = value; }
        }

        public string AssembleTemplate(int[] Indexes)
        {
            var t = new StringBuilder();

            for (int i = 0; i < Indexes.Length; i++)
                if (InputTemplate.Component[i].Reference != null)
                {
                    var componentItem = InputTemplate.Component[i].Reference.Components[Indexes[i]];

                    if (componentItem is FuzzTalkScriptTemplateComponentsTemplateComponentString)
                        t.Append((componentItem as FuzzTalkScriptTemplateComponentsTemplateComponentString).Value);
                    else if (componentItem is FuzzTalkScriptTemplateComponentsTemplateComponentStringOverflow)
                        t.Append((componentItem as FuzzTalkScriptTemplateComponentsTemplateComponentStringOverflow)
                            .Value.Repeat(Configuration.StringOverflowSize * Configuration.StringOverflowMultiplier));

                    //////////////////////////////////////////////////////////////////////////
                    // Todo Add integer overflow handling                                   
                    //////////////////////////////////////////////////////////////////////////
                }
                else if (InputTemplate.Component[i].Value != null)
                    t.Append(InputTemplate.Component[i].Value);

            return t.ToString();
        }

        public FuzzTalkScriptInputTemplate InitScript()
        {
            var template = InputTemplate;

            int index = 0;

            foreach (var component in TemplateComponents.TemplateComponent)
            {
                if (component.String != null)
                    foreach (var componentString in component.String)
                        componentString.Value = Regex.Unescape(componentString.Value);
            }

            foreach (var component in template.Component)
            {
                if (component.Value != null)
                    component.Value = Regex.Unescape(component.Value);

                component.Index = index++;

                component.Reference = TemplateComponents.TemplateComponent
                    .SingleOrDefault(x => x.Name == component.Name);

                if (component.Reference != null)
                {
                    component.Reference.Components = new ObservableCollection<object>();

                    if (component.Reference.String != null)
                    {
                        component.Reference.Components = 
                            new ObservableCollection<object>(component.Reference.Components
                                .Concat(component.Reference.String.OfType<object>())); ;
                    }

                    if (component.Reference.StringOverflow != null)
                    {
                        foreach (var testString in component.Reference.StringOverflow)
                            testString.Value = Regex.Unescape(testString.Value);

                        component.Reference.Components =
                            new ObservableCollection<object>(component.Reference.Components
                            .Concat(component.Reference.StringOverflow.OfType<object>()));
                    }

                    if (component.Reference.IntegerOverflows != null)
                    {
                        if (component.Reference.String == null)
                            component.Reference.String = new ObservableCollection<FuzzTalkScriptTemplateComponentsTemplateComponentString>();

                        var overflows = new List<FuzzTalkScriptTemplateComponentsTemplateComponentString>();

                        foreach (var overflowTag in component.Reference.IntegerOverflows)
                        {
                            foreach (string value in overflowTag.MinValues.Concat(overflowTag.MaxValues))
                            {
                                if (!overflows.Any(x => x.Value == value))
                                    overflows.Add(new FuzzTalkScriptTemplateComponentsTemplateComponentString()
                                    {
                                        Value = value
                                    });
                            }                            
                        }

                        component.Reference.Components =
                            new ObservableCollection<object>(component.Reference.Components
                            .Concat(overflows.OfType<object>()));
                    }
                }
            }

            return template;
        }

        public static ObservableCollection<FuzzTalkScriptParameter> GetParameters(string Filename)
        {
            var serializer = new XmlSerializer(typeof(FuzzTalkConfigWrapper));
            var scriptText = File.ReadAllText(Filename);
            var configWrapper = serializer.Deserialize(scriptText) as FuzzTalkConfigWrapper;

            return configWrapper.Configuration.Parameters;
        }

        public static FuzzTalkScript Deserialize(string Filename, Dictionary<string, string> Options)
        {
            var scriptText = File.ReadAllText(Filename);

            //scriptText = TemplateParser.ReplaceTokens(scriptText, Options);

            

            var serializer2 = new XmlSerializer(typeof(FuzzTalkConfigWrapper));
            var configWrapper = 
                serializer2.Deserialize(scriptText) as FuzzTalkConfigWrapper;

            scriptText = TemplateParser.ReplaceTokens(scriptText,
                configWrapper.Configuration.Parameters
                    .Where(x =>
                        x.DefaultValue != null &&
                        !Options.Keys.Contains(x.Name))
                    .ToDictionary(x => x.Name, x => x.DefaultValue)
                    .Concat(Options)
                    .ToDictionary(x => x.Key, x => x.Value));

            var serializer = new XmlSerializer(typeof(FuzzTalkScript));

            FuzzTalkScript script = serializer.Deserialize(scriptText) as FuzzTalkScript;

            //if (script.Configuration.Parameters.Any())
            //{
            //    script.Configuration.Parameters
            //        .Where(x =>
            //            x.DefaultValue != null &&
            //            !Options.Keys.Contains(x.Name))
            //        .ToDictionary(x => x.Name, x => x.DefaultValue)
            //        .Concat(Options)
            //        .ToDictionary(x => x.Key, x => x.Value)
                    
                

            //    scriptText = TemplateParser.ReplaceTokens(scriptText,
            //        script.Configuration.Parameters
            //            .Where(x => x.DefaultValue != null)
            //            .ToDictionary(x => x.Name, x => x.DefaultValue));

            //    script = serializer.Deserialize(scriptText) as FuzzTalkScript;
            //}

            return script;
        }
    }
}
