import math

from exploits.ACVE import ACVE

__author__ = "Jean Lejeune <jlejeune@immunit.ch>"
__copyright__ = "Copyright 2018, ImmunIT"


class Checker(ACVE):
    """This class checks if the given target is vulnerable to the CVE-2018-7600.
    """

    def __init__(self, request, logger):
        self.logger = logger
        self.request = request

    def run(self, version):
        if self._checkDefault():
            return

        if math.floor(version) == 7:
            self._checkDrupal7()
        elif math.floor(version) == 8:
            self._checkDrupal8()

    def _checkDefault(self):
        """Check if the CVE tested is present in the CHANGELOG default file.
        Return True if yes. False otherwise.
        """

        dfile = self.request.get("/CHANGELOG.txt")

        if dfile.status_code == 200 and "SA-CORE-2018-002" in str(dfile.content):
            self.logger.handle("\n[-] Application not vulnerable", self.logger.ERROR)
            return True

        return False

    def _checkDrupal7(self):
        """Check Drupal 7.x.
        """

        req = self.request.get("/includes/request-sanitizers.inc").status_code
        ref = self.request.get("/includes/LetMeGetYourStatusCode.inc").status_code

        if req != 404 and req != ref:
            self.logger.handle("\n[-] Application not vulnerable", self.logger.ERROR)
        elif req == 403 and req == ref:
            self.logger.handle("\n[-] Cannot state if the Application is vulnerable", self.logger.ERROR)
        else:
            self.logger.handle("\n[+] Application vulnerable", self.logger.SUCCESS)

    def _checkDrupal8(self):
        """Check Drupal 8.x.
        """

        req = self.request.get("/core/lib/Drupal/Core/Security/RequestSanitizer.php").status_code
        ref = self.request.get("/core/lib/Drupal/Core/Security/LetMeGetYourStatusCode.php").status_code

        if req != 404 and req != ref:
            self.logger.handle("\n[-] Application not vulnerable", self.logger.ERROR)
        else:
            self.logger.handle("\n[+] Application vulnerable", self.logger.SUCCESS)
