"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.SASQueryParameters = exports.SASProtocol = void 0;
const SasIPRange_js_1 = require("./SasIPRange.js");
const utils_common_js_1 = require("./utils/utils.common.js");
/**
 * Protocols for generated SAS.
 */
var SASProtocol;
(function (SASProtocol) {
    /**
     * Protocol that allows HTTPS only
     */
    SASProtocol["Https"] = "https";
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    SASProtocol["HttpsAndHttp"] = "https,http";
})(SASProtocol || (exports.SASProtocol = SASProtocol = {}));
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link QueueSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
class SASQueryParameters {
    /**
     * The storage API version.
     */
    version;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    protocol;
    /**
     * Optional. The start time for this SAS token.
     */
    startsOn;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    expiresOn;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link QueueSASPermissions} for more details.
     */
    permissions;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    services;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    resourceTypes;
    /**
     * Optional. The signed identifier (only for {@link QueueSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier;
    /**
     * Optional. The storage queue (only for {@link QueueSASSignatureValues}).
     */
    resource;
    /**
     * The signature for the SAS token.
     */
    signature;
    /**
     * Inner value of getter ipRange.
     */
    ipRangeInner;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange() {
        if (this.ipRangeInner) {
            return {
                end: this.ipRangeInner.end,
                start: this.ipRangeInner.start,
            };
        }
        return undefined;
    }
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param permissions - Representing the storage permissions
     * @param services - Representing the storage services being accessed (only for Account SAS)
     * @param resourceTypes - Representing the storage resource types being accessed (only for Account SAS)
     * @param protocol - Representing the allowed HTTP protocol(s)
     * @param startsOn - Representing the start time for this SAS token
     * @param expiresOn - Representing the expiry time for this SAS token
     * @param ipRange - Representing the range of valid IP addresses for this SAS token
     * @param identifier - Representing the signed identifier (only for Service SAS)
     * @param resource - Representing the storage queue (only for Service SAS)
     */
    constructor(version, signature, permissions, services, resourceTypes, protocol, startsOn, expiresOn, ipRange, identifier, resource) {
        this.version = version;
        this.services = services;
        this.resourceTypes = resourceTypes;
        this.expiresOn = expiresOn;
        this.permissions = permissions;
        this.protocol = protocol;
        this.startsOn = startsOn;
        this.ipRangeInner = ipRange;
        this.identifier = identifier;
        this.resource = resource;
        this.signature = signature;
    }
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString() {
        const params = ["sv", "ss", "srt", "spr", "st", "se", "sip", "si", "sr", "sp", "sig"];
        const queries = [];
        for (const param of params) {
            switch (param) {
                case "sv":
                    this.tryAppendQueryParameter(queries, param, this.version);
                    break;
                case "ss":
                    this.tryAppendQueryParameter(queries, param, this.services);
                    break;
                case "srt":
                    this.tryAppendQueryParameter(queries, param, this.resourceTypes);
                    break;
                case "spr":
                    this.tryAppendQueryParameter(queries, param, this.protocol);
                    break;
                case "st":
                    this.tryAppendQueryParameter(queries, param, this.startsOn ? (0, utils_common_js_1.truncatedISO8061Date)(this.startsOn, false) : undefined);
                    break;
                case "se":
                    this.tryAppendQueryParameter(queries, param, this.expiresOn ? (0, utils_common_js_1.truncatedISO8061Date)(this.expiresOn, false) : undefined);
                    break;
                case "sip":
                    this.tryAppendQueryParameter(queries, param, this.ipRange ? (0, SasIPRange_js_1.ipRangeToString)(this.ipRange) : undefined);
                    break;
                case "si":
                    this.tryAppendQueryParameter(queries, param, this.identifier);
                    break;
                case "sr":
                    this.tryAppendQueryParameter(queries, param, this.resource);
                    break;
                case "sp":
                    this.tryAppendQueryParameter(queries, param, this.permissions);
                    break;
                case "sig":
                    this.tryAppendQueryParameter(queries, param, this.signature);
                    break;
            }
        }
        return queries.join("&");
    }
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    tryAppendQueryParameter(queries, key, value) {
        if (!value) {
            return;
        }
        key = encodeURIComponent(key);
        value = encodeURIComponent(value);
        if (key.length > 0 && value.length > 0) {
            queries.push(`${key}=${value}`);
        }
    }
}
exports.SASQueryParameters = SASQueryParameters;
//# sourceMappingURL=SASQueryParameters.js.map