import type { SasIPRange } from "./SasIPRange.js";
/**
 * Protocols for generated SAS.
 */
export declare enum SASProtocol {
    /**
     * Protocol that allows HTTPS only
     */
    Https = "https",
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    HttpsAndHttp = "https,http"
}
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link FileSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export declare class SASQueryParameters {
    /**
     * The storage API version.
     */
    readonly version: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    readonly protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    readonly startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    readonly expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link FileSASPermissions}, or {@link ShareSASPermissions} for
     * more details.
     */
    readonly permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    readonly services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link FileSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    readonly identifier?: string;
    /**
     * Optional. The storage share or file path (only for {@link FileSASSignatureValues}).
     */
    readonly resource?: string;
    /**
     * The signature for the SAS token.
     */
    readonly signature: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     */
    readonly cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     */
    readonly contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     */
    readonly contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     */
    readonly contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     */
    readonly contentType?: string;
    /**
     * Inner value of getter ipRange.
     */
    private readonly ipRangeInner?;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange(): SasIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param permissions - Representing the storage permissions
     * @param services - Representing the storage services being accessed (only for Account SAS)
     * @param resourceTypes - Representing the storage resource types being accessed (only for Account SAS)
     * @param protocol - Representing the allowed HTTP protocol(s)
     * @param startsOn - Representing the start time for this SAS token
     * @param expiresOn - Representing the expiry time for this SAS token
     * @param ipRange - Representing the range of valid IP addresses for this SAS token
     * @param identifier - Representing the signed identifier (only for Service SAS)
     * @param resource - Representing the storage container or blob (only for Service SAS)
     * @param cacheControl - Representing the cache-control header (only for Blob/File Service SAS)
     * @param contentDisposition - Representing the content-disposition header (only for Blob/File Service SAS)
     * @param contentEncoding - Representing the content-encoding header (only for Blob/File Service SAS)
     * @param contentLanguage - Representing the content-language header (only for Blob/File Service SAS)
     * @param contentType - Representing the content-type header (only for Blob/File Service SAS)
     */
    constructor(version: string, signature: string, permissions?: string, services?: string, resourceTypes?: string, protocol?: SASProtocol, startsOn?: Date, expiresOn?: Date, ipRange?: SasIPRange, identifier?: string, resource?: string, cacheControl?: string, contentDisposition?: string, contentEncoding?: string, contentLanguage?: string, contentType?: string);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    private tryAppendQueryParameter;
}
//# sourceMappingURL=SASQueryParameters.d.ts.map