"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.fromAccountConnectionString = fromAccountConnectionString;
exports.getAccountConnectionString = getAccountConnectionString;
const core_auth_1 = require("@azure/core-auth");
/**
 * Gets client parameters from an Account Connection String
 * Only supported in Node.js not supported for Browsers
 * @param extractedCreds - parsed connection string
 * @param options - TablesServiceClient options
 */
function fromAccountConnectionString(extractedCreds, options = {}) {
    const sharedKeyCredential = new core_auth_1.AzureNamedKeyCredential(extractedCreds.accountName, extractedCreds.accountKey);
    return {
        url: extractedCreds.url,
        options,
        credential: sharedKeyCredential,
    };
}
function getAccountConnectionString(accountName, accountKey, defaultEndpointsProtocol, endpointSuffix, tableEndpoint) {
    if (!tableEndpoint) {
        // TableEndpoint is not present in the Account connection string
        // Can be obtained from `${defaultEndpointsProtocol}://${accountName}.table.${endpointSuffix}`
        const protocol = defaultEndpointsProtocol.toLowerCase();
        if (protocol !== "https" && protocol !== "http") {
            throw new Error("Invalid DefaultEndpointsProtocol in the provided Connection String. Expecting 'https' or 'http'");
        }
        if (!endpointSuffix) {
            throw new Error("Invalid EndpointSuffix in the provided Connection String");
        }
        tableEndpoint = `${defaultEndpointsProtocol}://${accountName}.table.${endpointSuffix}`;
    }
    if (!accountName) {
        throw new Error("Invalid AccountName in the provided Connection String");
    }
    else if (accountKey.length === 0) {
        throw new Error("Invalid AccountKey in the provided Connection String");
    }
    return {
        kind: "AccountConnString",
        url: tableEndpoint,
        accountName,
        accountKey,
    };
}
//# sourceMappingURL=accountConnectionString.js.map