import type { CreateTableEntityResponse, DeleteTableEntityOptions, GetAccessPolicyResponse, GetTableEntityOptions, GetTableEntityResponse, ListTableEntitiesOptions, SignedIdentifier, TableServiceClientOptions as TableClientOptions, TableEntity, TableEntityResult, TableEntityResultPage, TableTransactionResponse, TransactionAction, UpdateMode, UpdateTableEntityOptions } from "./models.js";
import type { DeleteTableEntityResponse, SetAccessPolicyResponse, UpdateEntityResponse, UpsertEntityResponse } from "./generatedModels.js";
import type { OperationOptions } from "@azure/core-client";
import type { NamedKeyCredential, SASCredential, TokenCredential } from "@azure/core-auth";
import type { PagedAsyncIterableIterator } from "@azure/core-paging";
import type { Pipeline } from "@azure/core-rest-pipeline";
/**
 * A TableClient represents a Client to the Azure Tables service allowing you
 * to perform operations on a single table.
 */
export declare class TableClient {
    /**
     * Table Account URL
     */
    url: string;
    /**
     * Represents a pipeline for making a HTTP request to a URL.
     * Pipelines can have multiple policies to manage manipulating each request before and after it is made to the server.
     */
    pipeline: Pipeline;
    private table;
    private generatedClient;
    private credential?;
    private clientOptions;
    private readonly allowInsecureConnection;
    /**
     * Name of the table to perform operations on.
     */
    readonly tableName: string;
    /**
     * Creates a new instance of the TableClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation, such as "https://myaccount.table.core.windows.net".
     * @param tableName - the name of the table
     * @param credential - NamedKeyCredential used to authenticate requests. Only Supported for Node
     * @param options - Optional. Options to configure the HTTP pipeline.
     *
     *
     * ### Example using an account name/key:
     *
     * ```ts snippet:ReadmeSampleCreateTableClient_NamedKeyCredential
     * import { AzureNamedKeyCredential, TableClient } from "@azure/data-tables";
     *
     * // Enter your storage account name and shared key
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * // Use AzureNamedKeyCredential with storage account and account key
     * // AzureNamedKeyCredential is only available in Node.js runtime, not in browsers
     * const credential = new AzureNamedKeyCredential(account, accountKey);
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     * ```
     */
    constructor(url: string, tableName: string, credential: NamedKeyCredential, options?: TableClientOptions);
    /**
     * Creates a new instance of the TableClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation, such as "https://myaccount.table.core.windows.net".
     * @param tableName - the name of the table
     * @param credential - SASCredential used to authenticate requests
     * @param options - Optional. Options to configure the HTTP pipeline.
     *
     *
     * ### Example using a SAS Token:
     *
     * ```ts snippet:ReadmeSampleCreateTableClient_SASToken
     * import { TableClient, AzureSASCredential } from "@azure/data-tables";
     *
     * const account = "<account name>";
     * const sas = "<service Shared Access Signature Token>";
     * const tableName = "<tableName>";
     *
     * const clientWithSAS = new TableClient(
     *   `https://${account}.table.core.windows.net`,
     *   tableName,
     *   new AzureSASCredential(sas),
     * );
     * ```
     */
    constructor(url: string, tableName: string, credential: SASCredential, options?: TableClientOptions);
    /**
     * Creates a new instance of the TableClient class.
     *
     * @param url - The URL of the service account that is the target of the desired operation, such as "https://myaccount.table.core.windows.net".
     * @param tableName - the name of the table
     * @param credential - Azure Active Directory credential used to authenticate requests
     * @param options - Optional. Options to configure the HTTP pipeline.
     *
     *
     * ### Example using an Azure Active Directory credential:
     *
     * ```ts snippet:ReadmeSampleCreateTableClient_TokenCredential
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const credential = new DefaultAzureCredential();
     * const account = "<account name>";
     * const tableName = "<tableName>";
     *
     * const clientWithAAD = new TableClient(
     *   `https://${account}.table.core.windows.net`,
     *   tableName,
     *   credential,
     * );
     * ```
     */
    constructor(url: string, tableName: string, credential: TokenCredential, options?: TableClientOptions);
    /**
     * Creates an instance of TableClient.
     *
     * @param url - A Client string pointing to Azure Storage table service, such as
     *              "https://myaccount.table.core.windows.net". You can append a SAS,
     *              such as "https://myaccount.table.core.windows.net?sasString".
     * @param tableName - the name of the table
     * @param options - Options to configure the HTTP pipeline.
     *
     * ### Example appending a SAS token:
     *
     * ```ts snippet:ReadmeSampleCreateTableClient_SASTokenURL
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account name>";
     * const sasToken = "<SAS token>";
     * const tableName = "<tableName>";
     *
     * const clientWithSAS = new TableClient(
     *   `https://${account}.table.core.windows.net?${sasToken}`,
     *   tableName,
     * );
     * ```
     */
    constructor(url: string, tableName: string, options?: TableClientOptions);
    /**
     * Permanently deletes the current table with all of its entities.
     * @param options - The options parameters.
     *
     * ### Example deleting a table
     * ```ts snippet:ReadmeSampleDeleteTable
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * await client.deleteTable();
     * ```
     */
    deleteTable(options?: OperationOptions): Promise<void>;
    /**
     *  Creates a table with the tableName passed to the client constructor
     * @param options - The options parameters.
     *
     * ### Example creating a table
     * ```ts snippet:ReadmeSampleTableClientCreateTable
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * // If the table 'newTable' already exists, createTable doesn't throw
     * await client.createTable();
     * ```
     */
    createTable(options?: OperationOptions): Promise<void>;
    /**
     * Returns a single entity in the table.
     * @param partitionKey - The partition key of the entity.
     * @param rowKey - The row key of the entity.
     * @param options - The options parameters.
     *
     * ### Example getting an entity
     * ```ts snippet:ReadmeSampleGetEntity
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const entity = await client.getEntity("<partitionKey>", "<rowKey>");
     * console.log(`Entity: PartitionKey: ${entity.partitionKey} RowKey: ${entity.rowKey}`);
     * ```
     */
    getEntity<T extends object = Record<string, unknown>>(partitionKey: string, rowKey: string, options?: GetTableEntityOptions): Promise<GetTableEntityResponse<TableEntityResult<T>>>;
    /**
     * Queries entities in a table.
     * @param options - The options parameters.
     *
     * Example listing entities
     * ```ts snippet:ReadmeSampleListEntities
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * let i = 0;
     * const entities = client.listEntities();
     * for await (const entity of entities) {
     *   console.log(`Entity${++i}: PartitionKey: ${entity.partitionKey} RowKey: ${entity.rowKey}`);
     * }
     * ```
     */
    listEntities<T extends object = Record<string, unknown>>(options?: ListTableEntitiesOptions): PagedAsyncIterableIterator<TableEntityResult<T>, TableEntityResultPage<T>>;
    private listEntitiesAll;
    private listEntitiesPage;
    private _listEntities;
    /**
     * Insert entity in the table.
     * @param entity - The properties for the table entity.
     * @param options - The options parameters.
     *
     * ### Example creating an entity
     * ```ts snippet:ReadmeSampleCreateEntity
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const testEntity = {
     *   partitionKey: "P1",
     *   rowKey: "R1",
     *   foo: "foo",
     *   bar: 123,
     * };
     * await client.createEntity(testEntity);
     * ```
     */
    createEntity<T extends object>(entity: TableEntity<T>, options?: OperationOptions): Promise<CreateTableEntityResponse>;
    /**
     * Deletes the specified entity in the table.
     * @param partitionKey - The partition key of the entity.
     * @param rowKey - The row key of the entity.
     * @param options - The options parameters.
     *
     * ### Example deleting an entity
     * ```ts snippet:ReadmeSampleDeleteEntity
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * // deleteEntity deletes the entity that matches exactly the partitionKey and rowKey
     * await client.deleteEntity("<partitionKey>", "<rowKey>");
     * ```
     */
    deleteEntity(partitionKey: string, rowKey: string, options?: DeleteTableEntityOptions): Promise<DeleteTableEntityResponse>;
    /**
     * Update an entity in the table.
     * @param entity - The properties of the entity to be updated.
     * @param mode - The different modes for updating the entity:
     *               - Merge: Updates an entity by updating the entity's properties without replacing the existing entity.
     *               - Replace: Updates an existing entity by replacing the entire entity.
     * @param options - The options parameters.
     *
     * ### Example updating an entity
     * ```ts snippet:ReadmeSampleUpdateEntity
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const entity = { partitionKey: "p1", rowKey: "r1", bar: "updatedBar" };
     *
     * // Update uses update mode "Merge" as default
     * // merge means that update will match a stored entity
     * // that has the same partitionKey and rowKey as the entity
     * // passed to the method and then will only update the properties present in it.
     * // Any other properties that are not defined in the entity passed to updateEntity
     * // will remain as they are in the service
     * await client.updateEntity(entity);
     *
     * // We can also set the update mode to Replace, which will match the entity passed
     * // to updateEntity with one stored in the service and replace with the new one.
     * // If there are any missing properties in the entity passed to updateEntity, they
     * // will be removed from the entity stored in the service
     * await client.updateEntity(entity, "Replace");
     * ```
     */
    updateEntity<T extends object>(entity: TableEntity<T>, mode?: UpdateMode, options?: UpdateTableEntityOptions): Promise<UpdateEntityResponse>;
    /**
     * Upsert an entity in the table.
     * @param entity - The properties for the table entity.
     * @param mode - The different modes for updating the entity:
     *               - Merge: Updates an entity by updating the entity's properties without replacing the existing entity.
     *               - Replace: Updates an existing entity by replacing the entire entity.
     * @param options - The options parameters.
     *
     * ### Example upserting an entity
     * ```ts snippet:ReadmeSampleUpsertEntity
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const entity = { partitionKey: "p1", rowKey: "r1", bar: "updatedBar" };
     *
     * // Upsert uses update mode "Merge" as default.
     * // This behaves similarly to update but creates the entity
     * // if it doesn't exist in the service
     * await client.upsertEntity(entity);
     *
     * // We can also set the update mode to Replace.
     * // This behaves similarly to update but creates the entity
     * // if it doesn't exist in the service
     * await client.upsertEntity(entity, "Replace");
     * ```
     */
    upsertEntity<T extends object>(entity: TableEntity<T>, mode?: UpdateMode, options?: OperationOptions): Promise<UpsertEntityResponse>;
    /**
     * Retrieves details about any stored access policies specified on the table that may be used with
     * Shared Access Signatures.
     * @param options - The options parameters.
     */
    getAccessPolicy(options?: OperationOptions): Promise<GetAccessPolicyResponse>;
    /**
     * Sets stored access policies for the table that may be used with Shared Access Signatures.
     * @param tableAcl - The Access Control List for the table.
     * @param options - The options parameters.
     */
    setAccessPolicy(tableAcl: SignedIdentifier[], options?: OperationOptions): Promise<SetAccessPolicyResponse>;
    /**
     * Submits a Transaction which is composed of a set of actions. You can provide the actions as a list
     * or you can use {@link TableTransaction} to help building the transaction.
     *
     * Example usage:
     * ```ts snippet:ReadmeSampleSubmitTransaction
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient, TransactionAction } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const actions: TransactionAction[] = [
     *   ["create", { partitionKey: "p1", rowKey: "1", data: "test1" }],
     *   ["delete", { partitionKey: "p1", rowKey: "2" }],
     *   ["update", { partitionKey: "p1", rowKey: "3", data: "newTest" }, "Merge"],
     * ];
     * const result = await client.submitTransaction(actions);
     * ```
     *
     * Example usage with TableTransaction:
     * ```ts snippet:ReadmeSampleSubmitTransactionWithTableTransaction
     * import { DefaultAzureCredential } from "@azure/identity";
     * import { TableClient, TableTransaction } from "@azure/data-tables";
     *
     * const account = "<account>";
     * const accountKey = "<accountkey>";
     * const tableName = "<tableName>";
     *
     * const credential = new DefaultAzureCredential();
     * const client = new TableClient(`https://${account}.table.core.windows.net`, tableName, credential);
     *
     * const transaction = new TableTransaction();
     *
     * // Call the available action in the TableTransaction object
     * transaction.createEntity({ partitionKey: "p1", rowKey: "1", data: "test1" });
     * transaction.deleteEntity("p1", "2");
     * transaction.updateEntity({ partitionKey: "p1", rowKey: "3", data: "newTest" }, "Merge");
     *
     * // submitTransaction with the actions list on the transaction.
     * const result = await client.submitTransaction(transaction.actions);
     * ```
     *
     * @param actions - tuple that contains the action to perform, and the entity to perform the action with
     * @param options - Options for the request.
     */
    submitTransaction(actions: TransactionAction[], options?: OperationOptions): Promise<TableTransactionResponse>;
    /**
     *
     * Creates an instance of TableClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                           [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                           Account connection string example -
     *                           `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                           SAS connection string example -
     *                           `BlobEndpoint=https://myaccount.table.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new TableClient from the given connection string.
     */
    static fromConnectionString(connectionString: string, tableName: string, options?: TableClientOptions): TableClient;
}
//# sourceMappingURL=TableClient.d.ts.map