﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/cost-optimization-hub/CostOptimizationHubRequest.h>
#include <aws/cost-optimization-hub/CostOptimizationHub_EXPORTS.h>
#include <aws/cost-optimization-hub/model/Filter.h>
#include <aws/cost-optimization-hub/model/OrderBy.h>

#include <utility>

namespace Aws {
namespace CostOptimizationHub {
namespace Model {

/**
 */
class ListRecommendationsRequest : public CostOptimizationHubRequest {
 public:
  AWS_COSTOPTIMIZATIONHUB_API ListRecommendationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListRecommendations"; }

  AWS_COSTOPTIMIZATIONHUB_API Aws::String SerializePayload() const override;

  AWS_COSTOPTIMIZATIONHUB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The constraints that you want all returned recommendations to match.</p>
   */
  inline const Filter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Filter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Filter>
  ListRecommendationsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ordering of recommendations by a dimension.</p>
   */
  inline const OrderBy& GetOrderBy() const { return m_orderBy; }
  inline bool OrderByHasBeenSet() const { return m_orderByHasBeenSet; }
  template <typename OrderByT = OrderBy>
  void SetOrderBy(OrderByT&& value) {
    m_orderByHasBeenSet = true;
    m_orderBy = std::forward<OrderByT>(value);
  }
  template <typename OrderByT = OrderBy>
  ListRecommendationsRequest& WithOrderBy(OrderByT&& value) {
    SetOrderBy(std::forward<OrderByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of all recommendations for a resource, or a single recommendation if
   * de-duped by <code>resourceId</code>.</p>
   */
  inline bool GetIncludeAllRecommendations() const { return m_includeAllRecommendations; }
  inline bool IncludeAllRecommendationsHasBeenSet() const { return m_includeAllRecommendationsHasBeenSet; }
  inline void SetIncludeAllRecommendations(bool value) {
    m_includeAllRecommendationsHasBeenSet = true;
    m_includeAllRecommendations = value;
  }
  inline ListRecommendationsRequest& WithIncludeAllRecommendations(bool value) {
    SetIncludeAllRecommendations(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of recommendations that are returned for the request.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListRecommendationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to retrieve the next set of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListRecommendationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Filter m_filter;
  bool m_filterHasBeenSet = false;

  OrderBy m_orderBy;
  bool m_orderByHasBeenSet = false;

  bool m_includeAllRecommendations{false};
  bool m_includeAllRecommendationsHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CostOptimizationHub
}  // namespace Aws
